package net.abstractfactory.plum.view.web.component.builder;

import java.lang.reflect.Modifier;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import org.apache.log4j.Logger;

import net.abstractfactory.plum.view.component.Component;
import net.abstractfactory.plum.view.web.component.WebComponent;
import net.abstractfactory.util.ReflectionFactory;
import net.abstractfactory.util.ReflectionHelper;

/**
 * create web component from abstract component
 * 
 * @author jack
 *
 */
public class WebComponentFactory {
	private static Logger logger = Logger.getLogger(WebComponentFactory.class);
	
	private static WebComponentFactory instance;
	
	public static void init(Set<String> builderBasePackages){
		instance = new WebComponentFactory(builderBasePackages);
	}
	
	public static WebComponentFactory getInstace(){
		if(instance==null){
			instance = new WebComponentFactory(new HashSet<String>());
		}
		
		return instance;
	}
	
	private Set<String> builderBasePackages;

	/**
	 * registered builders
	 * 
	 * key is component class
	 * 
	 */
	private Map<Class, WebComponentBuilder> builders = new HashMap<Class, WebComponentBuilder>();

	private WebComponentFactory(Set<String> builderBasePackages) {
		this.builderBasePackages = builderBasePackages;
		if(builderBasePackages.isEmpty()){
			//default extra package
			builderBasePackages.add("net.abstractfactory.plum.lib.extra");
		}
		
		scan(builderBasePackages);
	}
	
	public Set<String> getBuilderBasePackages() {
		return builderBasePackages;
	}

	public void setBuilderBasePackages(Set<String> builderBasePackages) {
		this.builderBasePackages = builderBasePackages;
	}

	private void scan(Set<String> basePackages) {
		
		Set<Class> scanResult = new HashSet<Class>();
		for (String basePackage : basePackages) {

			ReflectionHelper reflectionHelper = ReflectionFactory.getReflectionHelper();
			scanResult = reflectionHelper.scanClasses(basePackage, WebComponentBuilder.class);
		}

		for (Class<? extends WebComponentBuilder> builderClass : scanResult) {

			try {
				Class clazz = builderClass;

				if (clazz.isInterface())
					continue;

				if (Modifier.isAbstract(clazz.getModifiers()))
					continue;

				logger.debug("find view builder: " + builderClass);

				WebComponentBuilder builder = (WebComponentBuilder) clazz.newInstance();

				register(builder);

			} catch (Exception e) {
				throw new RuntimeException(e);
			}

		}
	}

	private void register(WebComponentBuilder builder) {
		Class componentClass = builder.getComponentClass();
		if (builders.containsKey(componentClass)) {
			throw new RuntimeException("web component builder already exist for component class:" + componentClass);
		}

		builders.put(componentClass, builder);
	}

	public WebComponent create(WebViewBuilder viewBuilder, Component component) {
		Class componentClass = component.getClass();
		WebComponentBuilder builder = builders.get(componentClass);
		if (builder == null) {
			throw new RuntimeException(
					"web component builder not found for component class:" + componentClass.getCanonicalName());
		}

		return builder.visit(viewBuilder, component);
	}
}
