package me.martiii.simplenetprotocolserver.protocol;

import me.martiii.simplenetprotocolserver.SimpleNetProtocolServer;

import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.Socket;
import java.util.HashMap;

public class ProtocolManager {
    private SimpleNetProtocolServer server;
    private HashMap<String, RequestHandlerInfo> requestHandlers;

    public ProtocolManager(SimpleNetProtocolServer server) {
        this.server = server;
        requestHandlers = new HashMap<>();
    }

    public void registerListener(Object listener) {
        for (Method method : listener.getClass().getDeclaredMethods()) {
            RequestHandler annotation = method.getAnnotation(RequestHandler.class);
            if (annotation != null) {
                String request = annotation.request();
                if (!requestHandlers.containsKey(request)) {
                    Class<?>[] params = method.getParameterTypes();
                    if (params.length > 1 && params[0].equals(Request.class) && params[1].equals(Socket.class)) {
                        requestHandlers.put(request, new RequestHandlerInfo(method, listener, annotation.expectedData()));
                    }
                }
            }
        }
    }

    public void handleRequest(String received, Socket socket) {
        String[] parts = received.split("(?<!\\\\):");
        String req = parts[0];
        if (requestHandlers.containsKey(req)) {
            String[] data = {};
            if (parts.length > 1) {
                data = new String[parts.length - 1];
                System.arraycopy(parts, 1, data, 0, parts.length - 1);
            }
            RequestHandlerInfo reqHandlesInfo = requestHandlers.get(req);
            if (data.length >= reqHandlesInfo.getExpectedData()) {
                Request request = new Request(data);
                try {
                    reqHandlesInfo.getMethod().invoke(reqHandlesInfo.getListener(), request, socket);
                } catch (IllegalAccessException | InvocationTargetException e) {
                    e.printStackTrace();
                }
            } else {
                System.out.println("Not enough data.");
                try {
                    socket.close();
                } catch (IOException e) {
                    e.printStackTrace();
                }
            }
        } else {
            System.out.println("Undefined request.");
            try {
                socket.close();
            } catch (IOException e) {
                e.printStackTrace();
            }
        }
    }

    public static class RequestHandlerInfo {
        private Method method;
        private Object listener;
        private int expectedData;

        public RequestHandlerInfo(Method method, Object listener, int expectedData) {
            this.method = method;
            this.listener = listener;
            this.expectedData = expectedData;
        }

        public Method getMethod() {
            return method;
        }

        public Object getListener() {
            return listener;
        }

        public int getExpectedData() {
            return expectedData;
        }
    }
}
