package io.r2mo.vertx.jooq.shared.postgres;

import org.jooq.Binding;
import org.jooq.BindingGetResultSetContext;
import org.jooq.BindingGetSQLInputContext;
import org.jooq.BindingGetStatementContext;
import org.jooq.BindingRegisterContext;
import org.jooq.BindingSQLContext;
import org.jooq.BindingSetSQLOutputContext;
import org.jooq.BindingSetStatementContext;
import org.jooq.RenderContext;
import org.jooq.impl.DSL;

import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.Types;
import java.util.Objects;
import java.util.function.Function;

/**
 * @author jensklingsporn
 */
public abstract class PGJsonToVertxJsonBinding<PG_JSON, VERTX_JSON> implements Binding<PG_JSON, VERTX_JSON> {

    abstract Function<String, PG_JSON> valueOf();

    abstract String coerce();

    // The converter does all the work
    // Rending a bind variable for the binding context's value and casting it to the json type
    @Override
    public void sql(final BindingSQLContext<VERTX_JSON> ctx) {
        // Depending on how you generate your SQL, you may need to explicitly distinguish
        // between jOOQ generating bind variables or inlined literals. If so, use this check:
        // ctx.render().paramType() == INLINED
        final RenderContext context = ctx.render().visit(DSL.val(ctx.convert(this.converter()).value()));
        context.sql(this.coerce());
    }

    // Registering VARCHAR types for JDBC CallableStatement OUT parameters
    @Override
    public void register(final BindingRegisterContext<VERTX_JSON> ctx) throws SQLException {
        ctx.statement().registerOutParameter(ctx.index(), Types.VARCHAR);
    }

    // Converting the JsonObject to a String value and setting that on a JDBC PreparedStatement
    @Override
    public void set(final BindingSetStatementContext<VERTX_JSON> ctx) throws SQLException {
        ctx.statement().setString(ctx.index(), Objects.toString(ctx.convert(this.converter()).value(), null));
    }

    // Getting a String value from a JDBC ResultSet and converting that to a JsonObject
    @Override
    public void get(final BindingGetResultSetContext<VERTX_JSON> ctx) throws SQLException {
        ctx.convert(this.converter()).value(this.valueOf().apply(ctx.resultSet().getString(ctx.index())));
    }

    // Getting a String value from a JDBC CallableStatement and converting that to a JsonObject
    @Override
    public void get(final BindingGetStatementContext<VERTX_JSON> ctx) throws SQLException {
        ctx.convert(this.converter()).value(this.valueOf().apply(ctx.statement().getString(ctx.index())));
    }

    // Setting a value on a JDBC SQLOutput (useful for Oracle OBJECT types)
    @Override
    public void set(final BindingSetSQLOutputContext<VERTX_JSON> ctx) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    // Getting a value from a JDBC SQLInput (useful for Oracle OBJECT types)
    @Override
    public void get(final BindingGetSQLInputContext<VERTX_JSON> ctx) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }
}
