/*
 * Copyright (c) 2021, 2022 - Yupiik SAS - https://www.yupiik.com
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package io.yupiik.bundlebee.maven.generated.mojo;

import org.apache.maven.plugins.annotations.Mojo;
import org.apache.maven.plugins.annotations.Parameter;

import io.yupiik.bundlebee.core.BundleBee;
import io.yupiik.bundlebee.maven.mojo.BaseMojo;

import java.util.Map;
import java.util.Set;
import java.util.stream.Stream;

/**
 * Diff an alveolus against a running cluster. The logic behind is to visit the configured alveolus and for each of its descriptor, query the cluster state and do a JSON-Diff between both. To avoid false positives, you will likely want to tune the ignored pointers which enable to drop dynamic data (managed by Kubernetes server).  The diff output has two types of diff:  * `JSON-Patch`: a JSON-Patch which, once applied on the actual state will bring up the state to the expected one, * `JSON`: means the Kubernetes server misses an alveolus descriptor and the expected one is fully printed  The diff line syntax is: `diff --$alveolusName a/$expectedLocalDescriptor b/$remoteDescriptor`. 
 */
@Mojo(name = "diff", requiresProject = false, threadSafe = true /* not strictly true but avoids warning inaccurate for builds */)
public class DiffMojo extends BaseMojo {
    /**
     * The attributes to keep from `StatefulSet` (`spec` children) descriptor on updates.
     */
    @Parameter(property = "bundlebee.kube.filters.statefuleset.spec.allowed", defaultValue = "replicas,template,updateStrategy,persistentVolumeClaimRetentionPolicy,minReadySeconds,serviceName,selector")
    private Set<String> kubeFiltersStatefulesetSpecAllowed;

    /**
     * Default header value for `PATCH` `content-type` requests header. It uses strategic merge patch algorithm but in some cases you just want to use `application/json` or (better) `application/merge-patch+json`. Annotation `io.yupiik.bundlebee/patchContentType` is also supported.
     */
    @Parameter(property = "bundlebee.kube.patchContentType", defaultValue = "application/strategic-merge-patch+json")
    private String kubePatchContentType;

    /**
     * If a proxy is needed to contact the target cluster API, its password if it needs an authentication (take care the JVM can nee `-Djdk.http.auth.tunneling.disabledSchemes=` options).
     */
    @Parameter(property = "bundlebee.kube.proxy.password", defaultValue = "<unset>")
    private String kubeProxyPassword;

    /**
     * Should HTTP client requests be limited and HTTP 427 responses be handled.
     */
    @Parameter(property = "bundlebee.kube.rateLimiter.enabled", defaultValue = "false")
    private boolean kubeRateLimiterEnabled;

    /**
     * By default a descriptor update is done using `PATCH` with strategic merge patch logic, if set to `true` it will use a plain `PUT`. Note that `io.yupiik.bundlebee/putOnUpdate` annotations can be set to `true` to force that in the descriptor itself and for cases it is not enough, you can set `force` to `true` to delete the descriptor before applying it again (move from clusterip to nodeport or the opposite in a serice for ex). Note that you can set it to `true` in a descriptor annotation `io.yupiik.bundlebee/force` too to not be global.
     */
    @Parameter(property = "bundlebee.kube.force", defaultValue = "false")
    private boolean kubeForce;

    /**
     * When using custom metadata (bundlebee ones or timestamp to force a rollout), where to inject them. Default uses labels since it enables to query them later on but you can switch it to annotations.
     */
    @Parameter(property = "bundlebee.kube.customMetadataInjectionPoint", defaultValue = "labels")
    private String kubeCustomMetadataInjectionPoint;

    /**
     * Default value for deletions of `propagationPolicy`. Values can be `Orphan`, `Foreground` and `Background`.
     */
    @Parameter(property = "bundlebee.kube.defaultPropagationPolicy", defaultValue = "Foreground")
    private String kubeDefaultPropagationPolicy;

    /**
     * When fetching a dependency using HTTP, the connection timeout for this dependency.
     */
    @Parameter(property = "bundlebee.maven.http.connectTimeout", defaultValue = "30000")
    private int mavenHttpConnectTimeout;

    /**
     * Enables to tolerate custom attributes in the descriptors. Typically used to drop `/$schema` attribute which enables a nice completion in editors. Values are `|` delimited and are either a JSON-Pointer (wrapped in a remove JSON-Patch) or directly a JSON-Patch. Using `none` ignores this processing.
     */
    @Parameter(property = "bundlebee.kube.implicitlyDroppedAttributes", defaultValue = "/$schema|/$bundlebeeIgnoredLintingRules")
    private String kubeImplicitlyDroppedAttributes;

    /**
     * Should SSL connector be validated or not.
     */
    @Parameter(property = "bundlebee.kube.validateSSL", defaultValue = "true")
    private boolean kubeValidateSSL;

    /**
     * If `true` http requests/responses to Kubernetes will be logged.
     */
    @Parameter(property = "bundlebee.kube.verbose", defaultValue = "false")
    private boolean kubeVerbose;

    /**
     * If `false` we first try to read `settings.xml` file(s) in `cache` location before the default one.
     */
    @Parameter(property = "bundlebee.maven.preferCustomSettingsXml", defaultValue = "true")
    private boolean mavenPreferCustomSettingsXml;

    /**
     * Enables to define resource mapping, syntax uses propeties one: `<lowercased resource kind>s = /apis/....`.
     */
    @Parameter(property = "bundlebee.kube.resourceMapping", defaultValue = "")
    private String kubeResourceMapping;

    /**
     * Rate limiting window duration in milliseconds (default being 1 second).
     */
    @Parameter(property = "bundlebee.kube.rateLimiter.window", defaultValue = "1000")
    private int kubeRateLimiterWindow;

    /**
     * The HTTP client connect timeout (in java Duration format), `none` can be used to ignore this setting.
     */
    @Parameter(property = "bundlebee.httpclient.connectTimeout", defaultValue = "none")
    private String httpclientConnectTimeout;

    /**
     * The HTTP client version, `none` mean the JVM default (v2), `HTTP_1_1` v1.1 and `HTTP_2` v2.0.
     */
    @Parameter(property = "bundlebee.httpclient.forcedHttpVersion", defaultValue = "none")
    private String httpclientForcedHttpVersion;

    /**
     * When kubeconfig (explicit or not) is used, the context to use. If not set it is taken from the kubeconfig itself.
     */
    @Parameter(property = "bundlebee.kube.context", defaultValue = "<unset>")
    private String kubeContext;

    /**
     * If a proxy is needed to contact the target cluster API, its port.
     */
    @Parameter(property = "bundlebee.kube.proxy.port", defaultValue = "3128")
    private int kubeProxyPort;

    /**
     * Where to cache maven dependencies. If set to `auto`, tries to read the system property `maven.repo.local` then the `settings.xml` `localRepository` and finally it would fallback on `$HOME/.m2/repository`.
     */
    @Parameter(property = "bundlebee.maven.cache", defaultValue = "auto")
    private String mavenCache;

    /**
     * When kubeconfig is not set the base API endpoint.
     */
    @Parameter(property = "bundlebee.kube.api", defaultValue = "http://localhost:8080")
    private String kubeApi;

    /**
     * If a proxy is needed to contact the target cluster API, its username if it needs an authentication (take care the JVM can nee `-Djdk.http.auth.tunneling.disabledSchemes=` options).
     */
    @Parameter(property = "bundlebee.kube.proxy.username", defaultValue = "<unset>")
    private String kubeProxyUsername;

    /**
     * If `true` we only use `cache` value and never fallback on default maven settings.xml location.
     */
    @Parameter(property = "bundlebee.maven.forceCustomSettingsXml", defaultValue = "false")
    private boolean mavenForceCustomSettingsXml;

    /**
     * Default snapshot repository, not set by default.
     */
    @Parameter(property = "bundlebee.maven.repositories.snapshot", defaultValue = "<unset>")
    private String mavenRepositoriesSnapshot;

    /**
     * `fieldValidation` - server side validation - value when applying a descriptor, values can be `Strict`, `Warn` pr `Ignore`. Note that using `skip` will ignore the query parameter.
     */
    @Parameter(property = "bundlebee.kube.fieldValidation", defaultValue = "Strict")
    private String kubeFieldValidation;

    /**
     * By default a descriptor update is done using `PATCH` with strategic merge patch logic, if set to `true` it will use a plain `PUT`. Note that `io.yupiik.bundlebee/putOnUpdate` annotations can be set to `true` to force that in the descriptor itself.
     */
    @Parameter(property = "bundlebee.kube.putOnUpdate", defaultValue = "false")
    private boolean kubePutOnUpdate;

    /**
     * Kubeconfig location. If set to `auto` it will try to guess from your `$HOME/.kube/config` file until you set it so `explicit` where it will use other `bundlebee.kube` properties to create the client. The content can also be set inline!
     */
    @Parameter(property = "kubeconfig", defaultValue = "auto")
    private String kubeconfig;

    /**
     * If `true` http requests/responses are skipped. Note that dry run implies verbose=true for the http client. Note that as of today, all responses are mocked by a HTTP 200 and an empty JSON payload.
     */
    @Parameter(property = "bundlebee.kube.dryRun", defaultValue = "false")
    private boolean kubeDryRun;

    /**
     * HTTP timeout in ms, ignored if <= 0.
     */
    @Parameter(property = "bundlebee.kube.http.timeout", defaultValue = "60000")
    private long kubeHttpTimeout;

    /**
     * If a proxy is needed to contact the target cluster API, its host, ignore if not set.
     */
    @Parameter(property = "bundlebee.kube.proxy.host", defaultValue = "<unset>")
    private String kubeProxyHost;

    /**
     * When `kubeconfig` is set to `explicit`, the bearer token to use (if set).
     */
    @Parameter(property = "bundlebee.kube.token", defaultValue = "<unset>")
    private String kubeToken;

    /**
     * How often to retry for a descriptor condition. Increasing it will reduce the pressure on the Kubernetes REST API (rate limiting for example).
     */
    @Parameter(property = "bundlebee.awaiter.retryInterval", defaultValue = "500")
    private long awaiterRetryInterval;

    /**
     * How many calls can be done if rate limiting is enabled. Note that setting it to `Integer.MAX_VALUE` will disable the client rate limiting and only enable server one.
     */
    @Parameter(property = "bundlebee.kube.rateLimiter.permits", defaultValue = "100")
    private int kubeRateLimiterPermits;

    /**
     * How many threads are allocated to async HTTP client, negative or zero value means to use common pool.
     */
    @Parameter(property = "bundlebee.httpclient.threads", defaultValue = "-1")
    private int httpclientThreads;

    /**
     * List of _kind_ of descriptors updates can be skipped, it is often useful for `PersistentVolumeClaim`.
     */
    @Parameter(property = "bundlebee.kube.skipUpdateForKinds", defaultValue = "PersistentVolumeClaim")
    private String kubeSkipUpdateForKinds;

    /**
     * Should YAML/JSON be logged when it can't be parsed.
     */
    @Parameter(property = "bundlebee.kube.logDescriptorOnParsingError", defaultValue = "true")
    private boolean kubeLogDescriptorOnParsingError;

    /**
     * Default release repository.
     */
    @Parameter(property = "bundlebee.maven.repositories.release", defaultValue = "https://repo.maven.apache.org/maven2/")
    private String mavenRepositoriesRelease;

    /**
     * If `true` GET http requests are not skipped when `dryRun` is true.
     */
    @Parameter(property = "bundlebee.kube.skipDryRunForGet", defaultValue = "false")
    private boolean kubeSkipDryRunForGet;

    /**
     * The HTTP client redirect policy. Default to `NORMAL`, can be set to `ALWAYS` or `NEVER`.
     */
    @Parameter(property = "bundlebee.httpclient.followRedirects", defaultValue = "NORMAL")
    private String httpclientFollowRedirects;

    /**
     * If a proxy is configured to use authentication, automatically set `-Djdk.http.auth.tunneling.disabledSchemes=`, note that setting it on the JVM is still more reliable depending how you run bundlebee (in particular with maven or embed). Important: the system property is "leaked", ie it is not cleaned up to limit side effect in concurrent mode.
     */
    @Parameter(property = "bundlebee.kube.proxy.setProxySystemProperties", defaultValue = "true")
    private boolean kubeProxySetProxySystemProperties;

    /**
     * Enable the download, i.e. ensure it runs only with local maven repository.
     */
    @Parameter(property = "bundlebee.maven.repositories.downloads.enabled", defaultValue = "false")
    private boolean mavenRepositoriesDownloadsEnabled;

    /**
     * When kubeconfig is not set the namespace to use.
     */
    @Parameter(property = "bundlebee.kube.namespace", defaultValue = "default")
    private String kubeNamespace;

    /**
     * Properties to define the headers to set per repository, syntax is `host1=headerName headerValue` and it supports as much lines as used repositories. Note that you can use maven `~/.m2/settings.xml` servers (potentially ciphered) username/password pairs. In this last case the server id must be `bundlebee.<server host>`. Still in settings.xml case, if the username is null the password value is used as raw `Authorization` header else username/password is encoded as a basic header.
     */
    @Parameter(property = "bundlebee.maven.repositories.httpHeaders", defaultValue = "<unset>")
    private String mavenRepositoriesHttpHeaders;

    /**
     * Alveolus name to inspect. When set to `auto`, it will look for all manifests found in the classpath. If you set manifest option, alveolus is set to `auto` and there is a single alveolus in it, this will default to it instead of using classpath deployment.
     */
    @Parameter(property = "bundlebee.diff.alveolus", defaultValue = "auto")
    private String alveolus;

    /**
     * Max concurrent requests to Kubernetes cluster. Value can't be less than `1`.
     */
    @Parameter(property = "bundlebee.diff.concurrency", defaultValue = "16")
    private int concurrency;

    /**
     * Same as `ignorableAttributes` but enables to keep defaults.
     */
    @Parameter(property = "bundlebee.diff.customIgnorableAttributes", defaultValue = "<unset>")
    private java.util.List customIgnorableAttributes;

    /**
     * Same as `ignoredPointers` but enables to keep defaults.
     */
    @Parameter(property = "bundlebee.diff.customIgnoredPointers", defaultValue = "<unset>")
    private java.util.List customIgnoredPointers;

    /**
     * If set only this descriptor is handled, not that you can use a regex if you make the value prefixed with `r/`. Note it generally only makes sense with verbose option.
     */
    @Parameter(property = "bundlebee.diff.descriptor", defaultValue = "<unset>")
    private String descriptor;

    /**
     * How to print the diff for each descriptor, `AUTO` means use `JSON-Patch` when there is a diff, `JSON` when a descriptor is missing and skip the content when it is the same. `JSON_PATCH` means always use `JSON-Patch` format. `JSON` means always show expected state.
     */
    @Parameter(property = "bundlebee.diff.diffType", defaultValue = "AUTO")
    private io.yupiik.bundlebee.core.command.impl.DiffCommand.DiffType diffType;

    /**
     * Diff location when `outputType` is `FILE`.
     */
    @Parameter(property = "bundlebee.diff.dumpLocation", defaultValue = "target/bundlebee.diff")
    private String dumpLocation;

    /**
     * Should JSON(-Diff) be formatted.
     */
    @Parameter(property = "bundlebee.diff.formatted", defaultValue = "true")
    private boolean formatted;

    /**
     * Root dependency to download to get the manifest. If set to `auto` it is assumed to be present in current classpath.
     */
    @Parameter(property = "bundlebee.diff.from", defaultValue = "auto")
    private String from;

    /**
     * A list (comma separated values) of attribute to ignore if in the Kubernetes model but not in local descriptor (it is mainly fields with defaults). Note that they behave as relative path (`endsWith`) so you should start them with a `/` in general.
     */
    @Parameter(property = "bundlebee.diff.ignorableAttributes", defaultValue = "/dnsPolicy,/terminationMessagePath,/terminationMessagePolicy,/schedulerName,/terminationGracePeriodSeconds,/resourceVersion,/uid,/spec/hostPath/type,/spec/persistentVolumeReclaimPolicy,/spec/claimRef/kind,/spec/claimRef/apiVersion,/spec/volumeName,/spec/volumeMode,/fieldRef/apiVersion")
    private java.util.List ignorableAttributes;

    /**
     * A list (comma separated values) of JSON-Pointer to ignore in the comparison of client/configured state and actual cluster state. Note that these ones will also affect `ignoreEmptyJsonObjectRemovals` since they are dropped before testing the value.
     */
    @Parameter(property = "bundlebee.diff.ignoredPointers", defaultValue = "/metadata/annotations,/metadata/creationTimestamp,/metadata/deletionGracePeriodSeconds,/metadata/generation,/metadata/managedFields,/metadata/ownerReferences,/metadata/resourceVersion,/metadata/selfLink,/metadata/uid,/status,/metadata/labels/bundlebee.root.alveolus.name,/metadata/labels/bundlebee.root.alveolus.version,/metadata/labels/bundlebee.timestamp")
    private java.util.List ignoredPointers;

    /**
     * Should a diff where the `op` is `remove` or `replace` and the associated `value` an empty JSON-Object/JSON-Array be ignored.
     */
    @Parameter(property = "bundlebee.diff.ignoreEmptyRemovals", defaultValue = "true")
    private boolean ignoreEmptyRemovals;

    /**
     * If `true`, the descriptors without any notable diff will be ignored from the report.
     */
    @Parameter(property = "bundlebee.diff.ignoreNoDiffEntries", defaultValue = "true")
    private boolean ignoreNoDiffEntries;

    /**
     * Manifest to load to start to find the alveolus. This optional setting mainly enables to use dependencies easily. Ignored if set to `skip`.
     */
    @Parameter(property = "bundlebee.diff.manifest", defaultValue = "skip")
    private String manifest;

    /**
     * If there are at least this number of differences then the build will fail, disabled if negative.
     */
    @Parameter(property = "bundlebee.diff.maxDifferences", defaultValue = "-1")
    private int maxDifferences;

    /**
     * How to dump the diff, by default (`LOG`) it will print it but `FILE` will store it in a local file (using `dumpLocation`).
     */
    @Parameter(property = "bundlebee.diff.outputType", defaultValue = "LOG")
    private io.yupiik.bundlebee.core.command.impl.DiffCommand.OutputType outputType;

    /**
     * If `true`, the local location of the description will be added to the diff.
     */
    @Parameter(property = "bundlebee.diff.showLocalSource", defaultValue = "false")
    private boolean showLocalSource;

    /**
     * Custom properties injected in the main, it is often used for placeholders.
     * If the key (tag in pom) starts with `bundlebee-placeholder-import` then the value is resolved as a properties file
     * which is injected in the resulting placeholders (indirect placeholders).
     */
    @Parameter(property = "bundlebee.diff.customPlaceholders")
    private Map<String, String> customPlaceholders;

    /**
     * Just an alias for the built-in manifest property to ease the pom configuration for all commands.
     */
    @Parameter(property = "bundlebee.manifest", defaultValue = "skip")
    private String defaultManifest;

    @Override
    public void doExecute() {
        new BundleBee().launch(Stream.concat(
                        Stream.of(
                                "diff",
                                "--bundlebee.kube.filters.statefuleset.spec.allowed",
                                String.join(",", kubeFiltersStatefulesetSpecAllowed),
                                "--bundlebee.kube.patchContentType",
                                String.valueOf(kubePatchContentType),
                                "--bundlebee.kube.proxy.password",
                                String.valueOf(kubeProxyPassword),
                                "--bundlebee.kube.rateLimiter.enabled",
                                String.valueOf(kubeRateLimiterEnabled),
                                "--bundlebee.kube.force",
                                String.valueOf(kubeForce),
                                "--bundlebee.kube.customMetadataInjectionPoint",
                                String.valueOf(kubeCustomMetadataInjectionPoint),
                                "--bundlebee.kube.defaultPropagationPolicy",
                                String.valueOf(kubeDefaultPropagationPolicy),
                                "--bundlebee.maven.http.connectTimeout",
                                String.valueOf(mavenHttpConnectTimeout),
                                "--bundlebee.kube.implicitlyDroppedAttributes",
                                String.valueOf(kubeImplicitlyDroppedAttributes),
                                "--bundlebee.kube.validateSSL",
                                String.valueOf(kubeValidateSSL),
                                "--bundlebee.kube.verbose",
                                String.valueOf(kubeVerbose),
                                "--bundlebee.maven.preferCustomSettingsXml",
                                String.valueOf(mavenPreferCustomSettingsXml),
                                "--bundlebee.kube.resourceMapping",
                                String.valueOf(kubeResourceMapping),
                                "--bundlebee.kube.rateLimiter.window",
                                String.valueOf(kubeRateLimiterWindow),
                                "--bundlebee.httpclient.connectTimeout",
                                String.valueOf(httpclientConnectTimeout),
                                "--bundlebee.httpclient.forcedHttpVersion",
                                String.valueOf(httpclientForcedHttpVersion),
                                "--bundlebee.kube.context",
                                String.valueOf(kubeContext),
                                "--bundlebee.kube.proxy.port",
                                String.valueOf(kubeProxyPort),
                                "--bundlebee.maven.cache",
                                String.valueOf(mavenCache),
                                "--bundlebee.kube.api",
                                String.valueOf(kubeApi),
                                "--bundlebee.kube.proxy.username",
                                String.valueOf(kubeProxyUsername),
                                "--bundlebee.maven.forceCustomSettingsXml",
                                String.valueOf(mavenForceCustomSettingsXml),
                                "--bundlebee.maven.repositories.snapshot",
                                String.valueOf(mavenRepositoriesSnapshot),
                                "--bundlebee.kube.fieldValidation",
                                String.valueOf(kubeFieldValidation),
                                "--bundlebee.kube.putOnUpdate",
                                String.valueOf(kubePutOnUpdate),
                                "--kubeconfig",
                                String.valueOf(kubeconfig),
                                "--bundlebee.kube.dryRun",
                                String.valueOf(kubeDryRun),
                                "--bundlebee.kube.http.timeout",
                                String.valueOf(kubeHttpTimeout),
                                "--bundlebee.kube.proxy.host",
                                String.valueOf(kubeProxyHost),
                                "--bundlebee.kube.token",
                                String.valueOf(kubeToken),
                                "--bundlebee.awaiter.retryInterval",
                                String.valueOf(awaiterRetryInterval),
                                "--bundlebee.kube.rateLimiter.permits",
                                String.valueOf(kubeRateLimiterPermits),
                                "--bundlebee.httpclient.threads",
                                String.valueOf(httpclientThreads),
                                "--bundlebee.kube.skipUpdateForKinds",
                                String.valueOf(kubeSkipUpdateForKinds),
                                "--bundlebee.kube.logDescriptorOnParsingError",
                                String.valueOf(kubeLogDescriptorOnParsingError),
                                "--bundlebee.maven.repositories.release",
                                String.valueOf(mavenRepositoriesRelease),
                                "--bundlebee.kube.skipDryRunForGet",
                                String.valueOf(kubeSkipDryRunForGet),
                                "--bundlebee.httpclient.followRedirects",
                                String.valueOf(httpclientFollowRedirects),
                                "--bundlebee.kube.proxy.setProxySystemProperties",
                                String.valueOf(kubeProxySetProxySystemProperties),
                                "--bundlebee.maven.repositories.downloads.enabled",
                                String.valueOf(mavenRepositoriesDownloadsEnabled),
                                "--bundlebee.kube.namespace",
                                String.valueOf(kubeNamespace),
                                "--bundlebee.maven.repositories.httpHeaders",
                                String.valueOf(mavenRepositoriesHttpHeaders),
                                "--alveolus",
                                String.valueOf(alveolus),
                                "--concurrency",
                                String.valueOf(concurrency),
                                "--customIgnorableAttributes",
                                String.join(",", customIgnorableAttributes),
                                "--customIgnoredPointers",
                                String.join(",", customIgnoredPointers),
                                "--descriptor",
                                String.valueOf(descriptor),
                                "--diffType",
                                String.valueOf(diffType),
                                "--dumpLocation",
                                String.valueOf(dumpLocation),
                                "--formatted",
                                String.valueOf(formatted),
                                "--from",
                                String.valueOf(from),
                                "--ignorableAttributes",
                                String.join(",", ignorableAttributes),
                                "--ignoredPointers",
                                String.join(",", ignoredPointers),
                                "--ignoreEmptyRemovals",
                                String.valueOf(ignoreEmptyRemovals),
                                "--ignoreNoDiffEntries",
                                String.valueOf(ignoreNoDiffEntries),
                                "--manifest",
                                String.valueOf("skip".equals(manifest) ? defaultManifest : manifest),
                                "--maxDifferences",
                                String.valueOf(maxDifferences),
                                "--outputType",
                                String.valueOf(outputType),
                                "--showLocalSource",
                                String.valueOf(showLocalSource)),
                        toArgs(customPlaceholders))
                .toArray(String[]::new));
    }
}
