/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.core.http;

import io.vertx.reactivex.RxHelper;
import io.vertx.reactivex.ObservableHelper;
import io.vertx.reactivex.FlowableHelper;
import io.vertx.reactivex.impl.AsyncResultMaybe;
import io.vertx.reactivex.impl.AsyncResultSingle;
import io.vertx.reactivex.impl.AsyncResultCompletable;
import io.vertx.reactivex.impl.AsyncResultFlowable;
import io.vertx.reactivex.WriteStreamObserver;
import io.vertx.reactivex.WriteStreamSubscriber;
import java.util.Map;
import java.util.Set;
import java.util.List;
import java.util.Iterator;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import io.vertx.core.Handler;
import io.vertx.core.AsyncResult;
import io.vertx.core.json.JsonObject;
import io.vertx.core.json.JsonArray;
import io.vertx.lang.rx.RxDelegate;
import io.vertx.lang.rx.RxGen;
import io.vertx.lang.rx.TypeArg;
import io.vertx.lang.rx.MappingIterator;

/**
 * Represents a server-side HTTP response.
 * <p>
 * An instance of this is created and associated to every instance of
 * {@link io.vertx.reactivex.core.http.HttpServerRequest} that.
 * <p>
 * It allows the developer to control the HTTP response that is sent back to the
 * client for a particular HTTP request.
 * <p>
 * It contains methods that allow HTTP headers and trailers to be set, and for a body to be written out to the response.
 * <p>
 * It also allows files to be streamed by the kernel directly from disk to the
 * outgoing HTTP connection, bypassing user space altogether (where supported by
 * the underlying operating system). This is a very efficient way of
 * serving files from the server since buffers do not have to be read one by one
 * from the file and written to the outgoing socket. If the developer wants to use directly a
 * {@link java.nio.channels.FileChannel} and manage its lifecycle use {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile}.
 * This is not yet supported in HTTP/2 for {@link io.vertx.core.http.impl.Http2ServerResponse}.
 * <p>
 * It implements {@link io.vertx.reactivex.core.streams.WriteStream} so it can be used with
 * {@link io.vertx.reactivex.core.streams.Pipe} to pipe data with flow control.
 *
 * <p>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpServerResponse original} non RX-ified interface using Vert.x codegen.
 */

@RxGen(io.vertx.core.http.HttpServerResponse.class)
public class HttpServerResponse implements RxDelegate, io.vertx.reactivex.core.streams.WriteStream<io.vertx.core.buffer.Buffer> {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpServerResponse that = (HttpServerResponse) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final TypeArg<HttpServerResponse> __TYPE_ARG = new TypeArg<>(    obj -> new HttpServerResponse((io.vertx.core.http.HttpServerResponse) obj),
    HttpServerResponse::getDelegate
  );

  private final io.vertx.core.http.HttpServerResponse delegate;
  
  public HttpServerResponse(io.vertx.core.http.HttpServerResponse delegate) {
    this.delegate = delegate;
  }

  public HttpServerResponse(Object delegate) {
    this.delegate = (io.vertx.core.http.HttpServerResponse)delegate;
  }

  @Override 
  public io.vertx.core.http.HttpServerResponse getDelegate() {
    return delegate;
  }

  private WriteStreamObserver<io.vertx.core.buffer.Buffer> observer;
  private WriteStreamSubscriber<io.vertx.core.buffer.Buffer> subscriber;

  public synchronized WriteStreamObserver<io.vertx.core.buffer.Buffer> toObserver() {
    if (observer == null) {
      observer = RxHelper.toObserver(getDelegate());
    }
    return observer;
  }

  public synchronized WriteStreamSubscriber<io.vertx.core.buffer.Buffer> toSubscriber() {
    if (subscriber == null) {
      subscriber = RxHelper.toSubscriber(getDelegate());
    }
    return subscriber;
  }

  private static final TypeArg<io.vertx.reactivex.core.http.HttpServerResponse> TYPE_ARG_0 = new TypeArg<io.vertx.reactivex.core.http.HttpServerResponse>(o1 -> io.vertx.reactivex.core.http.HttpServerResponse.newInstance((io.vertx.core.http.HttpServerResponse)o1), o1 -> o1.getDelegate());
  private static final TypeArg<io.vertx.reactivex.core.http.HttpServerResponse> TYPE_ARG_1 = new TypeArg<io.vertx.reactivex.core.http.HttpServerResponse>(o1 -> io.vertx.reactivex.core.http.HttpServerResponse.newInstance((io.vertx.core.http.HttpServerResponse)o1), o1 -> o1.getDelegate());
  private static final TypeArg<io.vertx.reactivex.core.http.HttpServerResponse> TYPE_ARG_2 = new TypeArg<io.vertx.reactivex.core.http.HttpServerResponse>(o1 -> io.vertx.reactivex.core.http.HttpServerResponse.newInstance((io.vertx.core.http.HttpServerResponse)o1), o1 -> o1.getDelegate());
  private static final TypeArg<io.vertx.reactivex.core.http.HttpServerResponse> TYPE_ARG_3 = new TypeArg<io.vertx.reactivex.core.http.HttpServerResponse>(o1 -> io.vertx.reactivex.core.http.HttpServerResponse.newInstance((io.vertx.core.http.HttpServerResponse)o1), o1 -> o1.getDelegate());

  /**
   * Write some data to the stream.
   *
   * <p> The data is usually put on an internal write queue, and the write actually happens
   * asynchronously. To avoid running out of memory by putting too much on the write queue,
   * check the {@link io.vertx.reactivex.core.streams.WriteStream#writeQueueFull} method before writing. This is done automatically if
   * using a .
   *
   * <p> When the <code>data</code> is moved from the queue to the actual medium, the returned
   *  will be completed with the write result, e.g the future is succeeded
   * when a server HTTP response buffer is written to the socket and failed if the remote
   * client has closed the socket while the data was still pending for write.
   * @param data the data to write
   * @return a future completed with the write result
   */
  public io.vertx.core.Future<java.lang.Void> write(io.vertx.core.buffer.Buffer data) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.write(data).map(val -> val);
    return ret;
  }

  /**
   * Write some data to the stream.
   *
   * <p> The data is usually put on an internal write queue, and the write actually happens
   * asynchronously. To avoid running out of memory by putting too much on the write queue,
   * check the {@link io.vertx.reactivex.core.streams.WriteStream#writeQueueFull} method before writing. This is done automatically if
   * using a .
   *
   * <p> When the <code>data</code> is moved from the queue to the actual medium, the returned
   *  will be completed with the write result, e.g the future is succeeded
   * when a server HTTP response buffer is written to the socket and failed if the remote
   * client has closed the socket while the data was still pending for write.
   * @param data the data to write
   * @return a future completed with the write result
   */
  public io.reactivex.Completable rxWrite(io.vertx.core.buffer.Buffer data) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.write(data).onComplete($handler);
    });
  }

  /**
   * This will return <code>true</code> if there are more bytes in the write queue than the value set using {@link io.vertx.reactivex.core.http.HttpServerResponse#setWriteQueueMaxSize}
   * @return <code>true</code> if write queue is full
   */
  public boolean writeQueueFull() { 
    boolean ret = delegate.writeQueueFull();
    return ret;
  }

  public io.vertx.reactivex.core.http.HttpServerResponse exceptionHandler(io.vertx.core.Handler<java.lang.Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  public io.vertx.reactivex.core.http.HttpServerResponse setWriteQueueMaxSize(int maxSize) { 
    delegate.setWriteQueueMaxSize(maxSize);
    return this;
  }

  public io.vertx.reactivex.core.http.HttpServerResponse drainHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.drainHandler(handler);
    return this;
  }

  /**
   * @return the HTTP status code of the response. The default is <code>200</code> representing <code>OK</code>.
   */
  public int getStatusCode() { 
    int ret = delegate.getStatusCode();
    return ret;
  }

  /**
   * Set the status code. If the status message hasn't been explicitly set, a default status message corresponding
   * to the code will be looked-up and used.
   * @param statusCode 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse setStatusCode(int statusCode) { 
    delegate.setStatusCode(statusCode);
    return this;
  }

  /**
   * @return the HTTP status message of the response. If this is not specified a default value will be used depending on what {@link io.vertx.reactivex.core.http.HttpServerResponse#setStatusCode} has been set to.
   */
  public java.lang.String getStatusMessage() { 
    java.lang.String ret = delegate.getStatusMessage();
    return ret;
  }

  /**
   * Set the status message
   * @param statusMessage 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse setStatusMessage(java.lang.String statusMessage) { 
    delegate.setStatusMessage(statusMessage);
    return this;
  }

  /**
   * If <code>chunked</code> is <code>true</code>, this response will use HTTP chunked encoding, and each call to write to the body
   * will correspond to a new HTTP chunk sent on the wire.
   * <p>
   * If chunked encoding is used the HTTP header <code>Transfer-Encoding</code> with a value of <code>Chunked</code> will be
   * automatically inserted in the response.
   * <p>
   * If <code>chunked</code> is <code>false</code>, this response will not use HTTP chunked encoding, and therefore the total size
   * of any data that is written in the respone body must be set in the <code>Content-Length</code> header <b>before</b> any
   * data is written out.
   * <p>
   * An HTTP chunked response is typically used when you do not know the total size of the request body up front.
   * @param chunked 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse setChunked(boolean chunked) { 
    delegate.setChunked(chunked);
    return this;
  }

  /**
   * @return is the response chunked?
   */
  public boolean isChunked() { 
    boolean ret = delegate.isChunked();
    return ret;
  }

  /**
   * @return The HTTP headers
   */
  public io.vertx.core.MultiMap headers() { 
    if (cached_0 != null) {
      return cached_0;
    }
    io.vertx.core.MultiMap ret = delegate.headers();
    cached_0 = ret;
    return ret;
  }

  /**
   * Put an HTTP header
   * @param name the header name
   * @param value the header value.
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putHeader(java.lang.String name, java.lang.String value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * @return The HTTP trailers
   */
  public io.vertx.core.MultiMap trailers() { 
    if (cached_1 != null) {
      return cached_1;
    }
    io.vertx.core.MultiMap ret = delegate.trailers();
    cached_1 = ret;
    return ret;
  }

  /**
   * Put an HTTP trailer
   * @param name the trailer name
   * @param value the trailer value
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putTrailer(java.lang.String name, java.lang.String value) { 
    delegate.putTrailer(name, value);
    return this;
  }

  /**
   * Set a close handler for the response, this is called when the underlying connection is closed and the response
   * was still using the connection.
   * <p>
   * For HTTP/1.x it is called when the connection is closed before <code>end()</code> is called, therefore it is not
   * guaranteed to be called.
   * <p>
   * For HTTP/2 it is called when the related stream is closed, and therefore it will be always be called.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse closeHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.closeHandler(handler);
    return this;
  }

  /**
   * Set an end handler for the response. This will be called when the response is disposed to allow consistent cleanup
   * of the response.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse endHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.endHandler(handler);
    return this;
  }

  /**
   * Send the response headers.
   * @return a future notified by the success or failure of the write
   */
  public io.vertx.core.Future<java.lang.Void> writeHead() { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.writeHead().map(val -> val);
    return ret;
  }

  /**
   * Send the response headers.
   * @return a future notified by the success or failure of the write
   */
  public io.reactivex.Completable rxWriteHead() { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.writeHead().onComplete($handler);
    });
  }

  /**
   * Write a {@link java.lang.String} to the response body, encoded using the encoding <code>enc</code>.
   * @param chunk the string to write
   * @param enc the encoding to use
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> write(java.lang.String chunk, java.lang.String enc) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.write(chunk, enc).map(val -> val);
    return ret;
  }

  /**
   * Write a {@link java.lang.String} to the response body, encoded using the encoding <code>enc</code>.
   * @param chunk the string to write
   * @param enc the encoding to use
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxWrite(java.lang.String chunk, java.lang.String enc) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.write(chunk, enc).onComplete($handler);
    });
  }

  /**
   * Write a {@link java.lang.String} to the response body, encoded in UTF-8.
   * @param chunk the string to write
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> write(java.lang.String chunk) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.write(chunk).map(val -> val);
    return ret;
  }

  /**
   * Write a {@link java.lang.String} to the response body, encoded in UTF-8.
   * @param chunk the string to write
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxWrite(java.lang.String chunk) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.write(chunk).onComplete($handler);
    });
  }

  /**
   * Used to write an interim 100 Continue response to signify that the client should send the rest of the request.
   * Must only be used if the request contains an "Expect:100-Continue" header
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.core.Future<java.lang.Void> writeContinue() { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.writeContinue().map(val -> val);
    return ret;
  }

  /**
   * Used to write an interim 100 Continue response to signify that the client should send the rest of the request.
   * Must only be used if the request contains an "Expect:100-Continue" header
   * @return a reference to this, so the API can be used fluently
   */
  public io.reactivex.Completable rxWriteContinue() { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.writeContinue().onComplete($handler);
    });
  }

  /**
   * Used to write an interim 103 Early Hints response to return some HTTP headers before the final HTTP message.
   * @param headers headers to write
   * @return a future
   */
  public io.vertx.core.Future<java.lang.Void> writeEarlyHints(io.vertx.core.MultiMap headers) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.writeEarlyHints(headers).map(val -> val);
    return ret;
  }

  /**
   * Used to write an interim 103 Early Hints response to return some HTTP headers before the final HTTP message.
   * @param headers headers to write
   * @return a future
   */
  public io.reactivex.Completable rxWriteEarlyHints(io.vertx.core.MultiMap headers) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.writeEarlyHints(headers).onComplete($handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but writes a String in UTF-8 encoding before ending the response.
   * @param chunk the string to write before ending the response
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> end(java.lang.String chunk) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.end(chunk).map(val -> val);
    return ret;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but writes a String in UTF-8 encoding before ending the response.
   * @param chunk the string to write before ending the response
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxEnd(java.lang.String chunk) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.end(chunk).onComplete($handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but writes a String with the specified encoding before ending the response.
   * @param chunk the string to write before ending the response
   * @param enc the encoding to use
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> end(java.lang.String chunk, java.lang.String enc) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.end(chunk, enc).map(val -> val);
    return ret;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but writes a String with the specified encoding before ending the response.
   * @param chunk the string to write before ending the response
   * @param enc the encoding to use
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxEnd(java.lang.String chunk, java.lang.String enc) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.end(chunk, enc).onComplete($handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but writes some data to the response body before ending. If the response is not chunked and
   * no other data has been written then the @code{Content-Length} header will be automatically set.
   * @param chunk the buffer to write before ending the response
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> end(io.vertx.core.buffer.Buffer chunk) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.end(chunk).map(val -> val);
    return ret;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#end} but writes some data to the response body before ending. If the response is not chunked and
   * no other data has been written then the @code{Content-Length} header will be automatically set.
   * @param chunk the buffer to write before ending the response
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxEnd(io.vertx.core.buffer.Buffer chunk) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.end(chunk).onComplete($handler);
    });
  }

  /**
   * Ends the response. If no data has been written to the response body,
   * the actual response won't get written until this method gets called.
   * <p>
   * Once the response has ended, it cannot be used any more.
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> end() { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.end().map(val -> val);
    return ret;
  }

  /**
   * Ends the response. If no data has been written to the response body,
   * the actual response won't get written until this method gets called.
   * <p>
   * Once the response has ended, it cannot be used any more.
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxEnd() { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.end().onComplete($handler);
    });
  }

  /**
   * Send the request with an empty body.
   * @return a future notified when the response has been written
   */
  public io.vertx.core.Future<java.lang.Void> send() { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.send().map(val -> val);
    return ret;
  }

  /**
   * Send the request with an empty body.
   * @return a future notified when the response has been written
   */
  public io.reactivex.Completable rxSend() { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.send().onComplete($handler);
    });
  }

  /**
   * Send the request with a string <code>body</code>.
   * @param body 
   * @return a future notified when the response has been written
   */
  public io.vertx.core.Future<java.lang.Void> send(java.lang.String body) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.send(body).map(val -> val);
    return ret;
  }

  /**
   * Send the request with a string <code>body</code>.
   * @param body 
   * @return a future notified when the response has been written
   */
  public io.reactivex.Completable rxSend(java.lang.String body) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.send(body).onComplete($handler);
    });
  }

  /**
   * Send the request with a buffer <code>body</code>.
   * @param body 
   * @return a future notified when the response has been written
   */
  public io.vertx.core.Future<java.lang.Void> send(io.vertx.core.buffer.Buffer body) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.send(body).map(val -> val);
    return ret;
  }

  /**
   * Send the request with a buffer <code>body</code>.
   * @param body 
   * @return a future notified when the response has been written
   */
  public io.reactivex.Completable rxSend(io.vertx.core.buffer.Buffer body) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.send(body).onComplete($handler);
    });
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   * @return a future notified when the last bytes of the response was sent
   */
  public io.vertx.core.Future<java.lang.Void> send(io.vertx.reactivex.core.streams.ReadStream<io.vertx.core.buffer.Buffer> body) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.send(body.getDelegate()).map(val -> val);
    return ret;
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   * @return a future notified when the last bytes of the response was sent
   */
  public io.reactivex.Completable rxSend(io.vertx.reactivex.core.streams.ReadStream<io.vertx.core.buffer.Buffer> body) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.send(body).onComplete($handler);
    });
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   * @return a future notified when the last bytes of the response was sent
   */
  public io.vertx.core.Future<java.lang.Void> send(io.reactivex.Flowable<io.vertx.core.buffer.Buffer> body) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.send(io.vertx.reactivex.impl.ReadStreamSubscriber.asReadStream(body, obj -> obj).resume()).map(val -> val);
    return ret;
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   * @return a future notified when the last bytes of the response was sent
   */
  public io.reactivex.Completable rxSend(io.reactivex.Flowable<io.vertx.core.buffer.Buffer> body) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.send(body).onComplete($handler);
    });
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param filename 
   * @return a future notified when the response has been written
   */
  public io.vertx.core.Future<java.lang.Void> sendFile(java.lang.String filename) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.sendFile(filename).map(val -> val);
    return ret;
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param filename 
   * @return a future notified when the response has been written
   */
  public io.reactivex.Completable rxSendFile(java.lang.String filename) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.sendFile(filename).onComplete($handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} using length @code{Long.MAX_VALUE} which means until the end of the
   * file.
   * @param filename path to the file to serve
   * @param offset offset to start serving from
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> sendFile(java.lang.String filename, long offset) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.sendFile(filename, offset).map(val -> val);
    return ret;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} using length @code{Long.MAX_VALUE} which means until the end of the
   * file.
   * @param filename path to the file to serve
   * @param offset offset to start serving from
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxSendFile(java.lang.String filename, long offset) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.sendFile(filename, offset).onComplete($handler);
    });
  }

  /**
   * Ask the OS to stream a file as specified by <code>filename</code> directly
   * from disk to the outgoing connection, bypassing userspace altogether
   * (where supported by the underlying operating system.
   * This is a very efficient way to serve files.<p>
   * The actual serve is asynchronous and may not complete until some time after this method has returned.
   * @param filename path to the file to serve
   * @param offset offset to start serving from
   * @param length the number of bytes to send
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> sendFile(java.lang.String filename, long offset, long length) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.sendFile(filename, offset, length).map(val -> val);
    return ret;
  }

  /**
   * Ask the OS to stream a file as specified by <code>filename</code> directly
   * from disk to the outgoing connection, bypassing userspace altogether
   * (where supported by the underlying operating system.
   * This is a very efficient way to serve files.<p>
   * The actual serve is asynchronous and may not complete until some time after this method has returned.
   * @param filename path to the file to serve
   * @param offset offset to start serving from
   * @param length the number of bytes to send
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxSendFile(java.lang.String filename, long offset, long length) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.sendFile(filename, offset, length).onComplete($handler);
    });
  }

  /**
   * @return has the response already ended?
   */
  public boolean ended() { 
    boolean ret = delegate.ended();
    return ret;
  }

  /**
   * @return has the underlying TCP connection corresponding to the request already been closed?
   */
  public boolean closed() { 
    boolean ret = delegate.closed();
    return ret;
  }

  /**
   * @return have the headers for the response already been written?
   */
  public boolean headWritten() { 
    boolean ret = delegate.headWritten();
    return ret;
  }

  /**
   * Provide a handler that will be called just before the headers are written to the wire.<p>
   * This provides a hook allowing you to add any more headers or do any more operations before this occurs.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse headersEndHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.headersEndHandler(handler);
    return this;
  }

  /**
   * Provides a handler that will be called after the last part of the body is written to the wire.
   * The handler is called asynchronously of when the response has been received by the client.
   * This provides a hook allowing you to do more operations once the request has been sent over the wire
   * such as resource cleanup.
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse bodyEndHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.bodyEndHandler(handler);
    return this;
  }

  /**
   * @return the total number of bytes written for the body of the response.
   */
  public long bytesWritten() { 
    long ret = delegate.bytesWritten();
    return ret;
  }

  /**
   * @return the id of the stream of this response,  for HTTP/1.x
   */
  public int streamId() { 
    int ret = delegate.streamId();
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with no headers.
   * @param method 
   * @param authority 
   * @param path 
   * @return 
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.http.HttpServerResponse> push(io.vertx.core.http.HttpMethod method, io.vertx.core.net.HostAndPort authority, java.lang.String path) { 
    io.vertx.core.Future<io.vertx.reactivex.core.http.HttpServerResponse> ret = delegate.push(method, authority, path).map(val -> io.vertx.reactivex.core.http.HttpServerResponse.newInstance((io.vertx.core.http.HttpServerResponse)val));
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with no headers.
   * @param method 
   * @param authority 
   * @param path 
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpServerResponse> rxPush(io.vertx.core.http.HttpMethod method, io.vertx.core.net.HostAndPort authority, java.lang.String path) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.push(method, authority, path).onComplete($handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @param headers 
   * @return 
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.http.HttpServerResponse> push(io.vertx.core.http.HttpMethod method, java.lang.String path, io.vertx.core.MultiMap headers) { 
    io.vertx.core.Future<io.vertx.reactivex.core.http.HttpServerResponse> ret = delegate.push(method, path, headers).map(val -> io.vertx.reactivex.core.http.HttpServerResponse.newInstance((io.vertx.core.http.HttpServerResponse)val));
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @param headers 
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpServerResponse> rxPush(io.vertx.core.http.HttpMethod method, java.lang.String path, io.vertx.core.MultiMap headers) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.push(method, path, headers).onComplete($handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @return 
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.http.HttpServerResponse> push(io.vertx.core.http.HttpMethod method, java.lang.String path) { 
    io.vertx.core.Future<io.vertx.reactivex.core.http.HttpServerResponse> ret = delegate.push(method, path).map(val -> io.vertx.reactivex.core.http.HttpServerResponse.newInstance((io.vertx.core.http.HttpServerResponse)val));
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#push} with the host copied from the current request.
   * @param method 
   * @param path 
   * @return 
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpServerResponse> rxPush(io.vertx.core.http.HttpMethod method, java.lang.String path) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.push(method, path).onComplete($handler);
    });
  }

  /**
   * Push a response to the client.<p/>
   *
   * The <code>handler</code> will be notified with a <i>success</i> when the push can be sent and with
   * a <i>failure</i> when the client has disabled push or reset the push before it has been sent.<p/>
   *
   * The <code>handler</code> may be queued if the client has reduced the maximum number of streams the server can push
   * concurrently.<p/>
   *
   * Push can be sent only for peer initiated streams and if the response is not ended.
   * @param method the method of the promised request
   * @param authority the authority of the promised request
   * @param path the path of the promised request
   * @param headers the headers of the promised request
   * @return a future notified when the response can be written
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.http.HttpServerResponse> push(io.vertx.core.http.HttpMethod method, io.vertx.core.net.HostAndPort authority, java.lang.String path, io.vertx.core.MultiMap headers) { 
    io.vertx.core.Future<io.vertx.reactivex.core.http.HttpServerResponse> ret = delegate.push(method, authority, path, headers).map(val -> io.vertx.reactivex.core.http.HttpServerResponse.newInstance((io.vertx.core.http.HttpServerResponse)val));
    return ret;
  }

  /**
   * Push a response to the client.<p/>
   *
   * The <code>handler</code> will be notified with a <i>success</i> when the push can be sent and with
   * a <i>failure</i> when the client has disabled push or reset the push before it has been sent.<p/>
   *
   * The <code>handler</code> may be queued if the client has reduced the maximum number of streams the server can push
   * concurrently.<p/>
   *
   * Push can be sent only for peer initiated streams and if the response is not ended.
   * @param method the method of the promised request
   * @param authority the authority of the promised request
   * @param path the path of the promised request
   * @param headers the headers of the promised request
   * @return a future notified when the response can be written
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpServerResponse> rxPush(io.vertx.core.http.HttpMethod method, io.vertx.core.net.HostAndPort authority, java.lang.String path, io.vertx.core.MultiMap headers) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.push(method, authority, path, headers).onComplete($handler);
    });
  }

  /**
   * Reset this HTTP/2 stream with the error code <code>0</code>.
   * @return 
   */
  public io.vertx.core.Future<java.lang.Void> reset() { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.reset().map(val -> val);
    return ret;
  }

  /**
   * Reset this HTTP/2 stream with the error code <code>0</code>.
   * @return 
   */
  public io.reactivex.Completable rxReset() { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.reset().onComplete($handler);
    });
  }

  /**
   * Reset this response:
   * <p/>
   * <ul>
   *   <li>for HTTP/2, send an HTTP/2 reset frame with the specified error <code>code</code></li>
   *   <li>for HTTP/1.x, close the connection when the current response has not yet been sent</li>
   * </ul>
   * <p/>
   * When the response has already been sent nothing happens and <code>false</code> is returned as indicator.
   * @param code the error code
   * @return <code>true</code> when reset has been performed
   */
  public io.vertx.core.Future<java.lang.Void> reset(long code) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.reset(code).map(val -> val);
    return ret;
  }

  /**
   * Reset this response:
   * <p/>
   * <ul>
   *   <li>for HTTP/2, send an HTTP/2 reset frame with the specified error <code>code</code></li>
   *   <li>for HTTP/1.x, close the connection when the current response has not yet been sent</li>
   * </ul>
   * <p/>
   * When the response has already been sent nothing happens and <code>false</code> is returned as indicator.
   * @param code the error code
   * @return <code>true</code> when reset has been performed
   */
  public io.reactivex.Completable rxReset(long code) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.reset(code).onComplete($handler);
    });
  }

  /**
   * Write an HTTP/2 frame to the response, allowing to extend the HTTP/2 protocol.<p>
   *
   * The frame is sent immediatly and is not subject to flow control.
   * @param type the 8-bit frame type
   * @param flags the 8-bit frame flags
   * @param payload the frame payload
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.core.Future<java.lang.Void> writeCustomFrame(int type, int flags, io.vertx.core.buffer.Buffer payload) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.writeCustomFrame(type, flags, payload).map(val -> val);
    return ret;
  }

  /**
   * Write an HTTP/2 frame to the response, allowing to extend the HTTP/2 protocol.<p>
   *
   * The frame is sent immediatly and is not subject to flow control.
   * @param type the 8-bit frame type
   * @param flags the 8-bit frame flags
   * @param payload the frame payload
   * @return a reference to this, so the API can be used fluently
   */
  public io.reactivex.Completable rxWriteCustomFrame(int type, int flags, io.vertx.core.buffer.Buffer payload) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.writeCustomFrame(type, flags, payload).onComplete($handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#writeCustomFrame} but with an {@link io.vertx.core.http.HttpFrame}.
   * @param frame the frame to write
   * @return 
   */
  public io.vertx.core.Future<java.lang.Void> writeCustomFrame(io.vertx.core.http.HttpFrame frame) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.writeCustomFrame(frame).map(val -> val);
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#writeCustomFrame} but with an {@link io.vertx.core.http.HttpFrame}.
   * @param frame the frame to write
   * @return 
   */
  public io.reactivex.Completable rxWriteCustomFrame(io.vertx.core.http.HttpFrame frame) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.writeCustomFrame(frame).onComplete($handler);
    });
  }

  /**
   * Sets the priority of the associated stream
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param streamPriority the priority for this request's stream
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse setStreamPriority(io.vertx.core.http.StreamPriority streamPriority) { 
    delegate.setStreamPriority(streamPriority);
    return this;
  }

  /**
   * Add a cookie. This will be sent back to the client in the response.
   * @param cookie the cookie
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpServerResponse addCookie(io.vertx.core.http.Cookie cookie) { 
    delegate.addCookie(cookie);
    return this;
  }

  /**
   * Expire a cookie, notifying a User Agent to remove it from its cookie jar.
   *
   * NOTE: This method will only remove the first occurrence of the given name. Users probably may want to use:
   * {@link io.vertx.reactivex.core.http.HttpServerResponse#removeCookies}
   * @param name the name of the cookie
   * @return the cookie, if it existed, or null
   */
  public io.vertx.core.http.Cookie removeCookie(java.lang.String name) { 
    io.vertx.core.http.Cookie ret = delegate.removeCookie(name);
    return ret;
  }

  /**
   * Remove a cookie from the cookie set. If invalidate is <code>true</code> then it will expire a cookie, notifying a User
   * Agent to remove it from its cookie jar.
   *
   * NOTE: This method will only expire the first occurrence of the given name. Users probably may want to use:
   * {@link io.vertx.reactivex.core.http.HttpServerResponse#removeCookies}
   * @param name the name of the cookie
   * @param invalidate 
   * @return the cookie, if it existed, or <code>null</code>
   */
  public io.vertx.core.http.Cookie removeCookie(java.lang.String name, boolean invalidate) { 
    io.vertx.core.http.Cookie ret = delegate.removeCookie(name, invalidate);
    return ret;
  }

  /**
   * Expire all cookies, notifying a User Agent to remove it from its cookie jar.
   *
   * NOTE: the returned  is read-only. This means any attempt to modify (add or remove to the set), will
   * throw {@link java.lang.UnsupportedOperationException}.
   * @param name the name of the cookie
   * @return a read only set of affected cookies, if they existed, or an empty set.
   */
  public java.util.Set<io.vertx.core.http.Cookie> removeCookies(java.lang.String name) { 
    java.util.Set<io.vertx.core.http.Cookie> ret = delegate.removeCookies(name);
    return ret;
  }

  /**
   * Remove all cookies from the cookie set. If invalidate is <code>true</code> then it will expire a cookie, notifying a
   * User Agent to remove it from its cookie jar.
   *
   * NOTE: the returned  is read-only. This means any attempt to modify (add or remove to the set), will
   * throw {@link java.lang.UnsupportedOperationException}.
   * @param name the name of the cookie
   * @param invalidate invalidate from the user agent
   * @return a read only set of affected cookies, if they existed, or an empty set.
   */
  public java.util.Set<io.vertx.core.http.Cookie> removeCookies(java.lang.String name, boolean invalidate) { 
    java.util.Set<io.vertx.core.http.Cookie> ret = delegate.removeCookies(name, invalidate);
    return ret;
  }

  /**
   * Expires a cookie from the cookie set. This will notify a User Agent to remove it from its cookie jar.
   * @param name the name of the cookie
   * @param domain the domain of the cookie
   * @param path the path of the cookie
   * @return the cookie, if it existed, or <code>null</code>
   */
  public io.vertx.core.http.Cookie removeCookie(java.lang.String name, java.lang.String domain, java.lang.String path) { 
    io.vertx.core.http.Cookie ret = delegate.removeCookie(name, domain, path);
    return ret;
  }

  /**
   * Remove a cookie from the cookie set. If invalidate is <code>true</code> then it will expire a cookie, notifying a User
   * Agent to remove it from its cookie jar.
   * @param name the name of the cookie
   * @param domain the domain of the cookie
   * @param path the path of the cookie
   * @param invalidate 
   * @return the cookie, if it existed, or <code>null</code>
   */
  public io.vertx.core.http.Cookie removeCookie(java.lang.String name, java.lang.String domain, java.lang.String path, boolean invalidate) { 
    io.vertx.core.http.Cookie ret = delegate.removeCookie(name, domain, path, invalidate);
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#putHeader} but using CharSequence
   * @param name 
   * @param value 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putHeader(java.lang.CharSequence name, java.lang.CharSequence value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#putHeader} but providing multiple values via a String Iterable
   * @param name 
   * @param values 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putHeader(java.lang.String name, java.lang.Iterable<java.lang.String> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#putHeader} but with CharSequence Iterable
   * @param name 
   * @param values 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putHeader(java.lang.CharSequence name, java.lang.Iterable<java.lang.CharSequence> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#putTrailer} but using CharSequence
   * @param name 
   * @param value 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putTrailer(java.lang.CharSequence name, java.lang.CharSequence value) { 
    delegate.putTrailer(name, value);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#putTrailer} but providing multiple values via a String Iterable
   * @param name 
   * @param values 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putTrailer(java.lang.String name, java.lang.Iterable<java.lang.String> values) { 
    delegate.putTrailer(name, values);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpServerResponse#putTrailer} but with CharSequence Iterable
   * @param name 
   * @param value 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpServerResponse putTrailer(java.lang.CharSequence name, java.lang.Iterable<java.lang.CharSequence> value) { 
    delegate.putTrailer(name, value);
    return this;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} using length @code{Long.MAX_VALUE} which means until the end of the
   * file.
   * @param channel the file channel to the file to serve
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> sendFile(java.nio.channels.FileChannel channel) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.sendFile(channel).map(val -> val);
    return ret;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} using length @code{Long.MAX_VALUE} which means until the end of the
   * file.
   * @param channel the file channel to the file to serve
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxSendFile(java.nio.channels.FileChannel channel) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.sendFile(channel).onComplete($handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} using length @code{Long.MAX_VALUE} which means until the end of the
   * file.
   * @param channel the file channel to the file to serve
   * @param offset offset to start serving from
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> sendFile(java.nio.channels.FileChannel channel, long offset) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.sendFile(channel, offset).map(val -> val);
    return ret;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} using length @code{Long.MAX_VALUE} which means until the end of the
   * file.
   * @param channel the file channel to the file to serve
   * @param offset offset to start serving from
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxSendFile(java.nio.channels.FileChannel channel, long offset) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.sendFile(channel, offset).onComplete($handler);
    });
  }

  /**
   * Ask the OS to stream a file as specified by <code>channel</code> directly
   * from disk to the outgoing connection, bypassing userspace altogether
   * (where supported by the underlying operating system). Contrary to {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile},
   * the caller is responsible to close <code>channel</code> when no more needed.
   * This is a very efficient way to serve files.<p>
   * The actual serve is asynchronous and may not complete until some time after this method has returned.
   * The developer is responsible to set the adequate Content-Type with {@link io.vertx.reactivex.core.http.HttpServerResponse#putHeader}. If not
   * application/octet-stream will be set as default Content-Type.
   * @param channel the file channel to the file to serve
   * @param offset offset to start serving from
   * @param length the number of bytes to send
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> sendFile(java.nio.channels.FileChannel channel, long offset, long length) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.sendFile(channel, offset, length).map(val -> val);
    return ret;
  }

  /**
   * Ask the OS to stream a file as specified by <code>channel</code> directly
   * from disk to the outgoing connection, bypassing userspace altogether
   * (where supported by the underlying operating system). Contrary to {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile},
   * the caller is responsible to close <code>channel</code> when no more needed.
   * This is a very efficient way to serve files.<p>
   * The actual serve is asynchronous and may not complete until some time after this method has returned.
   * The developer is responsible to set the adequate Content-Type with {@link io.vertx.reactivex.core.http.HttpServerResponse#putHeader}. If not
   * application/octet-stream will be set as default Content-Type.
   * @param channel the file channel to the file to serve
   * @param offset offset to start serving from
   * @param length the number of bytes to send
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxSendFile(java.nio.channels.FileChannel channel, long offset, long length) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.sendFile(channel, offset, length).onComplete($handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} with 
   * @param file the file to serve
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> sendFile(java.io.RandomAccessFile file) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.sendFile(file).map(val -> val);
    return ret;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} with 
   * @param file the file to serve
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxSendFile(java.io.RandomAccessFile file) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.sendFile(file).onComplete($handler);
    });
  }

  /**
   *
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} with 
   * @param file the file to serve
   * @param offset offset to start serving from
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> sendFile(java.io.RandomAccessFile file, long offset) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.sendFile(file, offset).map(val -> val);
    return ret;
  }

  /**
   *
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} with 
   * @param file the file to serve
   * @param offset offset to start serving from
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxSendFile(java.io.RandomAccessFile file, long offset) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.sendFile(file, offset).onComplete($handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} with 
   * @param file the file to serve
   * @param offset offset to start serving from
   * @param length the number of bytes to send
   * @return a future completed with the body result
   */
  public io.vertx.core.Future<java.lang.Void> sendFile(java.io.RandomAccessFile file, long offset, long length) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.sendFile(file, offset, length).map(val -> val);
    return ret;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpServerResponse#sendFile} with 
   * @param file the file to serve
   * @param offset offset to start serving from
   * @param length the number of bytes to send
   * @return a future completed with the body result
   */
  public io.reactivex.Completable rxSendFile(java.io.RandomAccessFile file, long offset, long length) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.sendFile(file, offset, length).onComplete($handler);
    });
  }

  private io.vertx.core.MultiMap cached_0;
  private io.vertx.core.MultiMap cached_1;
  public static HttpServerResponse newInstance(io.vertx.core.http.HttpServerResponse arg) {
    return arg != null ? new HttpServerResponse(arg) : null;
  }

}
