/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.core.http;

import io.vertx.reactivex.RxHelper;
import io.vertx.reactivex.ObservableHelper;
import io.vertx.reactivex.FlowableHelper;
import io.vertx.reactivex.impl.AsyncResultMaybe;
import io.vertx.reactivex.impl.AsyncResultSingle;
import io.vertx.reactivex.impl.AsyncResultCompletable;
import io.vertx.reactivex.impl.AsyncResultFlowable;
import io.vertx.reactivex.WriteStreamObserver;
import io.vertx.reactivex.WriteStreamSubscriber;
import java.util.Map;
import java.util.Set;
import java.util.List;
import java.util.Iterator;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import io.vertx.core.Handler;
import io.vertx.core.AsyncResult;
import io.vertx.core.json.JsonObject;
import io.vertx.core.json.JsonArray;
import io.vertx.lang.rx.RxDelegate;
import io.vertx.lang.rx.RxGen;
import io.vertx.lang.rx.TypeArg;
import io.vertx.lang.rx.MappingIterator;

/**
 * Represents a client-side HTTP request.
 * <p>
 * Instances are created by an {@link io.vertx.reactivex.core.http.HttpClient} instance, via one of the methods corresponding to the
 * specific HTTP methods, or the generic request methods. On creation the request will not have been written to the
 * wire.
 * <p>
 * Once a request has been obtained, headers can be set on it, and data can be written to its body if required. Once
 * you are ready to send the request, one of the {@link io.vertx.reactivex.core.http.HttpClientRequest#end} methods should be called.
 * <p>
 * Nothing is actually sent until the request has been internally assigned an HTTP connection.
 * <p>
 * The {@link io.vertx.reactivex.core.http.HttpClient} instance will return an instance of this class immediately, even if there are no HTTP
 * connections available in the pool. Any requests sent before a connection is assigned will be queued
 * internally and actually sent when an HTTP connection becomes available from the pool.
 * <p>
 * The headers of the request are queued for writing either when the {@link io.vertx.reactivex.core.http.HttpClientRequest#end} method is called, or, when the first
 * part of the body is written, whichever occurs first.
 * <p>
 * This class supports both chunked and non-chunked HTTP.
 * <p>
 * It implements {@link io.vertx.reactivex.core.streams.WriteStream} so it can be used with
 * {@link io.vertx.reactivex.core.streams.Pipe} to pipe data with flow control.
 * <p>
 * An example of using this class is as follows:
 * <p>
 *
 * <p>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.http.HttpClientRequest original} non RX-ified interface using Vert.x codegen.
 */

@RxGen(io.vertx.core.http.HttpClientRequest.class)
public class HttpClientRequest implements RxDelegate, io.vertx.reactivex.core.streams.WriteStream<io.vertx.core.buffer.Buffer> {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    HttpClientRequest that = (HttpClientRequest) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final TypeArg<HttpClientRequest> __TYPE_ARG = new TypeArg<>(    obj -> new HttpClientRequest((io.vertx.core.http.HttpClientRequest) obj),
    HttpClientRequest::getDelegate
  );

  private final io.vertx.core.http.HttpClientRequest delegate;
  
  public HttpClientRequest(io.vertx.core.http.HttpClientRequest delegate) {
    this.delegate = delegate;
  }

  public HttpClientRequest(Object delegate) {
    this.delegate = (io.vertx.core.http.HttpClientRequest)delegate;
  }

  @Override 
  public io.vertx.core.http.HttpClientRequest getDelegate() {
    return delegate;
  }

  private WriteStreamObserver<io.vertx.core.buffer.Buffer> observer;
  private WriteStreamSubscriber<io.vertx.core.buffer.Buffer> subscriber;

  public synchronized WriteStreamObserver<io.vertx.core.buffer.Buffer> toObserver() {
    if (observer == null) {
      observer = RxHelper.toObserver(getDelegate());
    }
    return observer;
  }

  public synchronized WriteStreamSubscriber<io.vertx.core.buffer.Buffer> toSubscriber() {
    if (subscriber == null) {
      subscriber = RxHelper.toSubscriber(getDelegate());
    }
    return subscriber;
  }

  private static final TypeArg<io.vertx.reactivex.core.http.HttpClientResponse> TYPE_ARG_0 = new TypeArg<io.vertx.reactivex.core.http.HttpClientResponse>(o1 -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)o1), o1 -> o1.getDelegate());
  private static final TypeArg<io.vertx.reactivex.core.http.HttpClientResponse> TYPE_ARG_1 = new TypeArg<io.vertx.reactivex.core.http.HttpClientResponse>(o1 -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)o1), o1 -> o1.getDelegate());
  private static final TypeArg<io.vertx.reactivex.core.http.HttpClientResponse> TYPE_ARG_2 = new TypeArg<io.vertx.reactivex.core.http.HttpClientResponse>(o1 -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)o1), o1 -> o1.getDelegate());
  private static final TypeArg<io.vertx.reactivex.core.http.HttpClientResponse> TYPE_ARG_3 = new TypeArg<io.vertx.reactivex.core.http.HttpClientResponse>(o1 -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)o1), o1 -> o1.getDelegate());
  private static final TypeArg<io.vertx.reactivex.core.http.HttpClientResponse> TYPE_ARG_4 = new TypeArg<io.vertx.reactivex.core.http.HttpClientResponse>(o1 -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)o1), o1 -> o1.getDelegate());
  private static final TypeArg<io.vertx.reactivex.core.http.HttpClientResponse> TYPE_ARG_5 = new TypeArg<io.vertx.reactivex.core.http.HttpClientResponse>(o1 -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)o1), o1 -> o1.getDelegate());
  private static final TypeArg<io.vertx.reactivex.core.http.HttpClientResponse> TYPE_ARG_6 = new TypeArg<io.vertx.reactivex.core.http.HttpClientResponse>(o1 -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)o1), o1 -> o1.getDelegate());

  /**
   * Write some data to the stream.
   *
   * <p> The data is usually put on an internal write queue, and the write actually happens
   * asynchronously. To avoid running out of memory by putting too much on the write queue,
   * check the {@link io.vertx.reactivex.core.streams.WriteStream#writeQueueFull} method before writing. This is done automatically if
   * using a .
   *
   * <p> When the <code>data</code> is moved from the queue to the actual medium, the returned
   *  will be completed with the write result, e.g the future is succeeded
   * when a server HTTP response buffer is written to the socket and failed if the remote
   * client has closed the socket while the data was still pending for write.
   * @param data the data to write
   * @return a future completed with the write result
   */
  public io.vertx.core.Future<java.lang.Void> write(io.vertx.core.buffer.Buffer data) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.write(data).map(val -> val);
    return ret;
  }

  /**
   * Write some data to the stream.
   *
   * <p> The data is usually put on an internal write queue, and the write actually happens
   * asynchronously. To avoid running out of memory by putting too much on the write queue,
   * check the {@link io.vertx.reactivex.core.streams.WriteStream#writeQueueFull} method before writing. This is done automatically if
   * using a .
   *
   * <p> When the <code>data</code> is moved from the queue to the actual medium, the returned
   *  will be completed with the write result, e.g the future is succeeded
   * when a server HTTP response buffer is written to the socket and failed if the remote
   * client has closed the socket while the data was still pending for write.
   * @param data the data to write
   * @return a future completed with the write result
   */
  public io.reactivex.Completable rxWrite(io.vertx.core.buffer.Buffer data) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.write(data).onComplete($handler);
    });
  }

  /**
   * This will return <code>true</code> if there are more bytes in the write queue than the value set using {@link io.vertx.reactivex.core.http.HttpClientRequest#setWriteQueueMaxSize}
   * @return <code>true</code> if write queue is full
   */
  public boolean writeQueueFull() { 
    boolean ret = delegate.writeQueueFull();
    return ret;
  }

  public io.vertx.reactivex.core.http.HttpClientRequest exceptionHandler(io.vertx.core.Handler<java.lang.Throwable> handler) { 
    delegate.exceptionHandler(handler);
    return this;
  }

  public io.vertx.reactivex.core.http.HttpClientRequest setWriteQueueMaxSize(int maxSize) { 
    delegate.setWriteQueueMaxSize(maxSize);
    return this;
  }

  public io.vertx.reactivex.core.http.HttpClientRequest drainHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.drainHandler(handler);
    return this;
  }

  /**
   * Override the request authority, when using HTTP/1.x this overrides the request <code>host</code> header, when using
   * HTTP/2 this sets the <code>authority</code> pseudo header. When the port is a negative value, the default
   * scheme port will be used.
   *
   * <p>The default request authority is the server host and port when connecting to the server.
   * @param authority override the request authority
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest authority(io.vertx.core.net.HostAndPort authority) { 
    delegate.authority(authority);
    return this;
  }

  /**
   * Set the request to follow HTTP redirects up to {@link io.vertx.core.http.HttpClientOptions}.
   * @param followRedirects <code>true</code> to follow HTTP redirects
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setFollowRedirects(boolean followRedirects) { 
    delegate.setFollowRedirects(followRedirects);
    return this;
  }

  /**
   * @return whether HTTP redirections should be followed
   */
  public boolean isFollowRedirects() { 
    boolean ret = delegate.isFollowRedirects();
    return ret;
  }

  /**
   * Set the max number of HTTP redirects this request will follow. The default is <code>0</code> which means
   * no redirects.
   * @param maxRedirects the number of HTTP redirect to follow
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setMaxRedirects(int maxRedirects) { 
    delegate.setMaxRedirects(maxRedirects);
    return this;
  }

  /**
   * @return the maximum number of HTTP redirections to follow
   */
  public int getMaxRedirects() { 
    int ret = delegate.getMaxRedirects();
    return ret;
  }

  /**
   * @return the number of followed redirections for the current HTTP request
   */
  public int numberOfRedirections() { 
    int ret = delegate.numberOfRedirections();
    return ret;
  }

  /**
   * If chunked is true then the request will be set into HTTP chunked mode
   * @param chunked true if chunked encoding
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setChunked(boolean chunked) { 
    delegate.setChunked(chunked);
    return this;
  }

  /**
   * @return Is the request chunked?
   */
  public boolean isChunked() { 
    boolean ret = delegate.isChunked();
    return ret;
  }

  /**
   * The HTTP method for the request.
   * @return 
   */
  public io.vertx.core.http.HttpMethod getMethod() { 
    io.vertx.core.http.HttpMethod ret = delegate.getMethod();
    return ret;
  }

  /**
   * Set the HTTP method for this request.
   * @param method the HTTP method
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setMethod(io.vertx.core.http.HttpMethod method) { 
    delegate.setMethod(method);
    return this;
  }

  /**
   * @return the absolute URI corresponding to the HTTP request
   */
  public java.lang.String absoluteURI() { 
    java.lang.String ret = delegate.absoluteURI();
    return ret;
  }

  /**
   * @return The URI of the request.
   */
  public java.lang.String getURI() { 
    java.lang.String ret = delegate.getURI();
    return ret;
  }

  /**
   * Set the request uri.
   * @param uri the request uri
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setURI(java.lang.String uri) { 
    delegate.setURI(uri);
    return this;
  }

  /**
   * @return The path part of the uri. For example /somepath/somemorepath/someresource.foo
   */
  public java.lang.String path() { 
    java.lang.String ret = delegate.path();
    return ret;
  }

  /**
   * @return the query part of the uri. For example someparam=32&amp;someotherparam=x
   */
  public java.lang.String query() { 
    java.lang.String ret = delegate.query();
    return ret;
  }

  /**
   * @return The HTTP headers
   */
  public io.vertx.core.MultiMap headers() { 
    if (cached_0 != null) {
      return cached_0;
    }
    io.vertx.core.MultiMap ret = delegate.headers();
    cached_0 = ret;
    return ret;
  }

  /**
   * Put an HTTP header
   * @param name The header name
   * @param value The header value
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest putHeader(java.lang.String name, java.lang.String value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * Set the trace operation of this request.
   * @param op the operation
   * @return @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest traceOperation(java.lang.String op) { 
    io.vertx.reactivex.core.http.HttpClientRequest ret = io.vertx.reactivex.core.http.HttpClientRequest.newInstance((io.vertx.core.http.HttpClientRequest)delegate.traceOperation(op));
    return ret;
  }

  /**
   * @return the trace operation of this request
   */
  public java.lang.String traceOperation() { 
    java.lang.String ret = delegate.traceOperation();
    return ret;
  }

  /**
   * @return the HTTP version for this request
   */
  public io.vertx.core.http.HttpVersion version() { 
    io.vertx.core.http.HttpVersion ret = delegate.version();
    return ret;
  }

  /**
   * Write a {@link java.lang.String} to the request body, encoded as UTF-8.
   * @param chunk the data chunk
   * @return a future completed with the result
   */
  public io.vertx.core.Future<java.lang.Void> write(java.lang.String chunk) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.write(chunk).map(val -> val);
    return ret;
  }

  /**
   * Write a {@link java.lang.String} to the request body, encoded as UTF-8.
   * @param chunk the data chunk
   * @return a future completed with the result
   */
  public io.reactivex.Completable rxWrite(java.lang.String chunk) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.write(chunk).onComplete($handler);
    });
  }

  /**
   * Write a {@link java.lang.String} to the request body, encoded using the encoding <code>enc</code>.
   * @param chunk the data chunk
   * @param enc the encoding
   * @return a future completed with the result
   */
  public io.vertx.core.Future<java.lang.Void> write(java.lang.String chunk, java.lang.String enc) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.write(chunk, enc).map(val -> val);
    return ret;
  }

  /**
   * Write a {@link java.lang.String} to the request body, encoded using the encoding <code>enc</code>.
   * @param chunk the data chunk
   * @param enc the encoding
   * @return a future completed with the result
   */
  public io.reactivex.Completable rxWrite(java.lang.String chunk, java.lang.String enc) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.write(chunk, enc).onComplete($handler);
    });
  }

  /**
   * If you send an HTTP request with the header <code>Expect</code> set to the value <code>100-continue</code>
   * and the server responds with an interim HTTP response with a status code of <code>100</code> and a Continue handler
   * has been set using this method, then the <code>handler</code> will be called.
   * <p>
   * You can then continue to write data to the request body and later end it. This is normally used in conjunction with
   * the {@link io.vertx.reactivex.core.http.HttpClientRequest#sendHead} method to force the request header to be written before the request has ended.
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest continueHandler(io.vertx.core.Handler<java.lang.Void> handler) { 
    delegate.continueHandler(handler);
    return this;
  }

  /**
   * If the server responds with an interim HTTP response with a status code of <code>103</code> and a Early Hints handler
   * has been set using this method, then the <code>handler</code> will be called.
   * @param handler 
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest earlyHintsHandler(io.vertx.core.Handler<io.vertx.core.MultiMap> handler) { 
    delegate.earlyHintsHandler(handler);
    return this;
  }

  public io.vertx.reactivex.core.http.HttpClientRequest redirectHandler(java.util.function.Function<io.vertx.reactivex.core.http.HttpClientResponse,io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientRequest>> handler) { 
    delegate.redirectHandler(new Function<io.vertx.core.http.HttpClientResponse,io.vertx.core.Future<io.vertx.core.http.HttpClientRequest>>() {
      public io.vertx.core.Future<io.vertx.core.http.HttpClientRequest> apply(io.vertx.core.http.HttpClientResponse arg) {
        io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientRequest> ret = handler.apply(io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)arg));
        return ret.map(val -> val.getDelegate());
      }
    });
    return this;
  }

  public io.vertx.reactivex.core.http.HttpClientRequest redirectHandler(io.reactivex.functions.Function<io.vertx.reactivex.core.http.HttpClientResponse,io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientRequest>> handler) { 
    delegate.redirectHandler(new Function<io.vertx.core.http.HttpClientResponse,io.vertx.core.Future<io.vertx.core.http.HttpClientRequest>>() {
      public io.vertx.core.Future<io.vertx.core.http.HttpClientRequest> apply(io.vertx.core.http.HttpClientResponse arg) {
        io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientRequest> ret;
        try {
          ret = handler.apply(io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)arg));
        } catch (Exception e) {
          return io.vertx.core.Future.failedFuture(e);
        }
        return io.vertx.reactivex.SingleHelper.toFuture(ret, obj -> obj.getDelegate());
      }
    });
    return this;
  }

  /**
   * Forces the head of the request to be written before {@link io.vertx.reactivex.core.http.HttpClientRequest#end} is called on the request or any data is
   * written to it.
   * <p>
   * This is normally used to implement HTTP 100-continue handling, see {@link io.vertx.reactivex.core.http.HttpClientRequest#continueHandler} for
   * more information.
   * @return a future notified when the {@link io.vertx.core.http.HttpVersion} if it can be determined or <code>null</code> otherwise
   */
  public io.vertx.core.Future<java.lang.Void> sendHead() { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.sendHead().map(val -> val);
    return ret;
  }

  /**
   * Forces the head of the request to be written before {@link io.vertx.reactivex.core.http.HttpClientRequest#end} is called on the request or any data is
   * written to it.
   * <p>
   * This is normally used to implement HTTP 100-continue handling, see {@link io.vertx.reactivex.core.http.HttpClientRequest#continueHandler} for
   * more information.
   * @return a future notified when the {@link io.vertx.core.http.HttpVersion} if it can be determined or <code>null</code> otherwise
   */
  public io.reactivex.Completable rxSendHead() { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.sendHead().onComplete($handler);
    });
  }

  /**
   * Create an HTTP tunnel to the server.
   *
   * <p> Send HTTP request headers to the server, then configures the transport to exchange
   * raw buffers when the server replies with an appropriate response:
   *
   * <ul>
   *   <li><code>200</code> for HTTP <code>CONNECT</code> method</li>
   *   <li><code>101</code> for HTTP/1.1 <code>GET</code> with <code>Upgrade</code> <code>connection</code> header</li>
   * </ul>
   *
   * <p> The <code>handler</code> is called after response headers are received.
   *
   * <p> Use {@link io.vertx.reactivex.core.http.HttpClientResponse#netSocket} to get a  for interacting
   * more conveniently with the server.
   *
   * <p> HTTP/1.1 pipe-lined requests are not supported.f
   * @return a future notified with the server response
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> connect() { 
    io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = delegate.connect().map(val -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)val));
    return ret;
  }

  /**
   * Create an HTTP tunnel to the server.
   *
   * <p> Send HTTP request headers to the server, then configures the transport to exchange
   * raw buffers when the server replies with an appropriate response:
   *
   * <ul>
   *   <li><code>200</code> for HTTP <code>CONNECT</code> method</li>
   *   <li><code>101</code> for HTTP/1.1 <code>GET</code> with <code>Upgrade</code> <code>connection</code> header</li>
   * </ul>
   *
   * <p> The <code>handler</code> is called after response headers are received.
   *
   * <p> Use {@link io.vertx.reactivex.core.http.HttpClientResponse#netSocket} to get a  for interacting
   * more conveniently with the server.
   *
   * <p> HTTP/1.1 pipe-lined requests are not supported.f
   * @return a future notified with the server response
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientResponse> rxConnect() { 
    return AsyncResultSingle.toSingle($handler -> {
      this.connect().onComplete($handler);
    });
  }

  /**
   * @return a future of the {@link io.vertx.reactivex.core.http.HttpClientResponse}, this method does not modify the current request being sent.
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> response() { 
    io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = delegate.response().map(val -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)val));
    return ret;
  }

  /**
   * @return a future of the {@link io.vertx.reactivex.core.http.HttpClientResponse}, this method does not modify the current request being sent.
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientResponse> rxResponse() { 
    return AsyncResultSingle.toSingle($handler -> {
      this.response().onComplete($handler);
    });
  }

  /**
   * Send the request with an empty body.
   * @return a future notified when the HTTP response is available
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> send() { 
    io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = delegate.send().map(val -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)val));
    return ret;
  }

  /**
   * Send the request with an empty body.
   * @return a future notified when the HTTP response is available
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientResponse> rxSend() { 
    return AsyncResultSingle.toSingle($handler -> {
      this.send().onComplete($handler);
    });
  }

  /**
   * Send the request with a string <code>body</code>.
   * @param body 
   * @return a future notified when the HTTP response is available
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> send(java.lang.String body) { 
    io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = delegate.send(body).map(val -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)val));
    return ret;
  }

  /**
   * Send the request with a string <code>body</code>.
   * @param body 
   * @return a future notified when the HTTP response is available
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientResponse> rxSend(java.lang.String body) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.send(body).onComplete($handler);
    });
  }

  /**
   * Send the request with a buffer <code>body</code>.
   * @param body 
   * @return a future notified when the HTTP response is available
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> send(io.vertx.core.buffer.Buffer body) { 
    io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = delegate.send(body).map(val -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)val));
    return ret;
  }

  /**
   * Send the request with a buffer <code>body</code>.
   * @param body 
   * @return a future notified when the HTTP response is available
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientResponse> rxSend(io.vertx.core.buffer.Buffer body) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.send(body).onComplete($handler);
    });
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClientRequest#send} but with a <code>form</code>. The content will be set to <code>application/x-www-form-urlencoded</code>
   * or <code>multipart/form-data</code> according to the nature of the form.
   * @param form the form to send
   * @return a future notified when the HTTP response is available
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> send(io.vertx.core.http.ClientForm form) { 
    io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = delegate.send(form).map(val -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)val));
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClientRequest#send} but with a <code>form</code>. The content will be set to <code>application/x-www-form-urlencoded</code>
   * or <code>multipart/form-data</code> according to the nature of the form.
   * @param form the form to send
   * @return a future notified when the HTTP response is available
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientResponse> rxSend(io.vertx.core.http.ClientForm form) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.send(form).onComplete($handler);
    });
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   * @return a future notified when the HTTP response is available
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> send(io.vertx.reactivex.core.streams.ReadStream<io.vertx.core.buffer.Buffer> body) { 
    io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = delegate.send(body.getDelegate()).map(val -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)val));
    return ret;
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   * @return a future notified when the HTTP response is available
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientResponse> rxSend(io.vertx.reactivex.core.streams.ReadStream<io.vertx.core.buffer.Buffer> body) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.send(body).onComplete($handler);
    });
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   * @return a future notified when the HTTP response is available
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> send(io.reactivex.Flowable<io.vertx.core.buffer.Buffer> body) { 
    io.vertx.core.Future<io.vertx.reactivex.core.http.HttpClientResponse> ret = delegate.send(io.vertx.reactivex.impl.ReadStreamSubscriber.asReadStream(body, obj -> obj).resume()).map(val -> io.vertx.reactivex.core.http.HttpClientResponse.newInstance((io.vertx.core.http.HttpClientResponse)val));
    return ret;
  }

  /**
   * Send the request with a stream <code>body</code>.
   *
   * <p> If the {@link io.vertx.reactivex.core.http.HttpHeaders} is set then the request assumes this is the
   * length of the {stream}, otherwise the request will set a chunked {@link io.vertx.reactivex.core.http.HttpHeaders}.
   * @param body 
   * @return a future notified when the HTTP response is available
   */
  public io.reactivex.Single<io.vertx.reactivex.core.http.HttpClientResponse> rxSend(io.reactivex.Flowable<io.vertx.core.buffer.Buffer> body) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.send(body).onComplete($handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but writes a String in UTF-8 encoding
   * @param chunk the data chunk
   * @return a future completed with the result
   */
  public io.vertx.core.Future<java.lang.Void> end(java.lang.String chunk) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.end(chunk).map(val -> val);
    return ret;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but writes a String in UTF-8 encoding
   * @param chunk the data chunk
   * @return a future completed with the result
   */
  public io.reactivex.Completable rxEnd(java.lang.String chunk) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.end(chunk).onComplete($handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but writes a String with the specified encoding
   * @param chunk the data chunk
   * @param enc the encoding
   * @return a future completed with the result
   */
  public io.vertx.core.Future<java.lang.Void> end(java.lang.String chunk, java.lang.String enc) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.end(chunk, enc).map(val -> val);
    return ret;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but writes a String with the specified encoding
   * @param chunk the data chunk
   * @param enc the encoding
   * @return a future completed with the result
   */
  public io.reactivex.Completable rxEnd(java.lang.String chunk, java.lang.String enc) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.end(chunk, enc).onComplete($handler);
    });
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but writes some data to the request body before ending. If the request is not chunked and
   * no other data has been written then the <code>Content-Length</code> header will be automatically set
   * @param chunk 
   * @return a future completed with the result
   */
  public io.vertx.core.Future<java.lang.Void> end(io.vertx.core.buffer.Buffer chunk) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.end(chunk).map(val -> val);
    return ret;
  }

  /**
   * Same as {@link io.vertx.reactivex.core.http.HttpClientRequest#end} but writes some data to the request body before ending. If the request is not chunked and
   * no other data has been written then the <code>Content-Length</code> header will be automatically set
   * @param chunk 
   * @return a future completed with the result
   */
  public io.reactivex.Completable rxEnd(io.vertx.core.buffer.Buffer chunk) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.end(chunk).onComplete($handler);
    });
  }

  /**
   * Ends the request. If no data has been written to the request body, and {@link io.vertx.reactivex.core.http.HttpClientRequest#sendHead} has not been called then
   * the actual request won't get written until this method gets called.
   * <p>
   * Once the request has ended, it cannot be used any more,
   * @return a future completed with the result
   */
  public io.vertx.core.Future<java.lang.Void> end() { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.end().map(val -> val);
    return ret;
  }

  /**
   * Ends the request. If no data has been written to the request body, and {@link io.vertx.reactivex.core.http.HttpClientRequest#sendHead} has not been called then
   * the actual request won't get written until this method gets called.
   * <p>
   * Once the request has ended, it cannot be used any more,
   * @return a future completed with the result
   */
  public io.reactivex.Completable rxEnd() { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.end().onComplete($handler);
    });
  }

  /**
   * Sets the amount of time after which, if the request does not return any data within the timeout period,
   * the request/response is closed and the related futures are failed with a {@link java.util.concurrent.TimeoutException},
   * e.g. <code>Future<HttpClientResponse></code> or <code>Future<Buffer></code> response body.
   * @param timeout the amount of time in milliseconds.
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest idleTimeout(long timeout) { 
    delegate.idleTimeout(timeout);
    return this;
  }

  /**
   * Set a push handler for this request.<p/>
   *
   * The handler is called when the client receives a <i>push promise</i> from the server. The handler can be called
   * multiple times, for each push promise.<p/>
   *
   * The handler is called with a <i>read-only</i> {@link io.vertx.reactivex.core.http.HttpClientRequest}, the following methods can be called:<p/>
   *
   * <ul>
   *   <li>{@link io.vertx.reactivex.core.http.HttpClientRequest#getMethod}</li>
   *   <li>{@link io.vertx.reactivex.core.http.HttpClientRequest#getURI}</li>
   *   <li>{@link io.vertx.reactivex.core.http.HttpClientRequest#headers}</li>
   * </ul>
   *
   * In addition the handler should call the {@link io.vertx.reactivex.core.http.HttpClientRequest#response} method to set an handler to
   * process the response.<p/>
   * @param handler the handler
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest pushHandler(io.vertx.core.Handler<io.vertx.reactivex.core.http.HttpClientRequest> handler) { 
    delegate.pushHandler(io.vertx.lang.reactivex.Helper.convertHandler(handler, event -> io.vertx.reactivex.core.http.HttpClientRequest.newInstance((io.vertx.core.http.HttpClientRequest)event)));
    return this;
  }

  /**
   * Reset this stream with the error code <code>0</code>.
   * @return 
   */
  public io.vertx.core.Future<java.lang.Void> reset() { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.reset().map(val -> val);
    return ret;
  }

  /**
   * Reset this stream with the error code <code>0</code>.
   * @return 
   */
  public io.reactivex.Completable rxReset() { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.reset().onComplete($handler);
    });
  }

  /**
   * Reset this request:
   * <p/>
   * <ul>
   *   <li>for HTTP/2, this performs send an HTTP/2 reset frame with the specified error <code>code</code></li>
   *   <li>for HTTP/1.x, this closes the connection when the current request is inflight</li>
   * </ul>
   * <p/>
   * When the request has not yet been sent, the request will be aborted and false is returned as indicator.
   * <p/>
   * @param code the error code
   * @return <code>true</code> when reset has been performed
   */
  public io.vertx.core.Future<java.lang.Void> reset(long code) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.reset(code).map(val -> val);
    return ret;
  }

  /**
   * Reset this request:
   * <p/>
   * <ul>
   *   <li>for HTTP/2, this performs send an HTTP/2 reset frame with the specified error <code>code</code></li>
   *   <li>for HTTP/1.x, this closes the connection when the current request is inflight</li>
   * </ul>
   * <p/>
   * When the request has not yet been sent, the request will be aborted and false is returned as indicator.
   * <p/>
   * @param code the error code
   * @return <code>true</code> when reset has been performed
   */
  public io.reactivex.Completable rxReset(long code) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.reset(code).onComplete($handler);
    });
  }

  /**
   * Reset this request:
   * <p/>
   * <ul>
   *   <li>for HTTP/2, send an HTTP/2 reset frame with the specified error <code>code</code></li>
   *   <li>for HTTP/1.x, close the connection when the current request is inflight</li>
   * </ul>
   * <p/>
   * When the request has not yet been sent, the request will be aborted and <code>false</code> is returned as indicator.
   * <p/>
   * @param code the error code
   * @param cause an optional cause that can be attached to the error code
   * @return true when reset has been performed
   */
  public io.vertx.core.Future<java.lang.Void> reset(long code, java.lang.Throwable cause) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.reset(code, cause).map(val -> val);
    return ret;
  }

  /**
   * Reset this request:
   * <p/>
   * <ul>
   *   <li>for HTTP/2, send an HTTP/2 reset frame with the specified error <code>code</code></li>
   *   <li>for HTTP/1.x, close the connection when the current request is inflight</li>
   * </ul>
   * <p/>
   * When the request has not yet been sent, the request will be aborted and <code>false</code> is returned as indicator.
   * <p/>
   * @param code the error code
   * @param cause an optional cause that can be attached to the error code
   * @return true when reset has been performed
   */
  public io.reactivex.Completable rxReset(long code, java.lang.Throwable cause) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.reset(code, cause).onComplete($handler);
    });
  }

  /**
   * @return the {@link io.vertx.reactivex.core.http.HttpConnection} associated with this request
   */
  public io.vertx.reactivex.core.http.HttpConnection connection() { 
    if (cached_1 != null) {
      return cached_1;
    }
    io.vertx.reactivex.core.http.HttpConnection ret = io.vertx.reactivex.core.http.HttpConnection.newInstance((io.vertx.core.http.HttpConnection)delegate.connection());
    cached_1 = ret;
    return ret;
  }

  /**
   * Write an HTTP/2 frame to the request, allowing to extend the HTTP/2 protocol.<p>
   *
   * The frame is sent immediatly and is not subject to flow control.<p>
   *
   * This method must be called after the request headers have been sent and only for the protocol HTTP/2.
   * The {@link io.vertx.reactivex.core.http.HttpClientRequest#sendHead} should be used for this purpose.
   * @param type the 8-bit frame type
   * @param flags the 8-bit frame flags
   * @param payload the frame payload
   * @return a reference to this, so the API can be used fluently
   */
  public io.vertx.core.Future<java.lang.Void> writeCustomFrame(int type, int flags, io.vertx.core.buffer.Buffer payload) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.writeCustomFrame(type, flags, payload).map(val -> val);
    return ret;
  }

  /**
   * Write an HTTP/2 frame to the request, allowing to extend the HTTP/2 protocol.<p>
   *
   * The frame is sent immediatly and is not subject to flow control.<p>
   *
   * This method must be called after the request headers have been sent and only for the protocol HTTP/2.
   * The {@link io.vertx.reactivex.core.http.HttpClientRequest#sendHead} should be used for this purpose.
   * @param type the 8-bit frame type
   * @param flags the 8-bit frame flags
   * @param payload the frame payload
   * @return a reference to this, so the API can be used fluently
   */
  public io.reactivex.Completable rxWriteCustomFrame(int type, int flags, io.vertx.core.buffer.Buffer payload) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.writeCustomFrame(type, flags, payload).onComplete($handler);
    });
  }

  /**
   * @return the id of the stream of this response,  when it is not yet determined, i.e the request has not been yet sent or it is not supported HTTP/1.x
   */
  public int streamId() { 
    int ret = delegate.streamId();
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClientRequest#writeCustomFrame} but with an {@link io.vertx.core.http.HttpFrame}.
   * @param frame the frame to write
   * @return 
   */
  public io.vertx.core.Future<java.lang.Void> writeCustomFrame(io.vertx.core.http.HttpFrame frame) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.writeCustomFrame(frame).map(val -> val);
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClientRequest#writeCustomFrame} but with an {@link io.vertx.core.http.HttpFrame}.
   * @param frame the frame to write
   * @return 
   */
  public io.reactivex.Completable rxWriteCustomFrame(io.vertx.core.http.HttpFrame frame) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.writeCustomFrame(frame).onComplete($handler);
    });
  }

  /**
   * Sets the priority of the associated stream.
   * <p/>
   * This is not implemented for HTTP/1.x.
   * @param streamPriority the priority of this request's stream
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest setStreamPriority(io.vertx.core.http.StreamPriority streamPriority) { 
    delegate.setStreamPriority(streamPriority);
    return this;
  }

  /**
   * @return the priority of the associated HTTP/2 stream for HTTP/2 otherwise <code>null</code>
   */
  public io.vertx.core.http.StreamPriority getStreamPriority() { 
    io.vertx.core.http.StreamPriority ret = delegate.getStreamPriority();
    return ret;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClientRequest#putHeader} but using CharSequence
   * @param name 
   * @param value 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest putHeader(java.lang.CharSequence name, java.lang.CharSequence value) { 
    delegate.putHeader(name, value);
    return this;
  }

  /**
   * Put an HTTP header with multiple values
   * @param name The header name
   * @param values The header values
   * @return @return a reference to this, so the API can be used fluently
   */
  public io.vertx.reactivex.core.http.HttpClientRequest putHeader(java.lang.String name, java.lang.Iterable<java.lang.String> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  /**
   * Like {@link io.vertx.reactivex.core.http.HttpClientRequest#putHeader} but using CharSequence
   * @param name 
   * @param values 
   * @return 
   */
  public io.vertx.reactivex.core.http.HttpClientRequest putHeader(java.lang.CharSequence name, java.lang.Iterable<java.lang.CharSequence> values) { 
    delegate.putHeader(name, values);
    return this;
  }

  private io.vertx.core.MultiMap cached_0;
  private io.vertx.reactivex.core.http.HttpConnection cached_1;
  public static HttpClientRequest newInstance(io.vertx.core.http.HttpClientRequest arg) {
    return arg != null ? new HttpClientRequest(arg) : null;
  }

}
