/*
 * Copyright 2014 Red Hat, Inc.
 *
 * Red Hat licenses this file to you under the Apache License, version 2.0
 * (the "License"); you may not use this file except in compliance with the
 * License.  You may obtain a copy of the License at:
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.  See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */

package io.vertx.reactivex.core.file;

import io.vertx.reactivex.RxHelper;
import io.vertx.reactivex.ObservableHelper;
import io.vertx.reactivex.FlowableHelper;
import io.vertx.reactivex.impl.AsyncResultMaybe;
import io.vertx.reactivex.impl.AsyncResultSingle;
import io.vertx.reactivex.impl.AsyncResultCompletable;
import io.vertx.reactivex.impl.AsyncResultFlowable;
import io.vertx.reactivex.WriteStreamObserver;
import io.vertx.reactivex.WriteStreamSubscriber;
import java.util.Map;
import java.util.Set;
import java.util.List;
import java.util.Iterator;
import java.util.function.Function;
import java.util.function.Supplier;
import java.util.stream.Collectors;
import io.vertx.core.Handler;
import io.vertx.core.AsyncResult;
import io.vertx.core.json.JsonObject;
import io.vertx.core.json.JsonArray;
import io.vertx.lang.rx.RxDelegate;
import io.vertx.lang.rx.RxGen;
import io.vertx.lang.rx.TypeArg;
import io.vertx.lang.rx.MappingIterator;

/**
 * Contains a broad set of operations for manipulating files on the file system.
 * <p>
 * A (potential) blocking and non blocking version of each operation is provided.
 * <p>
 * The non blocking versions take a handler which is called when the operation completes or an error occurs.
 * <p>
 * The blocking versions are named <code>xxxBlocking</code> and return the results, or throw exceptions directly.
 * In many cases, depending on the operating system and file system some of the potentially blocking operations
 * can return quickly, which is why we provide them, but it's highly recommended that you test how long they take to
 * return in your particular application before using them on an event loop.
 * <p>
 * Please consult the documentation for more information on file system support.
 *
 * <p>
 * NOTE: This class has been automatically generated from the {@link io.vertx.core.file.FileSystem original} non RX-ified interface using Vert.x codegen.
 */

@RxGen(io.vertx.core.file.FileSystem.class)
public class FileSystem implements RxDelegate {

  @Override
  public String toString() {
    return delegate.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) return true;
    if (o == null || getClass() != o.getClass()) return false;
    FileSystem that = (FileSystem) o;
    return delegate.equals(that.delegate);
  }
  
  @Override
  public int hashCode() {
    return delegate.hashCode();
  }

  public static final TypeArg<FileSystem> __TYPE_ARG = new TypeArg<>(    obj -> new FileSystem((io.vertx.core.file.FileSystem) obj),
    FileSystem::getDelegate
  );

  private final io.vertx.core.file.FileSystem delegate;
  
  public FileSystem(io.vertx.core.file.FileSystem delegate) {
    this.delegate = delegate;
  }

  public FileSystem(Object delegate) {
    this.delegate = (io.vertx.core.file.FileSystem)delegate;
  }

  @Override 
  public io.vertx.core.file.FileSystem getDelegate() {
    return delegate;
  }

  private static final TypeArg<io.vertx.reactivex.core.file.AsyncFile> TYPE_ARG_0 = new TypeArg<io.vertx.reactivex.core.file.AsyncFile>(o1 -> io.vertx.reactivex.core.file.AsyncFile.newInstance((io.vertx.core.file.AsyncFile)o1), o1 -> o1.getDelegate());

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * The copy will fail if the destination already exists.
   * @param from the path to copy from
   * @param to the path to copy to
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> copy(java.lang.String from, java.lang.String to) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.copy(from, to).map(val -> val);
    return ret;
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * The copy will fail if the destination already exists.
   * @param from the path to copy from
   * @param to the path to copy to
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxCopy(java.lang.String from, java.lang.String to) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.copy(from, to).onComplete($handler);
    });
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> copy(java.lang.String from, java.lang.String to, io.vertx.core.file.CopyOptions options) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.copy(from, to, options).map(val -> val);
    return ret;
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxCopy(java.lang.String from, java.lang.String to, io.vertx.core.file.CopyOptions options) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.copy(from, to, options).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#copy}
   * @param from 
   * @param to 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem copyBlocking(java.lang.String from, java.lang.String to) { 
    delegate.copyBlocking(from, to);
    return this;
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * If <code>recursive</code> is <code>true</code> and <code>from</code> represents a directory, then the directory and its contents
   * will be copied recursively to the destination <code>to</code>.
   * <p>
   * The copy will fail if the destination if the destination already exists.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param recursive 
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> copyRecursive(java.lang.String from, java.lang.String to, boolean recursive) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.copyRecursive(from, to, recursive).map(val -> val);
    return ret;
  }

  /**
   * Copy a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * If <code>recursive</code> is <code>true</code> and <code>from</code> represents a directory, then the directory and its contents
   * will be copied recursively to the destination <code>to</code>.
   * <p>
   * The copy will fail if the destination if the destination already exists.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param recursive 
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxCopyRecursive(java.lang.String from, java.lang.String to, boolean recursive) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.copyRecursive(from, to, recursive).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#copyRecursive}
   * @param from 
   * @param to 
   * @param recursive 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem copyRecursiveBlocking(java.lang.String from, java.lang.String to, boolean recursive) { 
    delegate.copyRecursiveBlocking(from, to, recursive);
    return this;
  }

  /**
   * Move a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * The move will fail if the destination already exists.
   * @param from the path to copy from
   * @param to the path to copy to
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> move(java.lang.String from, java.lang.String to) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.move(from, to).map(val -> val);
    return ret;
  }

  /**
   * Move a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * <p>
   * The move will fail if the destination already exists.
   * @param from the path to copy from
   * @param to the path to copy to
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxMove(java.lang.String from, java.lang.String to) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.move(from, to).onComplete($handler);
    });
  }

  /**
   * Move a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> move(java.lang.String from, java.lang.String to, io.vertx.core.file.CopyOptions options) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.move(from, to, options).map(val -> val);
    return ret;
  }

  /**
   * Move a file from the path <code>from</code> to path <code>to</code>, asynchronously.
   * @param from the path to copy from
   * @param to the path to copy to
   * @param options options describing how the file should be copied
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxMove(java.lang.String from, java.lang.String to, io.vertx.core.file.CopyOptions options) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.move(from, to, options).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#move}
   * @param from 
   * @param to 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem moveBlocking(java.lang.String from, java.lang.String to) { 
    delegate.moveBlocking(from, to);
    return this;
  }

  /**
   * Truncate the file represented by <code>path</code> to length <code>len</code> in bytes, asynchronously.
   * <p>
   * The operation will fail if the file does not exist or <code>len</code> is less than <code>zero</code>.
   * @param path the path to the file
   * @param len the length to truncate it to
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> truncate(java.lang.String path, long len) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.truncate(path, len).map(val -> val);
    return ret;
  }

  /**
   * Truncate the file represented by <code>path</code> to length <code>len</code> in bytes, asynchronously.
   * <p>
   * The operation will fail if the file does not exist or <code>len</code> is less than <code>zero</code>.
   * @param path the path to the file
   * @param len the length to truncate it to
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxTruncate(java.lang.String path, long len) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.truncate(path, len).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#truncate}
   * @param path 
   * @param len 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem truncateBlocking(java.lang.String path, long len) { 
    delegate.truncateBlocking(path, len);
    return this;
  }

  /**
   * Change the permissions on the file represented by <code>path</code> to <code>perms</code>, asynchronously.
   * <p>
   * The permission String takes the form rwxr-x--- as
   * specified <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * @param path the path to the file
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> chmod(java.lang.String path, java.lang.String perms) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.chmod(path, perms).map(val -> val);
    return ret;
  }

  /**
   * Change the permissions on the file represented by <code>path</code> to <code>perms</code>, asynchronously.
   * <p>
   * The permission String takes the form rwxr-x--- as
   * specified <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * @param path the path to the file
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxChmod(java.lang.String path, java.lang.String perms) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.chmod(path, perms).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem #chmod(String, String)}
   * @param path 
   * @param perms 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem chmodBlocking(java.lang.String path, java.lang.String perms) { 
    delegate.chmodBlocking(path, perms);
    return this;
  }

  /**
   * Change the permissions on the file represented by <code>path</code> to <code>perms</code>, asynchronously.<p>
   * The permission String takes the form rwxr-x--- as
   * specified in {<a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>}.
   * <p>
   * If the file is directory then all contents will also have their permissions changed recursively. Any directory permissions will
   * be set to <code>dirPerms</code>, whilst any normal file permissions will be set to <code>perms</code>.
   * @param path the path to the file
   * @param perms the permissions string
   * @param dirPerms the directory permissions
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> chmodRecursive(java.lang.String path, java.lang.String perms, java.lang.String dirPerms) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.chmodRecursive(path, perms, dirPerms).map(val -> val);
    return ret;
  }

  /**
   * Change the permissions on the file represented by <code>path</code> to <code>perms</code>, asynchronously.<p>
   * The permission String takes the form rwxr-x--- as
   * specified in {<a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>}.
   * <p>
   * If the file is directory then all contents will also have their permissions changed recursively. Any directory permissions will
   * be set to <code>dirPerms</code>, whilst any normal file permissions will be set to <code>perms</code>.
   * @param path the path to the file
   * @param perms the permissions string
   * @param dirPerms the directory permissions
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxChmodRecursive(java.lang.String path, java.lang.String perms, java.lang.String dirPerms) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.chmodRecursive(path, perms, dirPerms).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#chmodRecursive}
   * @param path 
   * @param perms 
   * @param dirPerms 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem chmodRecursiveBlocking(java.lang.String path, java.lang.String perms, java.lang.String dirPerms) { 
    delegate.chmodRecursiveBlocking(path, perms, dirPerms);
    return this;
  }

  /**
   * Change the ownership on the file represented by <code>path</code> to <code>user</code> and {code group}, asynchronously.
   * @param path the path to the file
   * @param user the user name, <code>null</code> will not change the user name
   * @param group the user group, <code>null</code> will not change the user group name
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> chown(java.lang.String path, java.lang.String user, java.lang.String group) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.chown(path, user, group).map(val -> val);
    return ret;
  }

  /**
   * Change the ownership on the file represented by <code>path</code> to <code>user</code> and {code group}, asynchronously.
   * @param path the path to the file
   * @param user the user name, <code>null</code> will not change the user name
   * @param group the user group, <code>null</code> will not change the user group name
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxChown(java.lang.String path, java.lang.String user, java.lang.String group) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.chown(path, user, group).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#chown}
   *
   * @param path 
   * @param user 
   * @param group 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem chownBlocking(java.lang.String path, java.lang.String user, java.lang.String group) { 
    delegate.chownBlocking(path, user, group);
    return this;
  }

  /**
   * Obtain properties for the file represented by <code>path</code>, asynchronously.
   * <p>
   * If the file is a link, the link will be followed.
   * @param path the path to the file
   * @return a future notified on completion
   */
  public io.vertx.core.Future<io.vertx.core.file.FileProps> props(java.lang.String path) { 
    io.vertx.core.Future<io.vertx.core.file.FileProps> ret = delegate.props(path).map(val -> val);
    return ret;
  }

  /**
   * Obtain properties for the file represented by <code>path</code>, asynchronously.
   * <p>
   * If the file is a link, the link will be followed.
   * @param path the path to the file
   * @return a future notified on completion
   */
  public io.reactivex.Single<io.vertx.core.file.FileProps> rxProps(java.lang.String path) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.props(path).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#props}
   * @param path 
   * @return 
   */
  public io.vertx.core.file.FileProps propsBlocking(java.lang.String path) { 
    io.vertx.core.file.FileProps ret = delegate.propsBlocking(path);
    return ret;
  }

  /**
   * Obtain properties for the link represented by <code>path</code>, asynchronously.
   * <p>
   * The link will not be followed.
   * @param path the path to the file
   * @return a future notified on completion
   */
  public io.vertx.core.Future<io.vertx.core.file.FileProps> lprops(java.lang.String path) { 
    io.vertx.core.Future<io.vertx.core.file.FileProps> ret = delegate.lprops(path).map(val -> val);
    return ret;
  }

  /**
   * Obtain properties for the link represented by <code>path</code>, asynchronously.
   * <p>
   * The link will not be followed.
   * @param path the path to the file
   * @return a future notified on completion
   */
  public io.reactivex.Single<io.vertx.core.file.FileProps> rxLprops(java.lang.String path) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.lprops(path).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#lprops}
   * @param path 
   * @return 
   */
  public io.vertx.core.file.FileProps lpropsBlocking(java.lang.String path) { 
    io.vertx.core.file.FileProps ret = delegate.lpropsBlocking(path);
    return ret;
  }

  /**
   * Create a hard link on the file system from <code>link</code> to <code>existing</code>, asynchronously.
   * @param link the link
   * @param existing the link destination
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> link(java.lang.String link, java.lang.String existing) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.link(link, existing).map(val -> val);
    return ret;
  }

  /**
   * Create a hard link on the file system from <code>link</code> to <code>existing</code>, asynchronously.
   * @param link the link
   * @param existing the link destination
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxLink(java.lang.String link, java.lang.String existing) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.link(link, existing).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#link}
   * @param link 
   * @param existing 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem linkBlocking(java.lang.String link, java.lang.String existing) { 
    delegate.linkBlocking(link, existing);
    return this;
  }

  /**
   * Create a symbolic link on the file system from <code>link</code> to <code>existing</code>, asynchronously.
   * @param link the link
   * @param existing the link destination
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> symlink(java.lang.String link, java.lang.String existing) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.symlink(link, existing).map(val -> val);
    return ret;
  }

  /**
   * Create a symbolic link on the file system from <code>link</code> to <code>existing</code>, asynchronously.
   * @param link the link
   * @param existing the link destination
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxSymlink(java.lang.String link, java.lang.String existing) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.symlink(link, existing).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#link}
   * @param link 
   * @param existing 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem symlinkBlocking(java.lang.String link, java.lang.String existing) { 
    delegate.symlinkBlocking(link, existing);
    return this;
  }

  /**
   * Unlinks the link on the file system represented by the path <code>link</code>, asynchronously.
   * @param link the link
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> unlink(java.lang.String link) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.unlink(link).map(val -> val);
    return ret;
  }

  /**
   * Unlinks the link on the file system represented by the path <code>link</code>, asynchronously.
   * @param link the link
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxUnlink(java.lang.String link) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.unlink(link).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#unlink}
   * @param link 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem unlinkBlocking(java.lang.String link) { 
    delegate.unlinkBlocking(link);
    return this;
  }

  /**
   * Returns the path representing the file that the symbolic link specified by <code>link</code> points to, asynchronously.
   * @param link the link
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.String> readSymlink(java.lang.String link) { 
    io.vertx.core.Future<java.lang.String> ret = delegate.readSymlink(link).map(val -> val);
    return ret;
  }

  /**
   * Returns the path representing the file that the symbolic link specified by <code>link</code> points to, asynchronously.
   * @param link the link
   * @return a future notified on completion
   */
  public io.reactivex.Single<java.lang.String> rxReadSymlink(java.lang.String link) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.readSymlink(link).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#readSymlink}
   * @param link 
   * @return 
   */
  public java.lang.String readSymlinkBlocking(java.lang.String link) { 
    java.lang.String ret = delegate.readSymlinkBlocking(link);
    return ret;
  }

  /**
   * Deletes the file represented by the specified <code>path</code>, asynchronously.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> delete(java.lang.String path) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.delete(path).map(val -> val);
    return ret;
  }

  /**
   * Deletes the file represented by the specified <code>path</code>, asynchronously.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxDelete(java.lang.String path) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.delete(path).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#delete}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem deleteBlocking(java.lang.String path) { 
    delegate.deleteBlocking(path);
    return this;
  }

  /**
   * Deletes the file represented by the specified <code>path</code>, asynchronously.
   * <p>
   * If the path represents a directory and <code>recursive = true</code> then the directory and its contents will be
   * deleted recursively.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> deleteRecursive(java.lang.String path) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.deleteRecursive(path).map(val -> val);
    return ret;
  }

  /**
   * Deletes the file represented by the specified <code>path</code>, asynchronously.
   * <p>
   * If the path represents a directory and <code>recursive = true</code> then the directory and its contents will be
   * deleted recursively.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxDeleteRecursive(java.lang.String path) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.deleteRecursive(path).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#deleteRecursive}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem deleteRecursiveBlocking(java.lang.String path) { 
    delegate.deleteRecursiveBlocking(path);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code>, asynchronously.
   * <p>
   * The operation will fail if the directory already exists.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> mkdir(java.lang.String path) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.mkdir(path).map(val -> val);
    return ret;
  }

  /**
   * Create the directory represented by <code>path</code>, asynchronously.
   * <p>
   * The operation will fail if the directory already exists.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxMkdir(java.lang.String path) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.mkdir(path).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#mkdir}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem mkdirBlocking(java.lang.String path) { 
    delegate.mkdirBlocking(path);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code>, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * <p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * <p>
   * The operation will fail if the directory already exists.
   * @param path path to the file
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> mkdir(java.lang.String path, java.lang.String perms) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.mkdir(path, perms).map(val -> val);
    return ret;
  }

  /**
   * Create the directory represented by <code>path</code>, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * <p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * <p>
   * The operation will fail if the directory already exists.
   * @param path path to the file
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxMkdir(java.lang.String path, java.lang.String perms) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.mkdir(path, perms).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#mkdir}
   * @param path 
   * @param perms 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem mkdirBlocking(java.lang.String path, java.lang.String perms) { 
    delegate.mkdirBlocking(path, perms);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code> and any non existent parents, asynchronously.
   * <p>
   * The operation will fail if the <code>path</code> already exists but is not a directory.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> mkdirs(java.lang.String path) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.mkdirs(path).map(val -> val);
    return ret;
  }

  /**
   * Create the directory represented by <code>path</code> and any non existent parents, asynchronously.
   * <p>
   * The operation will fail if the <code>path</code> already exists but is not a directory.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxMkdirs(java.lang.String path) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.mkdirs(path).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#mkdirs}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem mkdirsBlocking(java.lang.String path) { 
    delegate.mkdirsBlocking(path);
    return this;
  }

  /**
   * Create the directory represented by <code>path</code> and any non existent parents, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * <p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * <p>
   * The operation will fail if the <code>path</code> already exists but is not a directory.
   * @param path path to the file
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> mkdirs(java.lang.String path, java.lang.String perms) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.mkdirs(path, perms).map(val -> val);
    return ret;
  }

  /**
   * Create the directory represented by <code>path</code> and any non existent parents, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * <p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   * <p>
   * The operation will fail if the <code>path</code> already exists but is not a directory.
   * @param path path to the file
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxMkdirs(java.lang.String path, java.lang.String perms) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.mkdirs(path, perms).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#mkdirs}
   * @param path 
   * @param perms 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem mkdirsBlocking(java.lang.String path, java.lang.String perms) { 
    delegate.mkdirsBlocking(path, perms);
    return this;
  }

  /**
   * Read the contents of the directory specified by <code>path</code>, asynchronously.
   * <p>
   * The result is an array of String representing the paths of the files inside the directory.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.util.List<java.lang.String>> readDir(java.lang.String path) { 
    io.vertx.core.Future<java.util.List<java.lang.String>> ret = delegate.readDir(path).map(val -> val);
    return ret;
  }

  /**
   * Read the contents of the directory specified by <code>path</code>, asynchronously.
   * <p>
   * The result is an array of String representing the paths of the files inside the directory.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.reactivex.Single<java.util.List<java.lang.String>> rxReadDir(java.lang.String path) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.readDir(path).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#readDir}
   * @param path 
   * @return 
   */
  public java.util.List<java.lang.String> readDirBlocking(java.lang.String path) { 
    java.util.List<java.lang.String> ret = delegate.readDirBlocking(path);
    return ret;
  }

  /**
   * Read the contents of the directory specified by <code>path</code>, asynchronously.
   * <p>
   * The parameter <code>filter</code> is a regular expression. If <code>filter</code> is specified then only the paths that
   * match  @{filter}will be returned.
   * <p>
   * The result is an array of String representing the paths of the files inside the directory.
   * @param path path to the directory
   * @param filter the filter expression
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.util.List<java.lang.String>> readDir(java.lang.String path, java.lang.String filter) { 
    io.vertx.core.Future<java.util.List<java.lang.String>> ret = delegate.readDir(path, filter).map(val -> val);
    return ret;
  }

  /**
   * Read the contents of the directory specified by <code>path</code>, asynchronously.
   * <p>
   * The parameter <code>filter</code> is a regular expression. If <code>filter</code> is specified then only the paths that
   * match  @{filter}will be returned.
   * <p>
   * The result is an array of String representing the paths of the files inside the directory.
   * @param path path to the directory
   * @param filter the filter expression
   * @return a future notified on completion
   */
  public io.reactivex.Single<java.util.List<java.lang.String>> rxReadDir(java.lang.String path, java.lang.String filter) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.readDir(path, filter).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#readDir}
   * @param path 
   * @param filter 
   * @return 
   */
  public java.util.List<java.lang.String> readDirBlocking(java.lang.String path, java.lang.String filter) { 
    java.util.List<java.lang.String> ret = delegate.readDirBlocking(path, filter);
    return ret;
  }

  /**
   * Reads the entire file as represented by the path <code>path</code> as a , asynchronously.
   * <p>
   * Do not use this method to read very large files or you risk running out of available RAM.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.vertx.core.Future<io.vertx.core.buffer.Buffer> readFile(java.lang.String path) { 
    io.vertx.core.Future<io.vertx.core.buffer.Buffer> ret = delegate.readFile(path).map(val -> val);
    return ret;
  }

  /**
   * Reads the entire file as represented by the path <code>path</code> as a , asynchronously.
   * <p>
   * Do not use this method to read very large files or you risk running out of available RAM.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.reactivex.Single<io.vertx.core.buffer.Buffer> rxReadFile(java.lang.String path) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.readFile(path).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#readFile}
   * @param path 
   * @return 
   */
  public io.vertx.core.buffer.Buffer readFileBlocking(java.lang.String path) { 
    io.vertx.core.buffer.Buffer ret = delegate.readFileBlocking(path);
    return ret;
  }

  /**
   * Creates the file, and writes the specified <code>Buffer data</code> to the file represented by the path <code>path</code>,
   * asynchronously.
   * @param path path to the file
   * @param data 
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> writeFile(java.lang.String path, io.vertx.core.buffer.Buffer data) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.writeFile(path, data).map(val -> val);
    return ret;
  }

  /**
   * Creates the file, and writes the specified <code>Buffer data</code> to the file represented by the path <code>path</code>,
   * asynchronously.
   * @param path path to the file
   * @param data 
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxWriteFile(java.lang.String path, io.vertx.core.buffer.Buffer data) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.writeFile(path, data).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#writeFile}
   * @param path 
   * @param data 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem writeFileBlocking(java.lang.String path, io.vertx.core.buffer.Buffer data) { 
    delegate.writeFileBlocking(path, data);
    return this;
  }

  /**
   * Open the file represented by <code>path</code>, asynchronously.
   * <p>
   * The file is opened for both reading and writing. If the file does not already exist it will be created.
   * @param path path to the file
   * @param options options describing how the file should be opened
   * @return a future notified on completion
   */
  public io.vertx.core.Future<io.vertx.reactivex.core.file.AsyncFile> open(java.lang.String path, io.vertx.core.file.OpenOptions options) { 
    io.vertx.core.Future<io.vertx.reactivex.core.file.AsyncFile> ret = delegate.open(path, options).map(val -> io.vertx.reactivex.core.file.AsyncFile.newInstance((io.vertx.core.file.AsyncFile)val));
    return ret;
  }

  /**
   * Open the file represented by <code>path</code>, asynchronously.
   * <p>
   * The file is opened for both reading and writing. If the file does not already exist it will be created.
   * @param path path to the file
   * @param options options describing how the file should be opened
   * @return a future notified on completion
   */
  public io.reactivex.Single<io.vertx.reactivex.core.file.AsyncFile> rxOpen(java.lang.String path, io.vertx.core.file.OpenOptions options) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.open(path, options).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#open}
   * @param path 
   * @param options 
   * @return 
   */
  public io.vertx.reactivex.core.file.AsyncFile openBlocking(java.lang.String path, io.vertx.core.file.OpenOptions options) { 
    io.vertx.reactivex.core.file.AsyncFile ret = io.vertx.reactivex.core.file.AsyncFile.newInstance((io.vertx.core.file.AsyncFile)delegate.openBlocking(path, options));
    return ret;
  }

  /**
   * Creates an empty file with the specified <code>path</code>, asynchronously.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> createFile(java.lang.String path) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.createFile(path).map(val -> val);
    return ret;
  }

  /**
   * Creates an empty file with the specified <code>path</code>, asynchronously.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxCreateFile(java.lang.String path) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.createFile(path).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createFile}
   * @param path 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem createFileBlocking(java.lang.String path) { 
    delegate.createFileBlocking(path);
    return this;
  }

  /**
   * Creates an empty file with the specified <code>path</code> and permissions <code>perms</code>, asynchronously.
   * @param path path to the file
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Void> createFile(java.lang.String path, java.lang.String perms) { 
    io.vertx.core.Future<java.lang.Void> ret = delegate.createFile(path, perms).map(val -> val);
    return ret;
  }

  /**
   * Creates an empty file with the specified <code>path</code> and permissions <code>perms</code>, asynchronously.
   * @param path path to the file
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.reactivex.Completable rxCreateFile(java.lang.String path, java.lang.String perms) { 
    return AsyncResultCompletable.toCompletable($handler -> {
      this.createFile(path, perms).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createFile}
   * @param path 
   * @param perms 
   * @return 
   */
  public io.vertx.reactivex.core.file.FileSystem createFileBlocking(java.lang.String path, java.lang.String perms) { 
    delegate.createFileBlocking(path, perms);
    return this;
  }

  /**
   * Determines whether the file as specified by the path <code>path</code> exists, asynchronously.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.Boolean> exists(java.lang.String path) { 
    io.vertx.core.Future<java.lang.Boolean> ret = delegate.exists(path).map(val -> val);
    return ret;
  }

  /**
   * Determines whether the file as specified by the path <code>path</code> exists, asynchronously.
   * @param path path to the file
   * @return a future notified on completion
   */
  public io.reactivex.Single<java.lang.Boolean> rxExists(java.lang.String path) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.exists(path).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#exists}
   * @param path 
   * @return 
   */
  public boolean existsBlocking(java.lang.String path) { 
    boolean ret = delegate.existsBlocking(path);
    return ret;
  }

  /**
   * Returns properties of the file-system being used by the specified <code>path</code>, asynchronously.
   * @param path path to anywhere on the filesystem
   * @return a future notified on completion
   */
  public io.vertx.core.Future<io.vertx.core.file.FileSystemProps> fsProps(java.lang.String path) { 
    io.vertx.core.Future<io.vertx.core.file.FileSystemProps> ret = delegate.fsProps(path).map(val -> val);
    return ret;
  }

  /**
   * Returns properties of the file-system being used by the specified <code>path</code>, asynchronously.
   * @param path path to anywhere on the filesystem
   * @return a future notified on completion
   */
  public io.reactivex.Single<io.vertx.core.file.FileSystemProps> rxFsProps(java.lang.String path) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.fsProps(path).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#fsProps}
   * @param path 
   * @return 
   */
  public io.vertx.core.file.FileSystemProps fsPropsBlocking(java.lang.String path) { 
    io.vertx.core.file.FileSystemProps ret = delegate.fsPropsBlocking(path);
    return ret;
  }

  /**
   * Creates a new directory in the default temporary-file directory, using the given
   * prefix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.String> createTempDirectory(java.lang.String prefix) { 
    io.vertx.core.Future<java.lang.String> ret = delegate.createTempDirectory(prefix).map(val -> val);
    return ret;
  }

  /**
   * Creates a new directory in the default temporary-file directory, using the given
   * prefix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @return a future notified on completion
   */
  public io.reactivex.Single<java.lang.String> rxCreateTempDirectory(java.lang.String prefix) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.createTempDirectory(prefix).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createTempDirectory}
   * @param prefix 
   * @return 
   */
  public java.lang.String createTempDirectoryBlocking(java.lang.String prefix) { 
    java.lang.String ret = delegate.createTempDirectoryBlocking(prefix);
    return ret;
  }

  /**
   * Creates a new directory in the default temporary-file directory, using the given
   * prefix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.String> createTempDirectory(java.lang.String prefix, java.lang.String perms) { 
    io.vertx.core.Future<java.lang.String> ret = delegate.createTempDirectory(prefix, perms).map(val -> val);
    return ret;
  }

  /**
   * Creates a new directory in the default temporary-file directory, using the given
   * prefix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.reactivex.Single<java.lang.String> rxCreateTempDirectory(java.lang.String prefix, java.lang.String perms) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.createTempDirectory(prefix, perms).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createTempDirectory}
   * @param prefix 
   * @param perms 
   * @return 
   */
  public java.lang.String createTempDirectoryBlocking(java.lang.String prefix, java.lang.String perms) { 
    java.lang.String ret = delegate.createTempDirectoryBlocking(prefix, perms);
    return ret;
  }

  /**
   * Creates a new directory in the directory provided by the path <code>path</code>, using the given
   * prefix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.String> createTempDirectory(java.lang.String dir, java.lang.String prefix, java.lang.String perms) { 
    io.vertx.core.Future<java.lang.String> ret = delegate.createTempDirectory(dir, prefix, perms).map(val -> val);
    return ret;
  }

  /**
   * Creates a new directory in the directory provided by the path <code>path</code>, using the given
   * prefix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.reactivex.Single<java.lang.String> rxCreateTempDirectory(java.lang.String dir, java.lang.String prefix, java.lang.String perms) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.createTempDirectory(dir, prefix, perms).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createTempDirectory}
   * @param dir 
   * @param prefix 
   * @param perms 
   * @return 
   */
  public java.lang.String createTempDirectoryBlocking(java.lang.String dir, java.lang.String prefix, java.lang.String perms) { 
    java.lang.String ret = delegate.createTempDirectoryBlocking(dir, prefix, perms);
    return ret;
  }

  /**
   * Creates a new file in the default temporary-file directory, using the given
   * prefix and suffix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.String> createTempFile(java.lang.String prefix, java.lang.String suffix) { 
    io.vertx.core.Future<java.lang.String> ret = delegate.createTempFile(prefix, suffix).map(val -> val);
    return ret;
  }

  /**
   * Creates a new file in the default temporary-file directory, using the given
   * prefix and suffix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @return a future notified on completion
   */
  public io.reactivex.Single<java.lang.String> rxCreateTempFile(java.lang.String prefix, java.lang.String suffix) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.createTempFile(prefix, suffix).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createTempFile}
   * @param prefix 
   * @param suffix 
   * @return 
   */
  public java.lang.String createTempFileBlocking(java.lang.String prefix, java.lang.String suffix) { 
    java.lang.String ret = delegate.createTempFileBlocking(prefix, suffix);
    return ret;
  }

  /**
   * Creates a new file in the directory provided by the path <code>dir</code>, using the given
   * prefix and suffix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms 
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.String> createTempFile(java.lang.String prefix, java.lang.String suffix, java.lang.String perms) { 
    io.vertx.core.Future<java.lang.String> ret = delegate.createTempFile(prefix, suffix, perms).map(val -> val);
    return ret;
  }

  /**
   * Creates a new file in the directory provided by the path <code>dir</code>, using the given
   * prefix and suffix to generate its name, asynchronously.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms 
   * @return a future notified on completion
   */
  public io.reactivex.Single<java.lang.String> rxCreateTempFile(java.lang.String prefix, java.lang.String suffix, java.lang.String perms) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.createTempFile(prefix, suffix, perms).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createTempFile}
   * @param prefix 
   * @param suffix 
   * @param perms 
   * @return 
   */
  public java.lang.String createTempFileBlocking(java.lang.String prefix, java.lang.String suffix, java.lang.String perms) { 
    java.lang.String ret = delegate.createTempFileBlocking(prefix, suffix, perms);
    return ret;
  }

  /**
   * Creates a new file in the directory provided by the path <code>dir</code>, using the given
   * prefix and suffix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.vertx.core.Future<java.lang.String> createTempFile(java.lang.String dir, java.lang.String prefix, java.lang.String suffix, java.lang.String perms) { 
    io.vertx.core.Future<java.lang.String> ret = delegate.createTempFile(dir, prefix, suffix, perms).map(val -> val);
    return ret;
  }

  /**
   * Creates a new file in the directory provided by the path <code>dir</code>, using the given
   * prefix and suffix to generate its name, asynchronously.
   * <p>
   * The new directory will be created with permissions as specified by <code>perms</code>.
   * </p>
   * The permission String takes the form rwxr-x--- as specified
   * in <a href="http://download.oracle.com/javase/7/docs/api/java/nio/file/attribute/PosixFilePermissions.html">here</a>.
   *
   * <p>
   * As with the <code>File.createTempFile</code> methods, this method is only
   * part of a temporary-file facility.A {@link java.lang.Runtime},
   * or the {@link java.io.File} mechanism may be used to delete the directory automatically.
   * </p>
   * @param dir the path to directory in which to create the directory
   * @param prefix the prefix string to be used in generating the directory's name; may be <code>null</code>
   * @param suffix the suffix string to be used in generating the file's name; may be <code>null</code>, in which case "<code>.tmp</code>" is used
   * @param perms the permissions string
   * @return a future notified on completion
   */
  public io.reactivex.Single<java.lang.String> rxCreateTempFile(java.lang.String dir, java.lang.String prefix, java.lang.String suffix, java.lang.String perms) { 
    return AsyncResultSingle.toSingle($handler -> {
      this.createTempFile(dir, prefix, suffix, perms).onComplete($handler);
    });
  }

  /**
   * Blocking version of {@link io.vertx.reactivex.core.file.FileSystem#createTempFile}
   * @param dir 
   * @param prefix 
   * @param suffix 
   * @param perms 
   * @return 
   */
  public java.lang.String createTempFileBlocking(java.lang.String dir, java.lang.String prefix, java.lang.String suffix, java.lang.String perms) { 
    java.lang.String ret = delegate.createTempFileBlocking(dir, prefix, suffix, perms);
    return ret;
  }

  public static FileSystem newInstance(io.vertx.core.file.FileSystem arg) {
    return arg != null ? new FileSystem(arg) : null;
  }

}
