/*
 * Decompiled with CFR 0.152.
 */
package io.vertx.junit5;

import io.vertx.core.Future;
import io.vertx.core.Handler;
import io.vertx.core.impl.NoStackTraceThrowable;
import io.vertx.junit5.Checkpoint;
import io.vertx.junit5.VertxTestContext;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.assertj.core.api.AbstractThrowableAssert;
import org.assertj.core.api.Assertions;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;

@DisplayName(value="Unit tests for VertxTestContext")
class VertxTestContextTest {
    VertxTestContextTest() {
    }

    @Test
    @DisplayName(value="Check that failing with a null exception is forbidden")
    void fail_with_null() {
        VertxTestContext context = new VertxTestContext();
        ((AbstractThrowableAssert)Assertions.assertThatThrownBy(() -> context.failNow((Throwable)null)).isInstanceOf(NullPointerException.class)).hasMessage("The exception cannot be null");
    }

    @Test
    @DisplayName(value="Check the behavior of succeeding() and that it does not complete the test context")
    void check_async_assert() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.succeeding().handle((Object)Future.succeededFuture());
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.MILLISECONDS)).isFalse();
        Assertions.assertThat((boolean)context.completed()).isFalse();
        context = new VertxTestContext();
        context.succeeding().handle((Object)Future.failedFuture((Throwable)new RuntimeException("Plop")));
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.MILLISECONDS)).isTrue();
        Assertions.assertThat((boolean)context.failed()).isTrue();
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)context.causeOfFailure()).isInstanceOf(RuntimeException.class)).hasMessage("Plop");
    }

    @Test
    @DisplayName(value="Check the behavior of failing()")
    void check_async_assert_fail() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.failing().handle((Object)Future.failedFuture((String)"Bam"));
        context.awaitCompletion(1L, TimeUnit.MILLISECONDS);
        Assertions.assertThat((boolean)context.failed()).isFalse();
        context = new VertxTestContext();
        context.failing().handle((Object)Future.succeededFuture());
        context.awaitCompletion(1L, TimeUnit.MILLISECONDS);
        Assertions.assertThat((boolean)context.failed()).isTrue();
        Assertions.assertThat((Throwable)context.causeOfFailure()).hasMessage("The asynchronous result was expected to have failed");
    }

    @Test
    @DisplayName(value="Check the behavior of succeeding(callback)")
    void check_async_assert_with_handler() throws InterruptedException {
        VertxTestContext context;
        AtomicBoolean checker = new AtomicBoolean(false);
        VertxTestContext finalContext = context = new VertxTestContext();
        Handler nextHandler = obj -> {
            checker.set(true);
            finalContext.completeNow();
        };
        context.succeeding(nextHandler).handle((Object)Future.succeededFuture());
        Assertions.assertThat((boolean)context.awaitCompletion(2L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.completed()).isTrue();
        Assertions.assertThat((AtomicBoolean)checker).isTrue();
        checker.set(false);
        context = new VertxTestContext();
        context.succeeding(nextHandler).handle((Object)Future.failedFuture((Throwable)new RuntimeException("Plop")));
        Assertions.assertThat((boolean)context.awaitCompletion(2L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.failed()).isTrue();
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)context.causeOfFailure()).isInstanceOf(RuntimeException.class)).hasMessage("Plop");
        Assertions.assertThat((AtomicBoolean)checker).isFalse();
    }

    @Test
    @DisplayName(value="Check callback exception of succeeding(callback)")
    void check_succeeding_callback_with_exception() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        Handler nextHandler = obj -> {
            throw new RuntimeException("Boom");
        };
        context.succeeding(nextHandler).handle((Object)Future.succeededFuture());
        Assertions.assertThat((boolean)context.awaitCompletion(2L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.failed()).isTrue();
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)context.causeOfFailure()).isInstanceOf(RuntimeException.class)).hasMessage("Boom");
    }

    @Test
    @DisplayName(value="Check the behavior of failing(callback)")
    void check_async_assert_fail_with_handler() throws InterruptedException {
        VertxTestContext context;
        AtomicBoolean checker = new AtomicBoolean(false);
        VertxTestContext finalContext = context = new VertxTestContext();
        Handler nextHandler = ar -> {
            checker.set(true);
            finalContext.completeNow();
        };
        context.failing(nextHandler).handle((Object)Future.failedFuture((String)"Bam"));
        Assertions.assertThat((boolean)context.awaitCompletion(2L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.completed()).isTrue();
        Assertions.assertThat((AtomicBoolean)checker).isTrue();
        checker.set(false);
        context = new VertxTestContext();
        context.failing(nextHandler).handle((Object)Future.succeededFuture());
        Assertions.assertThat((boolean)context.awaitCompletion(2L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.failed()).isTrue();
        Assertions.assertThat((Throwable)context.causeOfFailure()).hasMessage("The asynchronous result was expected to have failed");
    }

    @Test
    @DisplayName(value="Check callback exception of failing(callback)")
    void check_failing_callback_with_exception() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        Handler nextHandler = throwable -> {
            throw new RuntimeException("Pow");
        };
        context.failing(nextHandler).handle((Object)Future.failedFuture((String)"some failure"));
        Assertions.assertThat((boolean)context.awaitCompletion(2L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.failed()).isTrue();
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)context.causeOfFailure()).isInstanceOf(RuntimeException.class)).hasMessage("Pow");
    }

    @Test
    @DisplayName(value="Check the behavior of verify() and no error")
    void check_verify_ok() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.verify(() -> {
            Assertions.assertThat((String)"ok").isEqualTo("ok");
            context.completeNow();
        });
        Assertions.assertThat((boolean)context.awaitCompletion(500L, TimeUnit.MILLISECONDS)).isTrue();
    }

    @Test
    @DisplayName(value="Check the behavior of verify() with an error")
    void check_verify_fail() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.verify(() -> {
            throw new RuntimeException("Bam");
        });
        Assertions.assertThat((boolean)context.awaitCompletion(500L, TimeUnit.MILLISECONDS)).isTrue();
        Assertions.assertThat((boolean)context.failed()).isTrue();
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)context.causeOfFailure()).isInstanceOf(RuntimeException.class)).hasMessage("Bam");
    }

    @Test
    @DisplayName(value="Check that flagging 2 checkpoints completes the test context")
    void check_checkpoint() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        Checkpoint a = context.checkpoint();
        Checkpoint b = context.checkpoint();
        new Thread(() -> ((Checkpoint)a).flag()).start();
        new Thread(() -> ((Checkpoint)b).flag()).start();
        Assertions.assertThat((boolean)context.awaitCompletion(500L, TimeUnit.MILLISECONDS)).isTrue();
    }

    @Test
    @DisplayName(value="Check that not flagging all checkpoints ends up in a timeout")
    void checK_not_all_checkpoints_passed_timesout() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        Checkpoint a = context.checkpoint(2);
        context.checkpoint();
        new Thread(() -> ((Checkpoint)a).flag()).start();
        new Thread(() -> ((Checkpoint)a).flag()).start();
        Assertions.assertThat((boolean)context.awaitCompletion(500L, TimeUnit.MILLISECONDS)).isFalse();
    }

    @Test
    @DisplayName(value="Check that flagging strict checkpoints more than expected fails the test context")
    void check_strict_checkpoint_overuse() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        Checkpoint a = context.checkpoint();
        Checkpoint b = context.checkpoint();
        new Thread(() -> ((Checkpoint)a).flag()).start();
        new Thread(() -> ((Checkpoint)a).flag()).start();
        Assertions.assertThat((boolean)context.awaitCompletion(500L, TimeUnit.MILLISECONDS)).isTrue();
        Assertions.assertThat((boolean)context.failed()).isTrue();
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)context.causeOfFailure()).isInstanceOf(IllegalStateException.class)).hasMessageContaining("flagged too many times");
    }

    @Test
    @DisplayName(value="Check that a lax checkpoint can be flagged more often than required")
    void check_lax_checkpoint_no_overuse() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        Checkpoint a = context.laxCheckpoint();
        Checkpoint b = context.checkpoint();
        new Thread(() -> {
            a.flag();
            a.flag();
            a.flag();
            b.flag();
        }).start();
        Assertions.assertThat((boolean)context.awaitCompletion(500L, TimeUnit.MILLISECONDS)).isTrue();
        Assertions.assertThat((boolean)context.failed()).isFalse();
        Assertions.assertThat((boolean)context.completed()).isTrue();
    }

    @Test
    @DisplayName(value="Check that failing an already completed context is not possible")
    void complete_then_fail() {
        VertxTestContext context = new VertxTestContext();
        context.completeNow();
        context.failNow((Throwable)new IllegalStateException("Oh"));
        Assertions.assertThat((boolean)context.completed()).isTrue();
        Assertions.assertThat((boolean)context.failed()).isFalse();
        Assertions.assertThat((Throwable)context.causeOfFailure()).isNull();
    }

    @Test
    @DisplayName(value="Just fail immediately and on the test runner thread")
    void just_fail() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.failNow((Throwable)new RuntimeException("Woops"));
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.failed()).isTrue();
    }

    @Test
    @DisplayName(value="Pass a success to a succeedingThenComplete() async handler")
    void check_succeedingThenComplete_success() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.succeedingThenComplete().handle((Object)Future.succeededFuture());
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.completed()).isTrue();
    }

    @Test
    @DisplayName(value="Pass a failure to a succeedingThenComplete() async handler")
    void check_succeedingThenComplete_failure() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.succeedingThenComplete().handle((Object)Future.failedFuture((Throwable)new RuntimeException("Boo!")));
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.completed()).isFalse();
        Assertions.assertThat((boolean)context.failed()).isTrue();
        Assertions.assertThat((Throwable)context.causeOfFailure()).hasMessage("Boo!");
    }

    @Test
    @DisplayName(value="Pass a failure to a failingThenComplete() async handler")
    void check_failingThenComplete_failure() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.failingThenComplete().handle((Object)Future.failedFuture((Throwable)new IllegalArgumentException("42")));
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.failed()).isFalse();
        Assertions.assertThat((boolean)context.completed()).isTrue();
        Assertions.assertThat((Throwable)context.causeOfFailure()).isNull();
    }

    @Test
    @DisplayName(value="Pass a success to a failingThenComplete() async handler")
    void check_failingThenComplete_success() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.failingThenComplete().handle((Object)Future.succeededFuture((Object)"gold"));
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.completed()).isFalse();
        Assertions.assertThat((boolean)context.failed()).isTrue();
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)context.causeOfFailure()).isInstanceOf(AssertionError.class)).hasMessage("The asynchronous result was expected to have failed");
    }

    @Test
    @DisplayName(value="Pass future assertComplete")
    void check_future_completion() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.assertComplete(Future.succeededFuture((Object)"bla")).compose(s -> context.assertComplete(Future.succeededFuture((Object)(s + "bla")))).onComplete(context.succeeding(res -> {
            Assertions.assertThat((String)res).isEqualTo("blabla");
            context.completeNow();
        }));
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.completed()).isTrue();
    }

    @Test
    @DisplayName(value="Fail future assertComplete")
    void check_future_completion_failure() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.assertComplete(Future.succeededFuture((Object)"bla")).compose(s -> context.assertComplete(Future.failedFuture((Throwable)new IllegalStateException(s + "bla")))).onComplete(context.succeeding(res -> context.completeNow()));
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.completed()).isFalse();
        Assertions.assertThat((boolean)context.failed()).isTrue();
        Assertions.assertThat((Throwable)context.causeOfFailure()).isInstanceOf(AssertionError.class);
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)context.causeOfFailure().getCause()).isInstanceOf(IllegalStateException.class)).hasMessage("blabla");
    }

    @Test
    @DisplayName(value="Pass future chain assertComplete")
    void check_future_chain_completion() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.assertComplete(Future.succeededFuture((Object)"bla").compose(s -> Future.failedFuture((Throwable)new IllegalStateException(s + "bla"))).recover(ex -> Future.succeededFuture((Object)ex.getMessage()))).onComplete(context.succeeding(res -> {
            Assertions.assertThat((Object)res).isEqualTo((Object)"blabla");
            context.completeNow();
        }));
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.completed()).isTrue();
    }

    @Test
    @DisplayName(value="Fail future chain assertComplete")
    void check_future_chain_completion_failure() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.assertComplete(Future.succeededFuture((Object)"bla").compose(s -> Future.failedFuture((Throwable)new IllegalStateException(s + "bla")))).onComplete(context.succeeding(res -> context.completeNow()));
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.completed()).isFalse();
        Assertions.assertThat((boolean)context.failed()).isTrue();
        Assertions.assertThat((Throwable)context.causeOfFailure()).isInstanceOf(AssertionError.class);
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)context.causeOfFailure().getCause()).isInstanceOf(IllegalStateException.class)).hasMessage("blabla");
    }

    @Test
    @DisplayName(value="Pass future assertFailure")
    void check_future_failing() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.assertFailure(Future.failedFuture((Throwable)new IllegalStateException("bla"))).recover(s -> context.assertFailure(Future.failedFuture((Throwable)new IllegalStateException(s.getMessage() + "bla")))).onComplete(context.failing(ex -> {
            ((AbstractThrowableAssert)Assertions.assertThat((Throwable)ex).isInstanceOf(IllegalStateException.class)).hasMessage("blabla");
            context.completeNow();
        }));
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.completed()).isTrue();
    }

    @Test
    @DisplayName(value="Fail future assertComplete")
    void check_future_failing_failure() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.assertFailure(Future.failedFuture((Throwable)new IllegalStateException("bla"))).recover(s -> context.assertFailure(Future.succeededFuture((Object)(s.getMessage() + "bla")))).onComplete(context.succeeding(res -> context.completeNow()));
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((boolean)context.completed()).isFalse();
        Assertions.assertThat((boolean)context.failed()).isTrue();
        ((AbstractThrowableAssert)Assertions.assertThat((Throwable)context.causeOfFailure()).isInstanceOf(AssertionError.class)).hasMessage("Future completed with value: blabla");
    }

    @Test
    @DisplayName(value="Call verify() with a block that throws an exception")
    void check_verify_with_exception() throws InterruptedException {
        VertxTestContext context = new VertxTestContext();
        context.verify(() -> {
            throw new RuntimeException("!");
        });
        Assertions.assertThat((boolean)context.awaitCompletion(1L, TimeUnit.SECONDS)).isTrue();
        Assertions.assertThat((Throwable)context.causeOfFailure()).hasMessage("!").isInstanceOf(RuntimeException.class);
    }

    @Test
    @DisplayName(value="Check that unsatisfied call sites are properly identified")
    void check_unsatisifed_checkpoint_callsites() {
        VertxTestContext context = new VertxTestContext();
        Checkpoint a = context.checkpoint();
        Checkpoint b = context.checkpoint(2);
        Assertions.assertThat((Iterable)context.unsatisfiedCheckpointCallSites()).hasSize(2);
        a.flag();
        b.flag();
        Assertions.assertThat((Iterable)context.unsatisfiedCheckpointCallSites()).hasSize(1);
        StackTraceElement element = (StackTraceElement)context.unsatisfiedCheckpointCallSites().iterator().next();
        Assertions.assertThat((String)element.getClassName()).isEqualTo(VertxTestContextTest.class.getName());
        Assertions.assertThat((String)element.getMethodName()).isEqualTo("check_unsatisifed_checkpoint_callsites");
        b.flag();
        Assertions.assertThat((Iterable)context.unsatisfiedCheckpointCallSites()).isEmpty();
    }

    @Test
    @DisplayName(value="Check failNow() called with a string")
    void check_fail_now_called_with_a_string() {
        VertxTestContext context = new VertxTestContext();
        context.failNow("error message");
        Assertions.assertThat((boolean)context.failed()).isTrue();
        Assertions.assertThat((Throwable)context.causeOfFailure()).isInstanceOf(NoStackTraceThrowable.class);
        Assertions.assertThat((Throwable)context.causeOfFailure()).hasMessage("error message");
    }
}

