/*
 * Decompiled with CFR 0.152.
 */
package redis.clients.jedis;

import java.lang.ref.WeakReference;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicLong;

public class ResourceLifecycleManager {
    private static final ResourceLifecycleManager INSTANCE = new ResourceLifecycleManager();
    private final ConcurrentHashMap<String, WeakReference<AutoCloseable>> trackedResources;
    private final ScheduledExecutorService cleanupExecutor;
    private final AtomicLong resourceIdCounter;
    private final Object shutdownLock = new Object();
    private volatile boolean isShutdown = false;

    private ResourceLifecycleManager() {
        this.trackedResources = new ConcurrentHashMap();
        this.cleanupExecutor = Executors.newSingleThreadScheduledExecutor(r -> {
            Thread t = new Thread(r, "jedis-resource-cleanup");
            t.setDaemon(true);
            return t;
        });
        this.resourceIdCounter = new AtomicLong(0L);
        this.cleanupExecutor.scheduleWithFixedDelay(this::cleanupDeadReferences, 30L, 30L, TimeUnit.SECONDS);
        Runtime.getRuntime().addShutdownHook(new Thread(this::shutdown));
    }

    public static ResourceLifecycleManager getInstance() {
        return INSTANCE;
    }

    public String registerResource(AutoCloseable resource) {
        if (this.isShutdown) {
            throw new IllegalStateException("ResourceLifecycleManager is shutdown");
        }
        String resourceId = "resource-" + this.resourceIdCounter.incrementAndGet();
        this.trackedResources.put(resourceId, new WeakReference<AutoCloseable>(resource));
        return resourceId;
    }

    public void unregisterResource(String resourceId) {
        if (resourceId != null) {
            this.trackedResources.remove(resourceId);
        }
    }

    public void scheduleCleanup(AutoCloseable resource, long delaySeconds) {
        if (this.isShutdown) {
            this.closeQuietly(resource);
            return;
        }
        this.cleanupExecutor.schedule(() -> this.closeQuietly(resource), delaySeconds, TimeUnit.SECONDS);
    }

    public int getTrackedResourceCount() {
        this.cleanupDeadReferences();
        return this.trackedResources.size();
    }

    public void forceCleanupAll() {
        this.trackedResources.values().forEach(ref -> {
            AutoCloseable resource = (AutoCloseable)ref.get();
            if (resource != null) {
                this.closeQuietly(resource);
            }
        });
        this.trackedResources.clear();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        Object object = this.shutdownLock;
        synchronized (object) {
            if (this.isShutdown) {
                return;
            }
            this.isShutdown = true;
        }
        this.forceCleanupAll();
        this.cleanupExecutor.shutdown();
        try {
            if (!this.cleanupExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                this.cleanupExecutor.shutdownNow();
            }
        }
        catch (InterruptedException e) {
            this.cleanupExecutor.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    public boolean isShutdown() {
        return this.isShutdown;
    }

    private void cleanupDeadReferences() {
        this.trackedResources.entrySet().removeIf(entry -> ((WeakReference)entry.getValue()).get() == null);
    }

    private void closeQuietly(AutoCloseable resource) {
        if (resource != null) {
            try {
                resource.close();
            }
            catch (Exception e) {
                System.err.println("Error closing resource:");
                e.printStackTrace();
            }
        }
    }

    public static class ManagedResource
    implements AutoCloseable {
        private final AutoCloseable delegate;
        private final String resourceId;
        private final long creationTime;
        private volatile boolean closed = false;

        public ManagedResource(AutoCloseable delegate) {
            this.delegate = delegate;
            this.resourceId = ResourceLifecycleManager.getInstance().registerResource(this);
            this.creationTime = System.currentTimeMillis();
        }

        @Override
        public void close() throws Exception {
            if (!this.closed) {
                this.closed = true;
                ResourceLifecycleManager.getInstance().unregisterResource(this.resourceId);
                this.delegate.close();
            }
        }

        public boolean isClosed() {
            return this.closed;
        }

        public long getAgeMillis() {
            return System.currentTimeMillis() - this.creationTime;
        }

        public String getResourceId() {
            return this.resourceId;
        }

        public AutoCloseable getDelegate() {
            return this.delegate;
        }
    }
}

