package org.nuiton.topia.persistence.script;

/*-
 * #%L
 * ToPIA Extension :: persistence
 * %%
 * Copyright (C) 2018 - 2019 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import org.nuiton.topia.persistence.TopiaException;

import java.io.BufferedWriter;
import java.io.Closeable;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.nio.charset.Charset;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Objects;
import java.util.function.Supplier;
import java.util.zip.GZIPOutputStream;

/**
 * Created by tchemit on 05/05/2018.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
@SuppressWarnings("WeakerAccess")
public class SqlScriptWriter implements Closeable {

    private final Supplier<OutputStream> target;
    private final boolean skipComment;
    private final boolean skipTrim;
    private final BufferedWriter writer;
    private long statementCount;

    private SqlScriptWriter(Supplier<OutputStream> target, boolean skipComment, boolean skipTrim, boolean gzip, Charset encoding) {
        this.target = target;
        this.skipComment = skipComment;
        this.skipTrim = skipTrim;

        //            Files.createDirectories(target.getParent());
        OutputStream stream = target.get();
        try {
//                stream = Files.newOutputStream(target);
            if (gzip) {
                stream = new GZIPOutputStream(stream);
            }
        } catch (IOException e) {
            throw new IllegalStateException("Can't get output stream from: " + target, e);
        }
        this.writer = new BufferedWriter(new OutputStreamWriter(stream, encoding));
    }

    public static SqlScriptWriter of(Path target) {
        return builder(target).build();
    }

    public static SqlScriptWriter of(OutputStream target) {
        return builder(target).build();
    }

    public static Builder builder(Path target) {
        return new Builder(() -> {
            try {
                return Files.newOutputStream(target);
            } catch (IOException e) {
                throw new TopiaException(e);
            }
        });
    }

    public static Builder builder(OutputStream target) {
        return new Builder(() -> target);
    }

    public void writeSql(String sql) {
        if (skipTrim && sql.trim().isEmpty()) {
            return;
        }
        if (skipComment && sql.trim().startsWith("--")) {
            return;
        }

        try {
            writer.write(sql);
            if (!sql.endsWith(";")) {
                writer.write(";");
            }
            writer.newLine();
        } catch (IOException e) {
            throw new IllegalStateException(String.format("can't write sql: [%s] to path: %s", sql, target), e);
        }
        statementCount++;
    }

    public long getStatementCount() {
        return statementCount;
    }

    @Override
    public void close() throws IOException {
        writer.close();
    }

    public void flush() throws IOException {
        writer.flush();
    }

    public void writeScript(Path sqlScriptFile) throws IOException {

        SqlScriptReader.Builder builder = SqlScriptReader.builder(sqlScriptFile);
        if (!skipComment) {
            builder.keepCommentLine();
        }
        if (!skipTrim) {
            builder.keepEmptyLine();
        }
        try (SqlScriptReader sqlScripReader = builder.build()) {
            for (String statement : sqlScripReader) {
                writeSql(statement);
            }
        }
    }

    public static class Builder {

        private final Supplier<OutputStream> source;
        private Charset encoding = StandardCharsets.UTF_8;
        private boolean skipComment = true;
        private boolean skipTrim = true;
        private boolean gzip;

        public Builder(Supplier<OutputStream> source) {
            this.source = Objects.requireNonNull(source);
        }

        public Builder gzip() {
            this.gzip = true;
            return this;
        }

        public Builder keepCommentLine() {
            this.skipComment = false;
            return this;
        }

        public Builder keepEmptyLine() {
            this.skipTrim = false;
            return this;
        }

        public Builder encoding(Charset encoding) {
            this.encoding = Objects.requireNonNull(encoding);
            return this;
        }

        public SqlScriptWriter build() {
            return new SqlScriptWriter(source, skipComment, skipTrim, gzip, encoding);
        }
    }

}
