package org.nuiton.topia.persistence.metadata;

/*
 * #%L
 * ToPIA Extension :: persistence
 * %%
 * Copyright (C) 2018 - 2019 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import com.google.common.base.Preconditions;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ArrayListMultimap;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Multimap;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.IOException;
import java.io.InputStreamReader;
import java.io.Reader;
import java.net.URL;
import java.util.Collection;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.stream.Stream;

/**
 * Méta-modèle topia simplifié qui contient des informations utile pour des algorithmes générique sur les entités.
 * <p>
 * Ce méta-modèle est juste un conteneur de méta-modèle d'entités.
 * <p>
 * Created on 03/01/16.
 *
 * @author Tony Chemit - dev@tchemit.fr
 * @since 5.0
 */
public class TopiaMetadataModel implements Iterable<TopiaMetadataEntity> {

    private static final Logger log = LogManager.getLogger(TopiaMetadataModel.class);

    /**
     * Metadata for all entities of the model.
     */
    protected final Map<String, TopiaMetadataEntity> entities = new LinkedHashMap<>();
    /**
     * To get all compositions for each entity type.
     */
    private final transient LoadingCache<String, Set<TopiaMetadataComposition>> compositions = CacheBuilder.newBuilder().build(new CacheLoader<String, Set<TopiaMetadataComposition>>() {
        @Override
        public Set<TopiaMetadataComposition> load(String key) {
            TopiaMetadataEntity source = Objects.requireNonNull(getEntity(key));
            ImmutableSet.Builder<TopiaMetadataComposition> builder = ImmutableSet.builder();
            for (Map.Entry<String, String> link : source.getManyToOneAssociations().entrySet()) {
                TopiaMetadataEntity target = Objects.requireNonNull(getEntity(link.getValue()));
                builder.add(new TopiaMetadataComposition(source, link.getKey(), target));
            }
            return builder.build();
        }
    });
    /**
     * To get all reverse compositions for each entity type.
     */
    private final transient LoadingCache<String, Set<TopiaMetadataComposition>> reverseCompositions = CacheBuilder.newBuilder().build(new CacheLoader<String, Set<TopiaMetadataComposition>>() {
        @Override
        public Set<TopiaMetadataComposition> load(String key) {
            TopiaMetadataEntity target = Objects.requireNonNull(getEntity(key));
            ImmutableSet.Builder<TopiaMetadataComposition> builder = ImmutableSet.builder();
            for (Iterator<TopiaMetadataEntity> it = streamWithoutAbstract().iterator(); it.hasNext(); ) {
                TopiaMetadataEntity source = it.next();
                for (Map.Entry<String, String> entry : source.getManyToOneAssociations().entrySet()) {
                    if (entry.getValue().equals(key)) {
                        builder.add(new TopiaMetadataComposition(source, entry.getKey(), target));
                    }
                }
            }
            return builder.build();
        }
    });
    /**
     * To get all reverse associations for each entity type.
     */
    private final transient LoadingCache<String, Set<TopiaMetadataAssociation>> reverseAssociations = CacheBuilder.newBuilder().build(new CacheLoader<String, Set<TopiaMetadataAssociation>>() {
        @Override
        public Set<TopiaMetadataAssociation> load(String key) {
            TopiaMetadataEntity target = Objects.requireNonNull(getEntity(key));
            ImmutableSet.Builder<TopiaMetadataAssociation> builder = ImmutableSet.builder();
            for (Iterator<TopiaMetadataEntity> it = streamWithoutAbstract().iterator(); it.hasNext(); ) {
                TopiaMetadataEntity source = it.next();
                for (Map.Entry<String, String> entry : source.getManyToManyAssociations().entrySet()) {
                    if (entry.getValue().equals(key)) {
                        builder.add(new TopiaMetadataAssociation(source, entry.getKey(), target));
                    }
                }
            }
            return builder.build();
        }
    });
    /**
     * To get all associations for each entity type.
     */
    private final transient LoadingCache<String, Set<TopiaMetadataAssociation>> associations = CacheBuilder.newBuilder().build(new CacheLoader<String, Set<TopiaMetadataAssociation>>() {
        @Override
        public Set<TopiaMetadataAssociation> load(String key) {
            TopiaMetadataEntity source = Objects.requireNonNull(getEntity(key));
            ImmutableSet.Builder<TopiaMetadataAssociation> builder = ImmutableSet.builder();
            for (Map.Entry<String, String> link : source.getManyToManyAssociations().entrySet()) {
                TopiaMetadataEntity target = Objects.requireNonNull(getEntity(link.getValue()));
                builder.add(new TopiaMetadataAssociation(source, link.getKey(), target));
            }
            return builder.build();
        }
    });
    /**
     * To get all schemas (key is schema, value is set of entities in this schema).
     */
    private transient Multimap<String, TopiaMetadataEntity> schemas;

    public static TopiaMetadataModel load(URL url) throws IOException {
        try (Reader reader = new InputStreamReader(url.openStream())) {
            Gson gson = new GsonBuilder().create();
            return gson.fromJson(reader, TopiaMetadataModel.class);
        }
    }

    public Collection<TopiaMetadataEntity> getEntities() {
        return entities.values();
    }

    public Multimap<String, TopiaMetadataEntity> getSchemas() {
        if (schemas==null) {
            schemas = ArrayListMultimap.create();
            for (TopiaMetadataEntity entity : entities.values()) {
             schemas.put(entity.getDbSchemaName(), entity);
            }
        }
        return schemas;
    }

    public Set<String> getSchemaNames() {
        return getSchemas().keySet();
    }

    public TopiaMetadataEntity getEntity(String type) {
        return entities.get(type);
    }

    public Optional<TopiaMetadataEntity> getOptionalEntity(String type) {
        return Optional.ofNullable(getEntity(type));
    }

    public void accept(TopiaMetadataModelVisitor visitor) {
        visitor.visitModelStart(this);
        for (TopiaMetadataEntity entity : entities.values()) {
            entity.accept(visitor, this);
        }
        visitor.visitModelEnd(this);
    }

    public TopiaMetadataEntity newEntity(String parent, String type, boolean isAbstract, String dbSchemaName, String dbTableName) {
        String literalName = (dbSchemaName == null ? "" : (dbSchemaName + "_")) + type;
        Preconditions.checkState(!entities.containsKey(literalName), literalName + " already in cache");
        TopiaMetadataEntity clazz = new TopiaMetadataEntity(parent, literalName, isAbstract, dbSchemaName, dbTableName);
        entities.put(literalName, clazz);
        log.debug("create new entity: " + clazz.getType());
        return clazz;
    }

    public Set<TopiaMetadataComposition> getReverseCompositions(TopiaMetadataEntity type) {
        return getReverseCompositions(Objects.requireNonNull(type).getType());
    }

    public Set<TopiaMetadataComposition> getReverseCompositions(String type) {
        try {
            return reverseCompositions.get(Objects.requireNonNull(type));
        } catch (ExecutionException e) {
            throw new RuntimeException("Can't get reverse compositions for type: " + type, e);
        }
    }

    public Set<TopiaMetadataComposition> getCompositions(TopiaMetadataEntity type) {
        return getCompositions(Objects.requireNonNull(type).getType());
    }

    public Set<TopiaMetadataComposition> getCompositions(String type) {
        try {
            return compositions.get(Objects.requireNonNull(type));
        } catch (ExecutionException e) {
            throw new RuntimeException("Can't get compositions for type: " + type, e);
        }
    }

    public Set<TopiaMetadataAssociation> getReverseAssociations(TopiaMetadataEntity type) {
        return getReverseAssociations(Objects.requireNonNull(type).getType());
    }

    public Set<TopiaMetadataAssociation> getReverseAssociations(String type) {
        try {
            return reverseAssociations.get(Objects.requireNonNull(type));
        } catch (ExecutionException e) {
            throw new RuntimeException("Can't get reverse associations for type: " + type, e);
        }
    }

    public Set<TopiaMetadataAssociation> getAssociations(TopiaMetadataEntity type) {
        return getAssociations(Objects.requireNonNull(type).getType());
    }

    public Set<TopiaMetadataAssociation> getAssociations(String type) {
        try {
            return associations.get(Objects.requireNonNull(type));
        } catch (ExecutionException e) {
            throw new RuntimeException("Can't get associations for type: " + type, e);
        }
    }

    @SuppressWarnings("NullableProblems")
    @Override
    public Iterator<TopiaMetadataEntity> iterator() {
        return stream().iterator();
    }

    public Stream<TopiaMetadataEntity> stream() {
        return entities.values().stream();
    }

    public Stream<TopiaMetadataEntity> streamWithoutAbstract() {
        return entities.values().stream().filter(e -> !e.isAbstract());
    }
}
