package org.nuiton.topia.persistence.internal.support;

/*-
 * #%L
 * ToPIA Extension :: persistence
 * %%
 * Copyright (C) 2018 - 2019 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/gpl-3.0.html>.
 * #L%
 */

import org.nuiton.topia.persistence.TopiaDao;
import org.nuiton.topia.persistence.TopiaDaoSupplier;
import org.nuiton.topia.persistence.TopiaEntity;
import org.nuiton.topia.persistence.TopiaEntityEnum;
import org.nuiton.topia.persistence.TopiaEntityEnumProvider;
import org.nuiton.topia.persistence.metadata.TopiaMetadataAssociation;
import org.nuiton.topia.persistence.metadata.TopiaMetadataComposition;
import org.nuiton.topia.persistence.metadata.TopiaMetadataEntity;
import org.nuiton.topia.persistence.metadata.TopiaMetadataModel;
import org.nuiton.topia.persistence.support.TopiaMetadataModelSupport;
import org.nuiton.topia.persistence.support.TopiaUsageSupport;

import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.Predicate;

/**
 * Created by tchemit on 19/09/2018.
 *
 * @author Tony Chemit - dev@tchemit.fr
 */
public class TopiaUsageSupportImpl implements TopiaUsageSupport {

    private final TopiaMetadataModelSupport topiaMetadataModelSupport;
    private final TopiaEntityEnumProvider topiaEntityEnumProvider;

    public TopiaUsageSupportImpl(TopiaMetadataModelSupport topiaMetadataModelSupport, TopiaEntityEnumProvider topiaEntityEnumProvider) {
        this.topiaMetadataModelSupport = topiaMetadataModelSupport;
        this.topiaEntityEnumProvider = topiaEntityEnumProvider;
    }

    @SuppressWarnings("unchecked")
    @Override
    public Map<Class<? extends TopiaEntity>, List<? extends TopiaEntity>> findReverseCompositions(TopiaDaoSupplier topiaDaoSupplier, TopiaEntity entity, Predicate<Class<? extends TopiaEntity>> entityTypeFilter) {
        Map<Class<? extends TopiaEntity>, List<? extends TopiaEntity>> builder = new LinkedHashMap<>();
        TopiaEntityEnum targetEntityEnum = topiaEntityEnumProvider.getEntityEnum(entity.getClass());
        Set<TopiaMetadataComposition> reverseCompositions = getMetadataModel().getReverseCompositions(targetEntityEnum.name());
        for (TopiaMetadataComposition link : reverseCompositions) {
            TopiaMetadataEntity owner = link.getOwner();
            TopiaEntityEnum sourceEntityEnum = topiaEntityEnumProvider.getEntityEnum(owner.getType());
            Class<? extends TopiaEntity> sourceType = sourceEntityEnum.getContract();
            TopiaDao<? extends TopiaEntity> dao = topiaDaoSupplier.getDao(sourceType);
            List<? extends TopiaEntity> entities = dao.forEquals(link.getTargetPropertyName(), entity).findAll();
            if (!entities.isEmpty()) {
                builder.merge(sourceType, entities, (a, b) -> {a.addAll((List)b); return a;});
            }
        }
        return builder;
    }

    @SuppressWarnings("unchecked")
    @Override
    public Map<Class<? extends TopiaEntity>, List<? extends TopiaEntity>> findReverseAssociations(TopiaDaoSupplier topiaDaoSupplier, TopiaEntity entity, Predicate<Class<? extends TopiaEntity>> entityTypeFilter) {
        Map<Class<? extends TopiaEntity>, List<? extends TopiaEntity>> builder = new LinkedHashMap<>();
        TopiaEntityEnum targetEntityEnum = topiaEntityEnumProvider.getEntityEnum(entity.getClass());
        Set<TopiaMetadataAssociation> reverseAssociations = getMetadataModel().getReverseAssociations(targetEntityEnum.name());
        for (TopiaMetadataAssociation link : reverseAssociations) {
            TopiaMetadataEntity owner = link.getOwner();
            TopiaEntityEnum sourceEntityEnum = topiaEntityEnumProvider.getEntityEnum(owner.getType());
            Class<? extends TopiaEntity> sourceType = sourceEntityEnum.getContract();
            if (entityTypeFilter.test(sourceType)) {
                TopiaDao<? extends TopiaEntity> dao = topiaDaoSupplier.getDao(sourceType);
                List<? extends TopiaEntity> entities = new LinkedList<>(dao.forContains(link.getTargetPropertyName(), entity).findAll());
                if (!entities.isEmpty()) {
                    builder.merge(sourceType, entities, (a, b) -> {a.addAll((List)b); return a;});
                }
            }
        }
        return builder;
    }

    @Override
    public Map<Class<? extends TopiaEntity>, Long> countReverseCompositions(TopiaDaoSupplier topiaDaoSupplier, TopiaEntity entity, Predicate<Class<? extends TopiaEntity>> entityTypeFilter) {
        Map<Class<? extends TopiaEntity>, Long> builder = new LinkedHashMap<>();
        TopiaEntityEnum targetEntityEnum = topiaEntityEnumProvider.getEntityEnum(entity.getClass());
        Set<TopiaMetadataComposition> reverseCompositions = getMetadataModel().getReverseCompositions(targetEntityEnum.name());
        for (TopiaMetadataComposition link : reverseCompositions) {
            TopiaMetadataEntity owner = link.getOwner();
            TopiaEntityEnum sourceEntityEnum = topiaEntityEnumProvider.getEntityEnum(owner.getType());
            Class<? extends TopiaEntity> sourceType = sourceEntityEnum.getContract();
            TopiaDao<? extends TopiaEntity> dao = topiaDaoSupplier.getDao(sourceType);
            long entities = dao.forEquals(link.getTargetPropertyName(), entity).count();
            if (entities > 0) {
                builder.merge(sourceType, entities, (a, b) -> a + b);
            }
        }
        return builder;
    }

    @Override
    public Map<Class<? extends TopiaEntity>, Long> countReverseAssociations(TopiaDaoSupplier topiaDaoSupplier, TopiaEntity entity, Predicate<Class<? extends TopiaEntity>> entityTypeFilter) {
        Map<Class<? extends TopiaEntity>, Long> builder = new LinkedHashMap<>();
        TopiaEntityEnum targetEntityEnum = topiaEntityEnumProvider.getEntityEnum(entity.getClass());
        Set<TopiaMetadataAssociation> reverseAssociations = getMetadataModel().getReverseAssociations(targetEntityEnum.name());
        for (TopiaMetadataAssociation link : reverseAssociations) {
            TopiaMetadataEntity owner = link.getOwner();
            TopiaEntityEnum sourceEntityEnum = topiaEntityEnumProvider.getEntityEnum(owner.getType());
            Class<? extends TopiaEntity> sourceType = sourceEntityEnum.getContract();
            TopiaDao<? extends TopiaEntity> dao = topiaDaoSupplier.getDao(sourceType);
            long entities = dao.forEquals(link.getTargetPropertyName(), entity).count();
            if (entities > 0) {
                builder.merge(sourceType, entities, (a, b) -> a + b);
            }
        }
        return builder;
    }

    private TopiaMetadataModel getMetadataModel() {
        return topiaMetadataModelSupport.getMetadataModel();
    }
}
