package io.ultreia.java4all.application.context;

/*-
 * #%L
 * Application context
 * %%
 * Copyright (C) 2019 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 *
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import io.ultreia.java4all.util.ServiceLoaders;
import io.ultreia.java4all.util.SingletonSupplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.util.Optional;
import java.util.function.Supplier;

public class ApplicationComponentSupplier<O> extends SingletonSupplier<O> {

    private static final Logger log = LogManager.getLogger(ApplicationComponentSupplier.class);
    private final Class<O> componentType;

    public static <O> ApplicationComponentSupplier<O> createFromConstructor(Class<O> componentType, boolean requireNotNull, Class... dependencies) {
        Supplier<O> supplier = createSupplierFromConstructor(componentType, dependencies);
        return new ApplicationComponentSupplier<O>(componentType, requireNotNull, supplier);
    }

    public static <O> ApplicationComponentSupplier<O> createFromServiceLoader(Class<O> componentType,boolean requireNotNull) {
        Supplier<O> supplier = createSupplierFromServiceLoader(componentType);
        return new ApplicationComponentSupplier<O>(componentType, requireNotNull,supplier);
    }

    public ApplicationComponentSupplier(Class<O> componentType, boolean requireNotNull, Supplier<O> supplier) {
        super(() -> {
            log.info(String.format("Create component value from supplier: %s - %s", componentType.getName(), supplier));
            return supplier.get();
        }, requireNotNull);
        this.componentType = componentType;
    }

    @Override
    public Optional<O> clear() {
        if (withValue()) {
            log.info(String.format("Clear component value: %s", componentType.getName()));
        }
        return super.clear();
    }

    private static <O> Supplier<O> createSupplierFromConstructor(Class<O> componentType, Class... dependencies) {
        if (dependencies.length == 0) {
            return () -> {
                log.info(String.format("Create component value from default constructor: %s", componentType.getName()));
                try {
                    return componentType.newInstance();
                } catch (Exception e) {
                    throw new IllegalArgumentException("Can't instantiate component: " + componentType.getName(), e);
                }
            };
        }

        return () -> {
            Object[] parameters = new Object[dependencies.length];
            ApplicationContext applicationContext = ApplicationContext.get();
            for (int i = 0; i < dependencies.length; i++) {
                Class<?> dependency = dependencies[i];
                ApplicationComponent<?> component = applicationContext.getComponent(dependency);
                parameters[i] = component.get();
            }
            try {
                log.info(String.format("Create component value from constructor: %s", componentType.getName()));
                return componentType.getDeclaredConstructor(dependencies).newInstance(parameters);
            } catch (Exception e) {
                throw new IllegalArgumentException("Can't get component: " + componentType.getName(), e);
            }
        };
    }

    private static <O> Supplier<O> createSupplierFromServiceLoader(Class<O> componentType) {

        return () -> {

            try {
                log.info(String.format("Create component value from service loader: %s", componentType.getName()));
                return ServiceLoaders.loadUniqueService(componentType);
            } catch (Exception e) {
                throw new IllegalArgumentException("Can't get component: " + componentType.getName(), e);
            }

        };
    }
}
