package io.ultreia.java4all.application.context;

/*-
 * #%L
 * Application context
 * %%
 * Copyright (C) 2019 Ultreia.io
 * %%
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Lesser Public License for more details.
 * 
 * You should have received a copy of the GNU General Lesser Public
 * License along with this program.  If not, see
 * <http://www.gnu.org/licenses/lgpl-3.0.html>.
 * #L%
 */

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import java.io.Closeable;
import java.io.IOException;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.function.Supplier;

public class ApplicationComponent<O> implements Supplier<O>, Closeable {

    private static final Logger log = LogManager.getLogger(ApplicationComponent.class);

    private final String name;
    private final Class<O> componentType;
    private final List<Class> dependencies;
    private final ApplicationComponentSupplier<O> supplier;
    private final List<Class> hints;

    public ApplicationComponent(String name, Class<O> componentType, boolean requireNotNull, Class[] hints, Class[] dependencies) {
        this(name, componentType, hints, ApplicationComponentSupplier.createFromConstructor(componentType, requireNotNull, dependencies));
        Collections.addAll(this.dependencies, dependencies);
    }

    public ApplicationComponent(String name, Class<O> componentType, boolean requireNotNull, Class[] hints) {
        this(name, componentType, hints, ApplicationComponentSupplier.createFromServiceLoader(componentType, requireNotNull));
    }

    public ApplicationComponent(String name, Class<O> componentType, Class[] hints, ApplicationComponentSupplier<O> supplier) {
        this.name = name;
        this.componentType = componentType;
        this.dependencies = new LinkedList<>();
        this.hints = new LinkedList<>();
        Collections.addAll(this.hints, hints);
        this.supplier = supplier;
    }

    @Override
    public O get() {
        return supplier.get();
    }

    public void clear() {
        supplier.clear();
    }

    public Class<O> getComponentType() {
        return componentType;
    }

    public String getName() {
        return name;
    }

    public void setValue(O value) {
        supplier.setSupplier(() -> value);
    }

    @Override
    public void close() throws IOException {
        log.info("Closing component: " + this);
        if (supplier.withValue()) {
            if (Closeable.class.isAssignableFrom(componentType)) {
                ((Closeable) supplier.get()).close();
            }
            supplier.clear();
        }
    }

    public boolean withDependencies() {
        return !getDependencies().isEmpty();
    }

    public List<Class> getDependencies() {
        return dependencies;
    }

    public void addExtraHints(Class<?>... hints) {
        Collections.addAll(this.hints, hints);
    }

    public List<Class> getHints() {
        return hints;
    }

    public String toString() {
        return String.format("%s - %s", name, componentType.getName());
    }
}
