/*
 * Copyright (c) 2025 Typeflows
 * This is beta software under the Typeflows Beta License.
 * See LICENSE file for details.
 *
 * For more information, visit https://typeflows.io/license
 */

package io.typeflows.gradle.tasks

import io.typeflows.plugin.TypeflowsRunner
import org.gradle.api.DefaultTask
import org.gradle.api.file.ConfigurableFileCollection
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.provider.Property
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.OutputDirectory
import org.gradle.api.tasks.PathSensitive
import org.gradle.api.tasks.PathSensitivity.RELATIVE
import org.gradle.api.tasks.TaskAction
import org.gradle.api.tasks.TaskExecutionException

abstract class TypeflowsExportTask : DefaultTask() {
    @get:OutputDirectory
    abstract val targetDirectory: DirectoryProperty

    @get:InputFiles
    @get:PathSensitive(RELATIVE)
    abstract val classpath: ConfigurableFileCollection

    @get:InputFiles
    @get:PathSensitive(RELATIVE)
    abstract val scanDirectories: ConfigurableFileCollection

    @get:Input
    abstract val typeflowsClass: Property<String>

    @TaskAction
    fun typeflowsExport() {
        val outputDirectory = targetDirectory.get().asFile

        TypeflowsRunner.execute(
            classpathElements = classpath.files.map { it.absolutePath },
            typeflowsClassName = typeflowsClass.get(),
            outputDirectory = outputDirectory,
            onSuccess = {
                logger.lifecycle("✅ Generated Typeflows from ${typeflowsClass.get()} into ${outputDirectory.absolutePath}")
            },
            onFailure = { exception ->
                logger.lifecycle("❌ Failed to generate Typeflows from ${typeflowsClass.get()} (${exception.javaClass.simpleName}): ${exception.message}")
                logger.lifecycle("Check that the class exists, compiles correctly, and implements Builder<TypeflowsRepo>")
                throw TaskExecutionException(this, exception)
            }
        )
    }
}
