/*
 * Copyright (c) 2025 Typeflows
 * This is beta software under the Typeflows Beta License.
 * See LICENSE file for details.
 *
 * For more information, visit https://typeflows.io/license
 */

package io.typeflows.gradle

import io.typeflows.gradle.tasks.InstallClaudeTask
import io.typeflows.gradle.tasks.InstallCopilotTask
import io.typeflows.gradle.tasks.InstallCursorTask
import io.typeflows.gradle.tasks.TypeflowsExportTask
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.plugins.JavaPluginExtension
import org.gradle.api.tasks.TaskContainer
import org.gradle.kotlin.dsl.create
import org.gradle.kotlin.dsl.register
import org.gradle.kotlin.dsl.the

class TypeflowsPlugin : Plugin<Project> {
    override fun apply(project: Project) {
        val extension = project.extensions.create<TypeflowsExtension>("typeflows").apply {
            targetDirectory.convention(project.layout.projectDirectory.dir("."))
            claudeDirectory.convention(project.layout.projectDirectory.dir(".claude/commands"))
            copilotDirectory.convention(project.layout.projectDirectory.dir(".github"))
            cursorDirectory.convention(project.layout.projectDirectory.dir(".cursor"))
            typeflowsClass.convention("Typeflows")
        }

        val tasks = project.tasks
        project.plugins.withId("java") {
            project.the<JavaPluginExtension>().sourceSets.create("typeflows")

            tasks.addExport(extension, project)
            tasks.addInstallClaude(extension, project)
            tasks.addInstallCopilot(extension, project)
            tasks.addInstallCursor(extension, project)

            tasks.named("assemble") {
                it.dependsOn("typeflowsExport")
            }
        }
    }
}


private fun TaskContainer.addExport(extension: TypeflowsExtension, project: Project) {
    register<TypeflowsExportTask>("typeflowsExport") {
        group = "typeflows"
        description = "Export Typeflows implementation to your repository directory"

        targetDirectory.set(extension.targetDirectory)
        typeflowsClass.set(extension.typeflowsClass)

        val javaExt = project.the<JavaPluginExtension>()
        val typeflowsSourceSet = javaExt.sourceSets.getByName("typeflows")

        scanDirectories.from(typeflowsSourceSet.output.classesDirs)
        classpath.from(typeflowsSourceSet.runtimeClasspath)
        dependsOn("typeflowsClasses")
    }
}

private fun TaskContainer.addInstallCopilot(extension: TypeflowsExtension, project: Project) {
    register<InstallCopilotTask>("typeflowsInstallCopilot") {
        group = "typeflows"
        description = "Install GitHub Copilot instructions from classpath to .github directory"

        copilotDirectory.set(extension.copilotDirectory)

        val javaExt = project.the<JavaPluginExtension>()
        val typeflowsSourceSet = javaExt.sourceSets.getByName("typeflows")

        classpath.from(typeflowsSourceSet.runtimeClasspath)
        dependsOn("typeflowsClasses")
    }
}

private fun TaskContainer.addInstallCursor(extension: TypeflowsExtension, project: Project) {
    register<InstallCursorTask>("typeflowsInstallCursor") {
        group = "typeflows"
        description = "Install Cursor rules from classpath to .cursor directory"

        cursorDirectory.set(extension.cursorDirectory)

        val javaExt = project.the<JavaPluginExtension>()
        val typeflowsSourceSet = javaExt.sourceSets.getByName("typeflows")

        classpath.from(typeflowsSourceSet.runtimeClasspath)
        dependsOn("typeflowsClasses")
    }
}

private fun TaskContainer.addInstallClaude(extension: TypeflowsExtension, project: Project) {
    register<InstallClaudeTask>("typeflowsInstallClaude") {
        group = "typeflows"
        description = "Install Claude commands from classpath to .claude directory"

        claudeDirectory.set(extension.claudeDirectory)

        val javaExt = project.the<JavaPluginExtension>()
        val typeflowsSourceSet = javaExt.sourceSets.getByName("typeflows")

        classpath.from(typeflowsSourceSet.runtimeClasspath)
        dependsOn("typeflowsClasses")
    }
}
