package io.typeflows.gradle.tasks

import io.typeflows.gradle.ClasspathExecutor
import org.gradle.api.DefaultTask
import org.gradle.api.file.ConfigurableFileCollection
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.Internal
import org.gradle.api.tasks.PathSensitive
import org.gradle.api.tasks.PathSensitivity.RELATIVE
import java.io.File
import java.util.jar.JarFile

abstract class InstallTask : DefaultTask() {
    @get:InputFiles
    @get:PathSensitive(RELATIVE)
    abstract val classpath: ConfigurableFileCollection

    @Internal
    protected abstract fun getTargetDirectory(): DirectoryProperty

    protected fun executeInstall() {
        val targetDir = getTargetDirectory().get().asFile.apply { mkdirs() }

        ClasspathExecutor.withClasspath(classpath) { classLoader ->
            var extracted = false

            extracted = extractFromPluginClasspath(targetDir) || extracted

            classpath.files
                .filter { it.extension == "jar" }
                .forEach { file -> extracted = extractFromJar(file, targetDir) || extracted }

            if (extracted) {
                processExtractedFiles(targetDir)
                logSuccess(targetDir)
            } else
                logger.lifecycle("${YELLOW}⚠️ No LLM instructions found${RESET}")
        }
    }

    protected open fun extractFromPluginClasspath(outputDir: File): Boolean {
        val pluginClassLoader = this::class.java.classLoader
        val instructionsResource = pluginClassLoader.getResource("llm")

        if (instructionsResource != null && instructionsResource.protocol == "jar") {
            val jarPath = instructionsResource.path.substringBefore("!/llm")
            if (jarPath.startsWith("file:")) {
                return extractFromJar(File(jarPath.removePrefix("file:")), outputDir)
            }
        }

        return false
    }

    protected open fun extractFromJar(jarFile: File, outputDir: File): Boolean {
        var extracted = false

        JarFile(jarFile).use { jar ->
            val instructionEntries = jar.entries().asSequence()
                .filter { it.name.startsWith("llm/") && !it.isDirectory }
                .toList()

            if (instructionEntries.isEmpty()) return false

            instructionEntries.forEach { entry ->
                val targetFile = File(outputDir, entry.name.removePrefix("llm/")).apply { parentFile?.mkdirs() }

                jar.getInputStream(entry).use { input -> targetFile.outputStream().use { input.copyTo(it) } }
                extracted = true
            }
        }

        return extracted
    }

    protected abstract fun processExtractedFiles(outputDir: File)
    protected abstract fun logSuccess(outputDir: File)
}
