package io.typeflows.gradle.tasks

import org.gradle.api.file.DirectoryProperty
import org.gradle.api.tasks.OutputDirectory
import org.gradle.api.tasks.TaskAction
import java.io.File
import java.util.Locale.getDefault

abstract class InstallCursorTask : InstallTask() {
    @get:OutputDirectory
    abstract val cursorDirectory: DirectoryProperty

    @TaskAction
    fun typeflowsInstallCursor() {
        executeInstall()
    }

    override fun getTargetDirectory() = cursorDirectory

    override fun processExtractedFiles(outputDir: File) {
        val rulesDir = File(outputDir.parent, ".cursor/rules").apply { mkdirs() }

        outputDir.listFiles { it.extension == "md" }?.forEach {
            transformToMdcFile(it, rulesDir)
            it.delete()
        }
    }

    private fun transformToMdcFile(sourceFile: File, rulesDir: File) {
        File(rulesDir, "${sourceFile.nameWithoutExtension}.mdc").writeText(
            transformContentToMdc(
                sourceFile.readText(),
                sourceFile.nameWithoutExtension
            )
        )
    }

    private fun transformContentToMdc(content: String, name: String) = """---
description: Typeflows rules for ${name.replaceFirstChar { if (it.isLowerCase()) it.titlecase(getDefault()) else it.toString() }} workflows and GitHub Actions
globs:
  - "**/*.kt"
  - "**/*.java"
  - "src/main/kotlin/**/*"
  - "src/main/java/**/*"
  - ".github/workflows/**/*.yml"
  - ".github/workflows/**/*.yaml"
alwaysApply: false
---

# Typeflows ${name.replaceFirstChar { if (it.isLowerCase()) it.titlecase(getDefault()) else it.toString() }} Rules

$content

## Cursor Integration Notes
These rules help Cursor AI understand Typeflows workflow patterns and generate appropriate code for GitHub Actions automation.
"""

    override fun logSuccess(outputDir: File) {
        logger.lifecycle(
            "${GREEN}✅ Cursor rules installed:${RESET} ${CYAN}.cursor/rules/${RESET}"
        )
    }
}
