package io.typeflows.gradle.tasks

import org.gradle.api.file.DirectoryProperty
import org.gradle.api.tasks.OutputDirectory
import org.gradle.api.tasks.TaskAction
import java.io.File

abstract class InstallCopilotTask : InstallTask() {
    @get:OutputDirectory
    abstract val copilotDirectory: DirectoryProperty

    @TaskAction
    fun typeflowsInstallCopilot() {
        executeInstall()
    }

    override fun getTargetDirectory() = copilotDirectory

    override fun processExtractedFiles(outputDir: File) {
        val instructionsDir = File(outputDir.parent, ".github/instructions").apply { mkdirs() }
        val promptsDir = File(outputDir.parent, ".github/prompts").apply { mkdirs() }

        outputDir.listFiles { it.extension == "md" }?.forEach {
            when {
                it.name.contains("import") -> transformToInstructionFile(it, instructionsDir)
                it.name.contains("doctor") -> transformToPromptFile(it, promptsDir)
                else -> transformToInstructionFile(it, instructionsDir)
            }
            it.delete()
        }
    }

    private fun transformToInstructionFile(sourceFile: File, instructionsDir: File) {
        val content = sourceFile.readText()
        val transformedContent = transformContentForInstructions(content)

        File(instructionsDir, "${sourceFile.nameWithoutExtension}.instructions.md").writeText(
            transformedContent
        )
    }

    private fun transformToPromptFile(sourceFile: File, promptsDir: File) {
        val content = sourceFile.readText()
        val transformedContent = transformContentForPrompts(content)

        val targetFile = File(promptsDir, "${sourceFile.nameWithoutExtension}.prompt.md")
        targetFile.writeText(transformedContent)
    }

    private fun transformContentForInstructions(content: String) = """---
description: Typeflows GitHub Actions conversion instructions for Kotlin/Java workflow builders
globs:
  - "**/*.kt"
  - "**/*.java"
  - "src/main/kotlin/**/*"
  - "src/main/java/**/*"
  - ".github/workflows/**/*.yml"
  - ".github/workflows/**/*.yaml"
author: Typeflows
version: 1.0
---

# Typeflows Instructions

$content

## Usage in Copilot
These instructions help GitHub Copilot understand how to work with Typeflows for GitHub Actions workflow generation.
"""

    private fun transformContentForPrompts(content: String): String {
        return """---
mode: 'agent'
description: Typeflows analysis and optimization prompt for GitHub Actions workflows
tools:
  - 'codebase'
  - 'githubRepo'
author: Typeflows
version: 1.0
---

$content

## Prompt Usage
Use this prompt template for analyzing and improving Typeflows-generated workflows.
"""
    }

    override fun logSuccess(outputDir: File) {
        logger.lifecycle(
            "${GREEN}✅ Copilot instructions installed:${RESET} ${CYAN}.github/instructions/ and .github/prompts/${RESET}"
        )
    }
}
