package io.typeflows.gradle.tasks

import io.typeflows.TypeflowsRepo
import io.typeflows.gradle.ClasspathExecutor
import io.typeflows.util.Builder
import org.gradle.api.DefaultTask
import org.gradle.api.file.ConfigurableFileCollection
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.provider.Property
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.OutputDirectory
import org.gradle.api.tasks.PathSensitive
import org.gradle.api.tasks.PathSensitivity.RELATIVE
import org.gradle.api.tasks.TaskAction
import org.gradle.api.tasks.TaskExecutionException

abstract class TypeflowsExportTask : DefaultTask() {
    @get:OutputDirectory
    abstract val targetDirectory: DirectoryProperty

    @get:InputFiles
    @get:PathSensitive(RELATIVE)
    abstract val classpath: ConfigurableFileCollection

    @get:InputFiles
    @get:PathSensitive(RELATIVE)
    abstract val scanDirectories: ConfigurableFileCollection

    @get:Input
    abstract val typeflowsClass: Property<String>

    @TaskAction
    fun typeflowsExport() {
        val outputDirectory = targetDirectory.get().asFile.apply { mkdirs() }

        ClasspathExecutor.withClasspath(classpath) { classLoader ->
            val clazz = typeflowsClass.get()
            try {
                @Suppress("UNCHECKED_CAST")
                val builder = classLoader.loadClass(clazz).getDeclaredConstructor().newInstance() as Builder<TypeflowsRepo>
                builder.build().writeTo(outputDirectory)

                logger.lifecycle(
                    "${GREEN}✅ Generated Typeflows from ${builder::class.simpleName}${RESET} ${CYAN}into${RESET} ${CYAN}${
                        outputDirectory.relativeTo(project.rootDir)
                    }${RESET}"
                )
            } catch (e: Exception) {
                logger.lifecycle(
                    "${YELLOW}❌ Could not instantiate Typeflows from Builder<TypeflowsRepo> instance of ${clazz}${RESET} ${CYAN}into${RESET} ${CYAN}${
                        outputDirectory.relativeTo(project.rootDir)
                    }${RESET}",
                )
                throw TaskExecutionException(this, e)
            }
        }
    }
}
