/*
 * Copyright (c) 2025 Incerto Group Ltd
 * This is beta software under the Typeflows Beta License.
 * See LICENSE file for details.
 *
 * For more information, visit https://typeflows.io/license
 */

package io.typeflows.gradle

import io.typeflows.gradle.task.InstallClaudeTask
import io.typeflows.gradle.task.InstallCopilotTask
import io.typeflows.gradle.task.InstallCursorTask
import io.typeflows.gradle.task.InstallJunieTask
import io.typeflows.gradle.task.TypeflowsExportTask
import org.gradle.api.Plugin
import org.gradle.api.Project
import org.gradle.api.plugins.JavaPluginExtension
import org.gradle.api.tasks.TaskContainer
import org.gradle.api.tasks.testing.Test
import org.gradle.kotlin.dsl.create
import org.gradle.kotlin.dsl.register
import org.gradle.kotlin.dsl.the

class TypeflowsPlugin : Plugin<Project> {
    override fun apply(project: Project) {
        val extension = project.extensions.create<TypeflowsExtension>("typeflows").apply {
            targetDirectory.convention(project.layout.projectDirectory.dir("."))
            typeflowsClass.convention("Typeflows")
        }

        val tasks = project.tasks
        project.plugins.withId("java") {
            val javaExt = project.the<JavaPluginExtension>()
            val typeflowsSourceSet = javaExt.sourceSets.create("typeflows")

            val typeflowsTestSourceSet = javaExt.sourceSets.create("typeflowsTest") {
                it.compileClasspath += typeflowsSourceSet.output + typeflowsSourceSet.compileClasspath
                it.runtimeClasspath += typeflowsSourceSet.output + typeflowsSourceSet.runtimeClasspath
            }

            project.configurations.getByName("typeflowsTestImplementation").extendsFrom(
                project.configurations.getByName("testImplementation")
            )
            project.configurations.getByName("typeflowsTestRuntimeOnly").extendsFrom(
                project.configurations.getByName("testRuntimeOnly")
            )

            tasks.register<Test>("typeflowsTest") {
                description = "Runs the Typeflows tests"
                group = "verification"

                testClassesDirs = typeflowsTestSourceSet.output.classesDirs
                classpath = typeflowsTestSourceSet.runtimeClasspath

                shouldRunAfter("test")

                useJUnitPlatform()
            }


            tasks.addExport(extension, project)
            tasks.addInstallClaude(extension, project)
            tasks.addInstallCopilot(extension, project)
            tasks.addInstallCursor(extension, project)
            tasks.addInstallJunie(extension, project)

            tasks.named("check") {
                it.dependsOn("typeflowsClasses", "typeflowsTest")
            }

            tasks.named("assemble") {
                it.dependsOn("typeflowsExport")
            }

        }
    }
}


private fun TaskContainer.addExport(extension: TypeflowsExtension, project: Project) {
    register<TypeflowsExportTask>("typeflowsExport") {
        group = "typeflows"
        description = "Export Typeflows implementation to your repository directory"

        targetDirectory.set(extension.targetDirectory)
        typeflowsClass.set(extension.typeflowsClass)

        val javaExt = project.the<JavaPluginExtension>()
        val typeflowsSourceSet = javaExt.sourceSets.getByName("typeflows")

        scanDirectories.from(typeflowsSourceSet.output.classesDirs)
        classpath.from(typeflowsSourceSet.runtimeClasspath)
        dependsOn("typeflowsClasses")
    }
}

private fun TaskContainer.addInstallCopilot(extension: TypeflowsExtension, project: Project) {
    register<InstallCopilotTask>("typeflowsInstallCopilot") {
        group = "typeflows"
        description = "Install GitHub Copilot instructions from classpath to .github directory"

        targetDirectory.set(extension.targetDirectory)

        val javaExt = project.the<JavaPluginExtension>()
        val typeflowsSourceSet = javaExt.sourceSets.getByName("typeflows")

        classpath.from(typeflowsSourceSet.runtimeClasspath)
        dependsOn("typeflowsClasses")
    }
}

private fun TaskContainer.addInstallCursor(extension: TypeflowsExtension, project: Project) {
    register<InstallCursorTask>("typeflowsInstallCursor") {
        group = "typeflows"
        description = "Install Cursor rules from classpath to .cursor directory"

        targetDirectory.set(extension.targetDirectory)

        val javaExt = project.the<JavaPluginExtension>()
        val typeflowsSourceSet = javaExt.sourceSets.getByName("typeflows")

        classpath.from(typeflowsSourceSet.runtimeClasspath)
        dependsOn("typeflowsClasses")
    }
}

private fun TaskContainer.addInstallJunie(extension: TypeflowsExtension, project: Project) {
    register<InstallJunieTask>("typeflowsInstallJunie") {
        group = "typeflows"
        description = "Install Junie rules from classpath to .cursor directory"

        targetDirectory.set(extension.targetDirectory)

        val javaExt = project.the<JavaPluginExtension>()
        val typeflowsSourceSet = javaExt.sourceSets.getByName("typeflows")

        classpath.from(typeflowsSourceSet.runtimeClasspath)
        dependsOn("typeflowsClasses")
    }
}

private fun TaskContainer.addInstallClaude(extension: TypeflowsExtension, project: Project) {
    register<InstallClaudeTask>("typeflowsInstallClaude") {
        group = "typeflows"
        description = "Install Claude commands from classpath to .claude directory"

        targetDirectory.set(extension.targetDirectory)

        val javaExt = project.the<JavaPluginExtension>()
        val typeflowsSourceSet = javaExt.sourceSets.getByName("typeflows")

        classpath.from(typeflowsSourceSet.runtimeClasspath)
        dependsOn("typeflowsClasses")
    }
}
