package io.typeflows.gradle

import io.typeflows.github.WorkflowBuilder
import java.io.File
import java.lang.reflect.Modifier.isAbstract

class WorkflowScanner(private val classLoader: ClassLoader) {
    fun findWorkflowBuilders(classDirectories: Set<File>): List<Class<*>> {
        val classNames = mutableSetOf<String>()

        classDirectories.forEach { directory ->
            if (directory.isDirectory) {
                scanDirectory(directory, classNames)
            }
        }

        return classNames
            .mapNotNull {
                val clazz = classLoader.loadClass(it)
                when {
                    clazz.isWorkflowBuilder() && !clazz.isInterface && !isAbstract(clazz.modifiers) -> clazz
                    else -> null
                }
            }
    }

    private fun scanDirectory(directory: File, classNames: MutableSet<String>) {
        directory.walkTopDown()
            .filter { it.extension == "class" }
            .filter { !it.name.equals("module-info.class", ignoreCase = true) }
            .forEach {
                classNames.add(
                    it.relativeTo(directory).path
                        .removeSuffix(".class")
                        .replace(File.separatorChar, '.')
                )
            }
    }

    private fun Class<*>.isWorkflowBuilder() = try {
        classLoader.loadClass(WorkflowBuilder::class.qualifiedName).isAssignableFrom(this)
    } catch (_: Exception) {
        false
    }
}
