package io.typeflows.gradle

import io.typeflows.github.format.Yaml
import io.typeflows.github.util.Renderable
import org.gradle.api.DefaultTask
import org.gradle.api.file.ConfigurableFileCollection
import org.gradle.api.file.DirectoryProperty
import org.gradle.api.provider.Property
import org.gradle.api.tasks.Input
import org.gradle.api.tasks.InputFiles
import org.gradle.api.tasks.OutputDirectory
import org.gradle.api.tasks.PathSensitive
import org.gradle.api.tasks.PathSensitivity.RELATIVE
import org.gradle.api.tasks.TaskAction
import java.io.File
import java.net.URLClassLoader

abstract class ExportWorkflowsTask : DefaultTask() {
    @get:OutputDirectory
    abstract val outputDir: DirectoryProperty

    @get:Input
    abstract val fileNamePattern: Property<String>

    @get:InputFiles
    @get:PathSensitive(RELATIVE)
    abstract val classpath: ConfigurableFileCollection

    @get:InputFiles
    @get:PathSensitive(RELATIVE)
    abstract val scanDirectories: ConfigurableFileCollection

    @TaskAction
    fun exportWorkflows() {
        val outputDirectory = outputDir.get().asFile.apply { mkdirs() }

        val urls = classpath.files.map { it.toURI().toURL() }.toTypedArray()
        val classLoader = URLClassLoader(urls, this::class.java.classLoader)

        WorkflowScanner(classLoader)
            .findWorkflowBuilders(scanDirectories.files)
            .forEach { builder ->
                val obj = builder.getDeclaredConstructor().newInstance()

                val originalClassLoader = Thread.currentThread().contextClassLoader
                try {
                    Thread.currentThread().contextClassLoader = classLoader

                    @Suppress("UNCHECKED_CAST")
                    (builder.getMethod("toWorkflows")(obj)

                        as List<Renderable<*>>).forEach {
                        val fileName = fileNamePattern.get().replace("{name}", obj::class.simpleName!!)

                        val outputFile = File(outputDirectory, fileName).apply {
                            writeText(Yaml.asFormatString(it.render()))
                        }

                        logger.lifecycle("Generated workflow: ${outputFile.path}")
                    }
                } finally {
                    Thread.currentThread().contextClassLoader = originalClassLoader
                }
            }
    }
}
