/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.bookkeeper.statelib.impl.mvcc;

import io.netty.buffer.ByteBuf;
import io.netty.util.Recycler;
import io.netty.util.ReferenceCountUtil;
import java.util.function.Predicate;
import org.apache.bookkeeper.api.kv.impl.result.KeyValueFactory;
import org.apache.bookkeeper.api.kv.impl.result.KeyValueImpl;
import org.apache.bookkeeper.api.kv.options.RangeOption;
import org.apache.bookkeeper.common.coder.Coder;
import org.apache.bookkeeper.common.util.Recycled;
import org.apache.bookkeeper.stream.proto.kv.store.ValueType;

/**
 * An object represents the mvcc metdata and value for a given key.
 */
public class MVCCRecord implements Recycled, Predicate<RangeOption<?>> {
    public static MVCCRecord newRecord() {
        return RECYCLER.get();
    }

    private static final Recycler<MVCCRecord> RECYCLER = new Recycler<MVCCRecord>() {
        @Override
        protected MVCCRecord newObject(Handle<MVCCRecord> handle) {
            return new MVCCRecord(handle);
        }
    };
    private ByteBuf value;
    private long number;
    private final Recycler.Handle<MVCCRecord> handle;
    private long createRev;
    private long modRev;
    private long version;
    private ValueType valueType = ValueType.BYTES;
    private long expireTime;

    private MVCCRecord(Recycler.Handle<MVCCRecord> handle) {
        this.handle = handle;
    }

    public MVCCRecord duplicate() {
        MVCCRecord record = newRecord();
        record.createRev = createRev;
        record.modRev = modRev;
        record.version = version;
        record.valueType = valueType;
        record.value = value.retainedSlice();
        record.number = number;
        record.expireTime = expireTime;
        return record;
    }

    public int compareModRev(long revision) {
        return Long.compare(modRev, revision);
    }

    public int compareCreateRev(long revision) {
        return Long.compare(createRev, revision);
    }

    public int compareVersion(long version) {
        return Long.compare(this.version, version);
    }

    public void setValue(ByteBuf buf, ValueType valueType) {
        if (null != value) {
            ReferenceCountUtil.release(value);
        }
        this.value = buf;
        this.valueType = valueType;
        if (ValueType.NUMBER == valueType) {
            this.number = buf.getLong(0);
        }
    }

    public boolean expired() {
        return expireTime != 0 && expireTime > System.currentTimeMillis();
    }

    private void reset() {
        if (null != value) {
            ReferenceCountUtil.release(value);
            value = null;
        }
        modRev = -1L;
        createRev = -1L;
        version = -1L;
        number = -1L;
        valueType = ValueType.BYTES;
    }

    @Override
    public void recycle() {
        reset();
        handle.recycle(this);
    }

    <K, V> KeyValueImpl<K, V> asKVRecord(KeyValueFactory<K, V> recordFactory, K key, Coder<V> valCoder) {
        return recordFactory.newKv().key(key).value(valCoder.decode(value)).createRevision(createRev).modifiedRevision(modRev).version(version).isNumber(ValueType.NUMBER == valueType).numberValue(number);
    }

    @Override
    public boolean test(RangeOption<?> rangeOption) {
        if (null == rangeOption) {
            return true;
        }
        return createRev >= rangeOption.minCreateRev() && createRev <= rangeOption.maxCreateRev() && modRev >= rangeOption.minModRev() && modRev <= rangeOption.maxModRev();
    }

    @Override
    public boolean equals(final Object o) {
        if (o == this) return true;
        if (!(o instanceof MVCCRecord)) return false;
        final MVCCRecord other = (MVCCRecord) o;
        if (!other.canEqual((Object) this)) return false;
        if (this.getNumber() != other.getNumber()) return false;
        if (this.getCreateRev() != other.getCreateRev()) return false;
        if (this.getModRev() != other.getModRev()) return false;
        if (this.getVersion() != other.getVersion()) return false;
        if (this.getExpireTime() != other.getExpireTime()) return false;
        final Object this$value = this.getValue();
        final Object other$value = other.getValue();
        if (this$value == null ? other$value != null : !this$value.equals(other$value)) return false;
        final Object this$handle = this.getHandle();
        final Object other$handle = other.getHandle();
        if (this$handle == null ? other$handle != null : !this$handle.equals(other$handle)) return false;
        final Object this$valueType = this.getValueType();
        final Object other$valueType = other.getValueType();
        if (this$valueType == null ? other$valueType != null : !this$valueType.equals(other$valueType)) return false;
        return true;
    }

    protected boolean canEqual(final Object other) {
        return other instanceof MVCCRecord;
    }

    @Override
    public int hashCode() {
        final int PRIME = 59;
        int result = 1;
        final long $number = this.getNumber();
        result = result * PRIME + (int) ($number >>> 32 ^ $number);
        final long $createRev = this.getCreateRev();
        result = result * PRIME + (int) ($createRev >>> 32 ^ $createRev);
        final long $modRev = this.getModRev();
        result = result * PRIME + (int) ($modRev >>> 32 ^ $modRev);
        final long $version = this.getVersion();
        result = result * PRIME + (int) ($version >>> 32 ^ $version);
        final long $expireTime = this.getExpireTime();
        result = result * PRIME + (int) ($expireTime >>> 32 ^ $expireTime);
        final Object $value = this.getValue();
        result = result * PRIME + ($value == null ? 43 : $value.hashCode());
        final Object $handle = this.getHandle();
        result = result * PRIME + ($handle == null ? 43 : $handle.hashCode());
        final Object $valueType = this.getValueType();
        result = result * PRIME + ($valueType == null ? 43 : $valueType.hashCode());
        return result;
    }

    @Override
    public String toString() {
        return "MVCCRecord(value=" + this.getValue() + ", number=" + this.getNumber() + ", handle=" + this.getHandle() + ", createRev=" + this.getCreateRev() + ", modRev=" + this.getModRev() + ", version=" + this.getVersion() + ", valueType=" + this.getValueType() + ", expireTime=" + this.getExpireTime() + ")";
    }

    public void setValue(final ByteBuf value) {
        this.value = value;
    }

    public void setNumber(final long number) {
        this.number = number;
    }

    public void setCreateRev(final long createRev) {
        this.createRev = createRev;
    }

    public void setModRev(final long modRev) {
        this.modRev = modRev;
    }

    public void setVersion(final long version) {
        this.version = version;
    }

    public void setValueType(final ValueType valueType) {
        this.valueType = valueType;
    }

    public void setExpireTime(final long expireTime) {
        this.expireTime = expireTime;
    }

    public ByteBuf getValue() {
        return this.value;
    }

    public long getNumber() {
        return this.number;
    }

    public Recycler.Handle<MVCCRecord> getHandle() {
        return this.handle;
    }

    public long getCreateRev() {
        return this.createRev;
    }

    public long getModRev() {
        return this.modRev;
    }

    public long getVersion() {
        return this.version;
    }

    public ValueType getValueType() {
        return this.valueType;
    }

    public long getExpireTime() {
        return this.expireTime;
    }
}
