/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.bookkeeper.statelib.api;

import java.io.File;
import java.time.Duration;
import java.util.Map;
import java.util.concurrent.ScheduledExecutorService;
import lombok.Builder.Default;
import org.apache.bookkeeper.common.coder.Coder;
import org.apache.bookkeeper.statelib.api.checkpoint.CheckpointStore;

/**
 * Specification for a state store.
 */
public class StateStoreSpec {
    private String name;
    private Coder<?> keyCoder;
    private Coder<?> valCoder;
    private File localStateStoreDir;
    private String stream;
    private boolean isReadonly;
    private ScheduledExecutorService writeIOScheduler;
    private ScheduledExecutorService readIOScheduler;
    private ScheduledExecutorService checkpointIOScheduler;
    private CheckpointStore checkpointStore;
    private Duration checkpointDuration;
    private Map<String, Object> configs;
    private boolean checkpointChecksumEnable;
    private boolean checkpointChecksumCompatible;
    private boolean localStorageCleanupEnable;
    // Max idle time while waiting to restore from a checkpoint.
    private Duration checkpointRestoreIdleLimit;
    private int ttlSeconds;

    private static Duration $default$checkpointDuration() {
        return Duration.ofMinutes(1);
    }

    private static boolean $default$checkpointChecksumEnable() {
        return true;
    }

    private static boolean $default$checkpointChecksumCompatible() {
        return true;
    }

    private static boolean $default$localStorageCleanupEnable() {
        return false;
    }

    private static Duration $default$checkpointRestoreIdleLimit() {
        return Duration.ofMinutes(5);
    }

    private static int $default$ttlSeconds() {
        return 0;
    }

    StateStoreSpec(final String name, final Coder<?> keyCoder, final Coder<?> valCoder, final File localStateStoreDir, final String stream, final boolean isReadonly, final ScheduledExecutorService writeIOScheduler, final ScheduledExecutorService readIOScheduler, final ScheduledExecutorService checkpointIOScheduler, final CheckpointStore checkpointStore, final Duration checkpointDuration, final Map<String, Object> configs, final boolean checkpointChecksumEnable, final boolean checkpointChecksumCompatible, final boolean localStorageCleanupEnable, final Duration checkpointRestoreIdleLimit, final int ttlSeconds) {
        this.name = name;
        this.keyCoder = keyCoder;
        this.valCoder = valCoder;
        this.localStateStoreDir = localStateStoreDir;
        this.stream = stream;
        this.isReadonly = isReadonly;
        this.writeIOScheduler = writeIOScheduler;
        this.readIOScheduler = readIOScheduler;
        this.checkpointIOScheduler = checkpointIOScheduler;
        this.checkpointStore = checkpointStore;
        this.checkpointDuration = checkpointDuration;
        this.configs = configs;
        this.checkpointChecksumEnable = checkpointChecksumEnable;
        this.checkpointChecksumCompatible = checkpointChecksumCompatible;
        this.localStorageCleanupEnable = localStorageCleanupEnable;
        this.checkpointRestoreIdleLimit = checkpointRestoreIdleLimit;
        this.ttlSeconds = ttlSeconds;
    }


    public static class StateStoreSpecBuilder {
        private String name;
        private Coder<?> keyCoder;
        private Coder<?> valCoder;
        private File localStateStoreDir;
        private String stream;
        private boolean isReadonly;
        private ScheduledExecutorService writeIOScheduler;
        private ScheduledExecutorService readIOScheduler;
        private ScheduledExecutorService checkpointIOScheduler;
        private CheckpointStore checkpointStore;
        private boolean checkpointDuration$set;
        private Duration checkpointDuration$value;
        private java.util.ArrayList<String> configs$key;
        private java.util.ArrayList<Object> configs$value;
        private boolean checkpointChecksumEnable$set;
        private boolean checkpointChecksumEnable$value;
        private boolean checkpointChecksumCompatible$set;
        private boolean checkpointChecksumCompatible$value;
        private boolean localStorageCleanupEnable$set;
        private boolean localStorageCleanupEnable$value;
        private boolean checkpointRestoreIdleLimit$set;
        private Duration checkpointRestoreIdleLimit$value;
        private boolean ttlSeconds$set;
        private int ttlSeconds$value;

        StateStoreSpecBuilder() {
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder name(final String name) {
            this.name = name;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder keyCoder(final Coder<?> keyCoder) {
            this.keyCoder = keyCoder;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder valCoder(final Coder<?> valCoder) {
            this.valCoder = valCoder;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder localStateStoreDir(final File localStateStoreDir) {
            this.localStateStoreDir = localStateStoreDir;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder stream(final String stream) {
            this.stream = stream;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder isReadonly(final boolean isReadonly) {
            this.isReadonly = isReadonly;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder writeIOScheduler(final ScheduledExecutorService writeIOScheduler) {
            this.writeIOScheduler = writeIOScheduler;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder readIOScheduler(final ScheduledExecutorService readIOScheduler) {
            this.readIOScheduler = readIOScheduler;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder checkpointIOScheduler(final ScheduledExecutorService checkpointIOScheduler) {
            this.checkpointIOScheduler = checkpointIOScheduler;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder checkpointStore(final CheckpointStore checkpointStore) {
            this.checkpointStore = checkpointStore;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder checkpointDuration(final Duration checkpointDuration) {
            this.checkpointDuration$value = checkpointDuration;
            checkpointDuration$set = true;
            return this;
        }

        public StateStoreSpec.StateStoreSpecBuilder config(final String configKey, final Object configValue) {
            if (this.configs$key == null) {
                this.configs$key = new java.util.ArrayList<String>();
                this.configs$value = new java.util.ArrayList<Object>();
            }
            this.configs$key.add(configKey);
            this.configs$value.add(configValue);
            return this;
        }

        public StateStoreSpec.StateStoreSpecBuilder configs(final java.util.Map<? extends String, ? extends Object> configs) {
            if (configs == null) {
                throw new NullPointerException("configs cannot be null");
            }
            if (this.configs$key == null) {
                this.configs$key = new java.util.ArrayList<String>();
                this.configs$value = new java.util.ArrayList<Object>();
            }
            for (final java.util.Map.Entry<? extends String, ? extends Object> $lombokEntry : configs.entrySet()) {
                this.configs$key.add($lombokEntry.getKey());
                this.configs$value.add($lombokEntry.getValue());
            }
            return this;
        }

        public StateStoreSpec.StateStoreSpecBuilder clearConfigs() {
            if (this.configs$key != null) {
                this.configs$key.clear();
                this.configs$value.clear();
            }
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder checkpointChecksumEnable(final boolean checkpointChecksumEnable) {
            this.checkpointChecksumEnable$value = checkpointChecksumEnable;
            checkpointChecksumEnable$set = true;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder checkpointChecksumCompatible(final boolean checkpointChecksumCompatible) {
            this.checkpointChecksumCompatible$value = checkpointChecksumCompatible;
            checkpointChecksumCompatible$set = true;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder localStorageCleanupEnable(final boolean localStorageCleanupEnable) {
            this.localStorageCleanupEnable$value = localStorageCleanupEnable;
            localStorageCleanupEnable$set = true;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder checkpointRestoreIdleLimit(final Duration checkpointRestoreIdleLimit) {
            this.checkpointRestoreIdleLimit$value = checkpointRestoreIdleLimit;
            checkpointRestoreIdleLimit$set = true;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public StateStoreSpec.StateStoreSpecBuilder ttlSeconds(final int ttlSeconds) {
            this.ttlSeconds$value = ttlSeconds;
            ttlSeconds$set = true;
            return this;
        }

        public StateStoreSpec build() {
            java.util.Map<String, Object> configs;
            switch (this.configs$key == null ? 0 : this.configs$key.size()) {
            case 0: 
                configs = java.util.Collections.emptyMap();
                break;
            case 1: 
                configs = java.util.Collections.singletonMap(this.configs$key.get(0), this.configs$value.get(0));
                break;
            default: 
                configs = new java.util.LinkedHashMap<String, Object>(this.configs$key.size() < 1073741824 ? 1 + this.configs$key.size() + (this.configs$key.size() - 3) / 3 : Integer.MAX_VALUE);
                for (int $i = 0; $i < this.configs$key.size(); $i++) configs.put(this.configs$key.get($i), (Object) this.configs$value.get($i));
                configs = java.util.Collections.unmodifiableMap(configs);
            }
            Duration checkpointDuration$value = this.checkpointDuration$value;
            if (!this.checkpointDuration$set) checkpointDuration$value = StateStoreSpec.$default$checkpointDuration();
            boolean checkpointChecksumEnable$value = this.checkpointChecksumEnable$value;
            if (!this.checkpointChecksumEnable$set) checkpointChecksumEnable$value = StateStoreSpec.$default$checkpointChecksumEnable();
            boolean checkpointChecksumCompatible$value = this.checkpointChecksumCompatible$value;
            if (!this.checkpointChecksumCompatible$set) checkpointChecksumCompatible$value = StateStoreSpec.$default$checkpointChecksumCompatible();
            boolean localStorageCleanupEnable$value = this.localStorageCleanupEnable$value;
            if (!this.localStorageCleanupEnable$set) localStorageCleanupEnable$value = StateStoreSpec.$default$localStorageCleanupEnable();
            Duration checkpointRestoreIdleLimit$value = this.checkpointRestoreIdleLimit$value;
            if (!this.checkpointRestoreIdleLimit$set) checkpointRestoreIdleLimit$value = StateStoreSpec.$default$checkpointRestoreIdleLimit();
            int ttlSeconds$value = this.ttlSeconds$value;
            if (!this.ttlSeconds$set) ttlSeconds$value = StateStoreSpec.$default$ttlSeconds();
            return new StateStoreSpec(this.name, this.keyCoder, this.valCoder, this.localStateStoreDir, this.stream, this.isReadonly, this.writeIOScheduler, this.readIOScheduler, this.checkpointIOScheduler, this.checkpointStore, checkpointDuration$value, configs, checkpointChecksumEnable$value, checkpointChecksumCompatible$value, localStorageCleanupEnable$value, checkpointRestoreIdleLimit$value, ttlSeconds$value);
        }

        @Override
        public String toString() {
            return "StateStoreSpec.StateStoreSpecBuilder(name=" + this.name + ", keyCoder=" + this.keyCoder + ", valCoder=" + this.valCoder + ", localStateStoreDir=" + this.localStateStoreDir + ", stream=" + this.stream + ", isReadonly=" + this.isReadonly + ", writeIOScheduler=" + this.writeIOScheduler + ", readIOScheduler=" + this.readIOScheduler + ", checkpointIOScheduler=" + this.checkpointIOScheduler + ", checkpointStore=" + this.checkpointStore + ", checkpointDuration$value=" + this.checkpointDuration$value + ", configs$key=" + this.configs$key + ", configs$value=" + this.configs$value + ", checkpointChecksumEnable$value=" + this.checkpointChecksumEnable$value + ", checkpointChecksumCompatible$value=" + this.checkpointChecksumCompatible$value + ", localStorageCleanupEnable$value=" + this.localStorageCleanupEnable$value + ", checkpointRestoreIdleLimit$value=" + this.checkpointRestoreIdleLimit$value + ", ttlSeconds$value=" + this.ttlSeconds$value + ")";
        }
    }

    public static StateStoreSpec.StateStoreSpecBuilder builder() {
        return new StateStoreSpec.StateStoreSpecBuilder();
    }

    public String getName() {
        return this.name;
    }

    public Coder<?> getKeyCoder() {
        return this.keyCoder;
    }

    public Coder<?> getValCoder() {
        return this.valCoder;
    }

    public File getLocalStateStoreDir() {
        return this.localStateStoreDir;
    }

    public String getStream() {
        return this.stream;
    }

    public boolean isReadonly() {
        return this.isReadonly;
    }

    public ScheduledExecutorService getWriteIOScheduler() {
        return this.writeIOScheduler;
    }

    public ScheduledExecutorService getReadIOScheduler() {
        return this.readIOScheduler;
    }

    public ScheduledExecutorService getCheckpointIOScheduler() {
        return this.checkpointIOScheduler;
    }

    public CheckpointStore getCheckpointStore() {
        return this.checkpointStore;
    }

    public Duration getCheckpointDuration() {
        return this.checkpointDuration;
    }

    public Map<String, Object> getConfigs() {
        return this.configs;
    }

    public boolean isCheckpointChecksumEnable() {
        return this.checkpointChecksumEnable;
    }

    public boolean isCheckpointChecksumCompatible() {
        return this.checkpointChecksumCompatible;
    }

    public boolean isLocalStorageCleanupEnable() {
        return this.localStorageCleanupEnable;
    }

    public Duration getCheckpointRestoreIdleLimit() {
        return this.checkpointRestoreIdleLimit;
    }

    public int getTtlSeconds() {
        return this.ttlSeconds;
    }
}
