/*
 * Decompiled with CFR 0.152.
 */
package io.prestosql.geospatial.serde;

import com.google.common.base.Verify;
import com.google.common.collect.Iterables;
import io.airlift.slice.BasicSliceInput;
import io.airlift.slice.DynamicSliceOutput;
import io.airlift.slice.Slice;
import io.airlift.slice.SliceInput;
import io.airlift.slice.SliceOutput;
import io.prestosql.geospatial.GeometryUtils;
import io.prestosql.geospatial.serde.GeometrySerializationType;
import java.util.ArrayList;
import java.util.Objects;
import org.locationtech.jts.geom.Coordinate;
import org.locationtech.jts.geom.Envelope;
import org.locationtech.jts.geom.Geometry;
import org.locationtech.jts.geom.GeometryFactory;
import org.locationtech.jts.geom.LineString;
import org.locationtech.jts.geom.LinearRing;
import org.locationtech.jts.geom.MultiPoint;
import org.locationtech.jts.geom.Point;
import org.locationtech.jts.geom.Polygon;

public final class JtsGeometrySerde {
    private static final GeometryFactory GEOMETRY_FACTORY = new GeometryFactory();

    private JtsGeometrySerde() {
    }

    public static Geometry deserialize(Slice shape) {
        Objects.requireNonNull(shape, "shape is null");
        BasicSliceInput input = shape.getInput();
        Verify.verify((input.available() > 0 ? 1 : 0) != 0);
        GeometrySerializationType type = GeometrySerializationType.getForCode(input.readByte());
        return JtsGeometrySerde.readGeometry(input, type);
    }

    private static Geometry readGeometry(BasicSliceInput input, GeometrySerializationType type) {
        switch (type) {
            case POINT: {
                return JtsGeometrySerde.readPoint((SliceInput)input);
            }
            case MULTI_POINT: {
                return JtsGeometrySerde.readMultiPoint((SliceInput)input);
            }
            case LINE_STRING: {
                return JtsGeometrySerde.readPolyline((SliceInput)input, false);
            }
            case MULTI_LINE_STRING: {
                return JtsGeometrySerde.readPolyline((SliceInput)input, true);
            }
            case POLYGON: {
                return JtsGeometrySerde.readPolygon((SliceInput)input, false);
            }
            case MULTI_POLYGON: {
                return JtsGeometrySerde.readPolygon((SliceInput)input, true);
            }
            case GEOMETRY_COLLECTION: {
                return JtsGeometrySerde.readGeometryCollection(input);
            }
            case ENVELOPE: {
                return JtsGeometrySerde.readEnvelope((SliceInput)input);
            }
        }
        throw new UnsupportedOperationException("Unexpected type: " + (Object)((Object)type));
    }

    private static Point readPoint(SliceInput input) {
        Coordinate coordinates = JtsGeometrySerde.readCoordinate(input);
        if (Double.isNaN(coordinates.x) || Double.isNaN(coordinates.y)) {
            return GEOMETRY_FACTORY.createPoint();
        }
        return GEOMETRY_FACTORY.createPoint(coordinates);
    }

    private static Geometry readMultiPoint(SliceInput input) {
        JtsGeometrySerde.skipEsriType(input);
        JtsGeometrySerde.skipEnvelope(input);
        int pointCount = input.readInt();
        Point[] points = new Point[pointCount];
        for (int i = 0; i < pointCount; ++i) {
            points[i] = JtsGeometrySerde.readPoint(input);
        }
        return GEOMETRY_FACTORY.createMultiPoint(points);
    }

    private static Geometry readPolyline(SliceInput input, boolean multitype) {
        JtsGeometrySerde.skipEsriType(input);
        JtsGeometrySerde.skipEnvelope(input);
        int partCount = input.readInt();
        if (partCount == 0) {
            if (multitype) {
                return GEOMETRY_FACTORY.createMultiLineString();
            }
            return GEOMETRY_FACTORY.createLineString();
        }
        int pointCount = input.readInt();
        int[] startIndexes = new int[partCount];
        for (int i = 0; i < partCount; ++i) {
            startIndexes[i] = input.readInt();
        }
        int[] partLengths = new int[partCount];
        if (partCount > 1) {
            partLengths[0] = startIndexes[1];
            for (int i = 1; i < partCount - 1; ++i) {
                partLengths[i] = startIndexes[i + 1] - startIndexes[i];
            }
        }
        partLengths[partCount - 1] = pointCount - startIndexes[partCount - 1];
        LineString[] lineStrings = new LineString[partCount];
        for (int i = 0; i < partCount; ++i) {
            lineStrings[i] = GEOMETRY_FACTORY.createLineString(JtsGeometrySerde.readCoordinates(input, partLengths[i]));
        }
        if (multitype) {
            return GEOMETRY_FACTORY.createMultiLineString(lineStrings);
        }
        Verify.verify((lineStrings.length == 1 ? 1 : 0) != 0);
        return lineStrings[0];
    }

    private static Geometry readPolygon(SliceInput input, boolean multitype) {
        JtsGeometrySerde.skipEsriType(input);
        JtsGeometrySerde.skipEnvelope(input);
        int partCount = input.readInt();
        if (partCount == 0) {
            if (multitype) {
                return GEOMETRY_FACTORY.createMultiPolygon();
            }
            return GEOMETRY_FACTORY.createPolygon();
        }
        int pointCount = input.readInt();
        int[] startIndexes = new int[partCount];
        for (int i = 0; i < partCount; ++i) {
            startIndexes[i] = input.readInt();
        }
        int[] partLengths = new int[partCount];
        if (partCount > 1) {
            partLengths[0] = startIndexes[1];
            for (int i = 1; i < partCount - 1; ++i) {
                partLengths[i] = startIndexes[i + 1] - startIndexes[i];
            }
        }
        partLengths[partCount - 1] = pointCount - startIndexes[partCount - 1];
        LinearRing shell = null;
        ArrayList<LinearRing> holes = new ArrayList<LinearRing>();
        ArrayList<Polygon> polygons = new ArrayList<Polygon>();
        for (int i = 0; i < partCount; ++i) {
            Coordinate[] coordinates = JtsGeometrySerde.readCoordinates(input, partLengths[i]);
            if (JtsGeometrySerde.isClockwise(coordinates)) {
                if (shell != null) {
                    polygons.add(GEOMETRY_FACTORY.createPolygon(shell, holes.toArray(new LinearRing[0])));
                    holes.clear();
                } else {
                    Verify.verify((boolean)holes.isEmpty(), (String)"shell is null but holes found", (Object[])new Object[0]);
                }
                shell = GEOMETRY_FACTORY.createLinearRing(coordinates);
                continue;
            }
            Verify.verifyNotNull(shell, (String)"shell is null but hole found", (Object[])new Object[0]);
            holes.add(GEOMETRY_FACTORY.createLinearRing(coordinates));
        }
        polygons.add(GEOMETRY_FACTORY.createPolygon(shell, holes.toArray(new LinearRing[0])));
        if (multitype) {
            return GEOMETRY_FACTORY.createMultiPolygon(polygons.toArray(new Polygon[0]));
        }
        return (Geometry)Iterables.getOnlyElement(polygons);
    }

    private static Geometry readGeometryCollection(BasicSliceInput input) {
        ArrayList<Geometry> geometries = new ArrayList<Geometry>();
        while (input.available() > 0) {
            input.readInt();
            GeometrySerializationType type = GeometrySerializationType.getForCode(input.readByte());
            geometries.add(JtsGeometrySerde.readGeometry(input, type));
        }
        return GEOMETRY_FACTORY.createGeometryCollection(geometries.toArray(new Geometry[0]));
    }

    private static Geometry readEnvelope(SliceInput input) {
        Coordinate[] coordinates;
        Verify.verify((input.available() > 0 ? 1 : 0) != 0);
        double xMin = input.readDouble();
        double yMin = input.readDouble();
        double xMax = input.readDouble();
        double yMax = input.readDouble();
        coordinates = new Coordinate[]{new Coordinate(xMin, yMin), new Coordinate(xMin, yMax), new Coordinate(xMax, yMax), new Coordinate(xMax, yMin), coordinates[0]};
        return GEOMETRY_FACTORY.createPolygon(coordinates);
    }

    private static void skipEsriType(SliceInput input) {
        input.readInt();
    }

    private static void skipEnvelope(SliceInput input) {
        Objects.requireNonNull(input, "input is null");
        int skipLength = 32;
        Verify.verify((input.skip((long)skipLength) == (long)skipLength ? 1 : 0) != 0);
    }

    private static Coordinate readCoordinate(SliceInput input) {
        Objects.requireNonNull(input, "input is null");
        return new Coordinate(input.readDouble(), input.readDouble());
    }

    private static Coordinate[] readCoordinates(SliceInput input, int count) {
        Objects.requireNonNull(input, "input is null");
        Verify.verify((count > 0 ? 1 : 0) != 0);
        Coordinate[] coordinates = new Coordinate[count];
        for (int i = 0; i < count; ++i) {
            coordinates[i] = JtsGeometrySerde.readCoordinate(input);
        }
        return coordinates;
    }

    public static Slice serialize(Geometry geometry) {
        Objects.requireNonNull(geometry, "input is null");
        DynamicSliceOutput output = new DynamicSliceOutput(100);
        JtsGeometrySerde.writeGeometry(geometry, output);
        return output.slice();
    }

    private static void writeGeometry(Geometry geometry, DynamicSliceOutput output) {
        switch (geometry.getGeometryType()) {
            case "Point": {
                JtsGeometrySerde.writePoint((Point)geometry, (SliceOutput)output);
                break;
            }
            case "MultiPoint": {
                JtsGeometrySerde.writeMultiPoint((MultiPoint)geometry, (SliceOutput)output);
                break;
            }
            case "LineString": {
                JtsGeometrySerde.writePolyline(geometry, (SliceOutput)output, false);
                break;
            }
            case "MultiLineString": {
                JtsGeometrySerde.writePolyline(geometry, (SliceOutput)output, true);
                break;
            }
            case "Polygon": {
                JtsGeometrySerde.writePolygon(geometry, (SliceOutput)output, false);
                break;
            }
            case "MultiPolygon": {
                JtsGeometrySerde.writePolygon(geometry, (SliceOutput)output, true);
                break;
            }
            case "GeometryCollection": {
                JtsGeometrySerde.writeGeometryCollection(geometry, output);
                break;
            }
            default: {
                throw new IllegalArgumentException("Unsupported geometry type : " + geometry.getGeometryType());
            }
        }
    }

    private static void writePoint(Point point, SliceOutput output) {
        output.writeByte(GeometrySerializationType.POINT.code());
        if (!point.isEmpty()) {
            JtsGeometrySerde.writeCoordinate(point.getCoordinate(), output);
        } else {
            output.writeDouble(Double.NaN);
            output.writeDouble(Double.NaN);
        }
    }

    private static void writeMultiPoint(MultiPoint geometry, SliceOutput output) {
        output.writeByte(GeometrySerializationType.MULTI_POINT.code());
        output.writeInt(EsriShapeType.MULTI_POINT.code);
        JtsGeometrySerde.writeEnvelope((Geometry)geometry, output);
        output.writeInt(geometry.getNumPoints());
        for (Coordinate coordinate : geometry.getCoordinates()) {
            JtsGeometrySerde.writeCoordinate(coordinate, output);
        }
    }

    private static void writePolyline(Geometry geometry, SliceOutput output, boolean multitype) {
        int numParts;
        int numPoints = geometry.getNumPoints();
        if (multitype) {
            numParts = geometry.getNumGeometries();
            output.writeByte(GeometrySerializationType.MULTI_LINE_STRING.code());
        } else {
            numParts = numPoints > 0 ? 1 : 0;
            output.writeByte(GeometrySerializationType.LINE_STRING.code());
        }
        output.writeInt(EsriShapeType.POLYLINE.code);
        JtsGeometrySerde.writeEnvelope(geometry, output);
        output.writeInt(numParts);
        output.writeInt(numPoints);
        int partIndex = 0;
        for (int i = 0; i < numParts; ++i) {
            output.writeInt(partIndex);
            partIndex += geometry.getGeometryN(i).getNumPoints();
        }
        JtsGeometrySerde.writeCoordinates(geometry.getCoordinates(), output);
    }

    private static void writePolygon(Geometry geometry, SliceOutput output, boolean multitype) {
        int numGeometries = geometry.getNumGeometries();
        int numParts = 0;
        int numPoints = geometry.getNumPoints();
        for (int i = 0; i < numGeometries; ++i) {
            Polygon polygon = (Polygon)geometry.getGeometryN(i);
            if (polygon.getNumPoints() <= 0) continue;
            numParts += polygon.getNumInteriorRing() + 1;
        }
        if (multitype) {
            output.writeByte(GeometrySerializationType.MULTI_POLYGON.code());
        } else {
            output.writeByte(GeometrySerializationType.POLYGON.code());
        }
        output.writeInt(EsriShapeType.POLYGON.code);
        JtsGeometrySerde.writeEnvelope(geometry, output);
        output.writeInt(numParts);
        output.writeInt(numPoints);
        if (numParts == 0) {
            return;
        }
        int[] partIndexes = new int[numParts];
        boolean[] shellPart = new boolean[numParts];
        int currentPart = 0;
        int currentPoint = 0;
        for (int i = 0; i < numGeometries; ++i) {
            Polygon polygon = (Polygon)geometry.getGeometryN(i);
            partIndexes[currentPart] = currentPoint;
            shellPart[currentPart] = true;
            ++currentPart;
            currentPoint += polygon.getExteriorRing().getNumPoints();
            int holesCount = polygon.getNumInteriorRing();
            for (int holeIndex = 0; holeIndex < holesCount; ++holeIndex) {
                partIndexes[currentPart] = currentPoint;
                shellPart[currentPart] = false;
                ++currentPart;
                currentPoint += polygon.getInteriorRingN(holeIndex).getNumPoints();
            }
        }
        for (int partIndex : partIndexes) {
            output.writeInt(partIndex);
        }
        Coordinate[] coordinates = geometry.getCoordinates();
        JtsGeometrySerde.canonicalizePolygonCoordinates(coordinates, partIndexes, shellPart);
        JtsGeometrySerde.writeCoordinates(coordinates, output);
    }

    private static void writeGeometryCollection(Geometry collection, DynamicSliceOutput output) {
        output.appendByte(GeometrySerializationType.GEOMETRY_COLLECTION.code());
        for (int geometryIndex = 0; geometryIndex < collection.getNumGeometries(); ++geometryIndex) {
            Geometry geometry = collection.getGeometryN(geometryIndex);
            int startPosition = output.size();
            output.appendInt(0);
            JtsGeometrySerde.writeGeometry(geometry, output);
            int endPosition = output.size();
            int length = endPosition - startPosition - 4;
            output.getUnderlyingSlice().setInt(startPosition, length);
        }
    }

    private static void writeCoordinate(Coordinate coordinate, SliceOutput output) {
        output.writeDouble(GeometryUtils.translateToAVNaN(coordinate.x));
        output.writeDouble(GeometryUtils.translateToAVNaN(coordinate.y));
    }

    private static void writeCoordinates(Coordinate[] coordinates, SliceOutput output) {
        for (Coordinate coordinate : coordinates) {
            JtsGeometrySerde.writeCoordinate(coordinate, output);
        }
    }

    private static void writeEnvelope(Geometry geometry, SliceOutput output) {
        if (geometry.isEmpty()) {
            for (int i = 0; i < 4; ++i) {
                output.writeDouble(Double.NaN);
            }
            return;
        }
        Envelope envelope = geometry.getEnvelopeInternal();
        output.writeDouble(envelope.getMinX());
        output.writeDouble(envelope.getMinY());
        output.writeDouble(envelope.getMaxX());
        output.writeDouble(envelope.getMaxY());
    }

    private static void canonicalizePolygonCoordinates(Coordinate[] coordinates, int[] partIndexes, boolean[] shellPart) {
        for (int part = 0; part < partIndexes.length - 1; ++part) {
            JtsGeometrySerde.canonicalizePolygonCoordinates(coordinates, partIndexes[part], partIndexes[part + 1], shellPart[part]);
        }
        if (partIndexes.length > 0) {
            JtsGeometrySerde.canonicalizePolygonCoordinates(coordinates, partIndexes[partIndexes.length - 1], coordinates.length, shellPart[partIndexes.length - 1]);
        }
    }

    private static void canonicalizePolygonCoordinates(Coordinate[] coordinates, int start, int end, boolean isShell) {
        boolean isClockwise = JtsGeometrySerde.isClockwise(coordinates, start, end);
        if (isShell && !isClockwise || !isShell && isClockwise) {
            JtsGeometrySerde.reverse(coordinates, start, end);
        }
    }

    private static boolean isClockwise(Coordinate[] coordinates) {
        return JtsGeometrySerde.isClockwise(coordinates, 0, coordinates.length);
    }

    private static boolean isClockwise(Coordinate[] coordinates, int start, int end) {
        double area = 0.0;
        for (int i = start + 1; i < end; ++i) {
            area += (coordinates[i].x - coordinates[i - 1].x) * (coordinates[i].y + coordinates[i - 1].y);
        }
        return (area += (coordinates[start].x - coordinates[end - 1].x) * (coordinates[start].y + coordinates[end - 1].y)) > 0.0;
    }

    private static void reverse(Coordinate[] coordinates, int start, int end) {
        Verify.verify((start <= end ? 1 : 0) != 0, (String)"start must be less or equal than end", (Object[])new Object[0]);
        for (int i = start; i < start + (end - start) / 2; ++i) {
            Coordinate buffer = coordinates[i];
            coordinates[i] = coordinates[start + end - i - 1];
            coordinates[start + end - i - 1] = buffer;
        }
    }

    private static enum EsriShapeType {
        POINT(1),
        POLYLINE(3),
        POLYGON(5),
        MULTI_POINT(8);

        final int code;

        private EsriShapeType(int code) {
            this.code = code;
        }
    }
}

