/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.mledger.impl;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.primitives.Longs;
import io.netty.buffer.ByteBuf;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.Executor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import org.apache.bookkeeper.client.api.LedgerEntry;
import org.apache.bookkeeper.client.api.ReadHandle;
import org.apache.bookkeeper.mledger.AsyncCallbacks;
import org.apache.bookkeeper.mledger.Entry;
import org.apache.bookkeeper.mledger.ManagedLedgerException;
import org.apache.bookkeeper.mledger.impl.EntryCache;
import org.apache.bookkeeper.mledger.impl.EntryCacheDefaultEvictionPolicy;
import org.apache.bookkeeper.mledger.impl.EntryCacheEvictionPolicy;
import org.apache.bookkeeper.mledger.impl.EntryCacheImpl;
import org.apache.bookkeeper.mledger.impl.EntryImpl;
import org.apache.bookkeeper.mledger.impl.ManagedLedgerFactoryImpl;
import org.apache.bookkeeper.mledger.impl.ManagedLedgerFactoryMBeanImpl;
import org.apache.bookkeeper.mledger.impl.ManagedLedgerImpl;
import org.apache.bookkeeper.mledger.impl.PositionImpl;
import org.apache.bookkeeper.mledger.util.SafeRun;
import org.apache.commons.lang3.tuple.Pair;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class EntryCacheManager {
    private final long maxSize;
    private final long evictionTriggerThreshold;
    private final double cacheEvictionWatermark;
    private final AtomicLong currentSize = new AtomicLong(0L);
    private final ConcurrentMap<String, EntryCache> caches = Maps.newConcurrentMap();
    private final EntryCacheEvictionPolicy evictionPolicy;
    private final AtomicBoolean evictionInProgress = new AtomicBoolean(false);
    private final ManagedLedgerFactoryImpl mlFactory;
    protected final ManagedLedgerFactoryMBeanImpl mlFactoryMBean;
    protected static final double MB = 1048576.0;
    private static final double evictionTriggerThresholdPercent = 0.98;
    private static final Logger log = LoggerFactory.getLogger(EntryCacheManager.class);

    public EntryCacheManager(ManagedLedgerFactoryImpl factory) {
        this.maxSize = factory.getConfig().getMaxCacheSize();
        this.evictionTriggerThreshold = (long)((double)this.maxSize * 0.98);
        this.cacheEvictionWatermark = factory.getConfig().getCacheEvictionWatermark();
        this.evictionPolicy = new EntryCacheDefaultEvictionPolicy();
        this.mlFactory = factory;
        this.mlFactoryMBean = factory.mbean;
        log.info("Initialized managed-ledger entry cache of {} Mb", (Object)((double)this.maxSize / 1048576.0));
    }

    public EntryCache getEntryCache(ManagedLedgerImpl ml) {
        if (this.maxSize == 0L) {
            return new EntryCacheDisabled(ml);
        }
        EntryCacheImpl newEntryCache = new EntryCacheImpl(this, ml, this.mlFactory.getConfig().isCopyEntriesInCache());
        EntryCache currentEntryCache = this.caches.putIfAbsent(ml.getName(), newEntryCache);
        if (currentEntryCache != null) {
            return currentEntryCache;
        }
        return newEntryCache;
    }

    void removeEntryCache(String name) {
        EntryCache entryCache = (EntryCache)this.caches.remove(name);
        if (entryCache == null) {
            return;
        }
        long size = entryCache.getSize();
        entryCache.clear();
        if (log.isDebugEnabled()) {
            log.debug("Removed cache for {} - Size: {} -- Current Size: {}", new Object[]{name, (double)size / 1048576.0, (double)this.currentSize.get() / 1048576.0});
        }
    }

    boolean hasSpaceInCache() {
        long currentSize = this.currentSize.get();
        if (currentSize > this.evictionTriggerThreshold && this.evictionInProgress.compareAndSet(false, true)) {
            this.mlFactory.scheduledExecutor.execute((Runnable)SafeRun.safeRun(() -> {
                long sizeToEvict = currentSize - (long)((double)this.maxSize * this.cacheEvictionWatermark);
                long startTime = System.nanoTime();
                log.info("Triggering cache eviction. total size: {} Mb -- Need to discard: {} Mb", (Object)((double)currentSize / 1048576.0), (Object)((double)sizeToEvict / 1048576.0));
                try {
                    this.evictionPolicy.doEviction(Lists.newArrayList(this.caches.values()), sizeToEvict);
                    long endTime = System.nanoTime();
                    double durationMs = (double)TimeUnit.NANOSECONDS.toMicros(endTime - startTime) / 1000.0;
                    log.info("Eviction completed. Removed {} Mb in {} ms", (Object)((double)(currentSize - this.currentSize.get()) / 1048576.0), (Object)durationMs);
                }
                finally {
                    this.mlFactoryMBean.recordCacheEviction();
                    this.evictionInProgress.set(false);
                }
            }));
        }
        return currentSize < this.maxSize;
    }

    void entryAdded(long size) {
        this.currentSize.addAndGet(size);
    }

    void entriesRemoved(long size) {
        this.currentSize.addAndGet(-size);
    }

    public long getSize() {
        return this.currentSize.get();
    }

    public long getMaxSize() {
        return this.maxSize;
    }

    public void clear() {
        this.caches.values().forEach(EntryCache::clear);
    }

    public static Entry create(long ledgerId, long entryId, ByteBuf data) {
        return EntryImpl.create(ledgerId, entryId, data);
    }

    protected class EntryCacheDisabled
    implements EntryCache {
        private final ManagedLedgerImpl ml;

        public EntryCacheDisabled(ManagedLedgerImpl ml) {
            this.ml = ml;
        }

        @Override
        public String getName() {
            return this.ml.getName();
        }

        @Override
        public boolean insert(EntryImpl entry) {
            return false;
        }

        @Override
        public void invalidateEntries(PositionImpl lastPosition) {
        }

        @Override
        public void invalidateAllEntries(long ledgerId) {
        }

        @Override
        public void clear() {
        }

        @Override
        public Pair<Integer, Long> evictEntries(long sizeToFree) {
            return Pair.of((Object)0, (Object)0L);
        }

        @Override
        public void invalidateEntriesBeforeTimestamp(long timestamp) {
        }

        @Override
        public void asyncReadEntry(ReadHandle lh, long firstEntry, long lastEntry, boolean isSlowestReader, AsyncCallbacks.ReadEntriesCallback callback, Object ctx) {
            ((CompletableFuture)lh.readAsync(firstEntry, lastEntry).thenAcceptAsync(ledgerEntries -> {
                ArrayList entries = Lists.newArrayList();
                long totalSize = 0L;
                try {
                    for (LedgerEntry e : ledgerEntries) {
                        EntryImpl entry = EntryImpl.create(e);
                        entries.add(entry);
                        totalSize += (long)entry.getLength();
                    }
                }
                finally {
                    ledgerEntries.close();
                }
                EntryCacheManager.this.mlFactoryMBean.recordCacheMiss(entries.size(), totalSize);
                this.ml.mbean.addReadEntriesSample(entries.size(), totalSize);
                callback.readEntriesComplete(entries, ctx);
            }, (Executor)this.ml.getExecutor().chooseThread((Object)this.ml.getName()))).exceptionally(exception -> {
                callback.readEntriesFailed(ManagedLedgerImpl.createManagedLedgerException(exception), ctx);
                return null;
            });
        }

        @Override
        public void asyncReadEntry(ReadHandle lh, PositionImpl position, AsyncCallbacks.ReadEntryCallback callback, Object ctx) {
            lh.readAsync(position.getEntryId(), position.getEntryId()).whenCompleteAsync((ledgerEntries, exception) -> {
                if (exception != null) {
                    this.ml.invalidateLedgerHandle(lh);
                    callback.readEntryFailed(ManagedLedgerImpl.createManagedLedgerException(exception), ctx);
                    return;
                }
                try {
                    Iterator iterator = ledgerEntries.iterator();
                    if (iterator.hasNext()) {
                        LedgerEntry ledgerEntry = (LedgerEntry)iterator.next();
                        EntryImpl returnEntry = EntryImpl.create(ledgerEntry);
                        EntryCacheManager.this.mlFactoryMBean.recordCacheMiss(1, returnEntry.getLength());
                        this.ml.getMBean().addReadEntriesSample(1, returnEntry.getLength());
                        callback.readEntryComplete(returnEntry, ctx);
                    } else {
                        callback.readEntryFailed(new ManagedLedgerException("Could not read given position"), ctx);
                    }
                }
                finally {
                    ledgerEntries.close();
                }
            }, (Executor)this.ml.getExecutor().chooseThread((Object)this.ml.getName()));
        }

        @Override
        public long getSize() {
            return 0L;
        }

        @Override
        public int compareTo(EntryCache other) {
            return Longs.compare((long)this.getSize(), (long)other.getSize());
        }
    }
}

