/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.admin.impl;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import org.apache.pulsar.broker.web.PulsarWebResource;
import org.apache.pulsar.broker.web.RestException;
import org.apache.pulsar.client.admin.PulsarAdminException;
import org.apache.pulsar.shade.com.google.common.collect.Lists;
import org.apache.pulsar.shade.io.swagger.annotations.ApiOperation;
import org.apache.pulsar.shade.io.swagger.annotations.ApiParam;
import org.apache.pulsar.shade.io.swagger.annotations.ApiResponse;
import org.apache.pulsar.shade.io.swagger.annotations.ApiResponses;
import org.apache.pulsar.shade.javax.ws.rs.DELETE;
import org.apache.pulsar.shade.javax.ws.rs.DefaultValue;
import org.apache.pulsar.shade.javax.ws.rs.GET;
import org.apache.pulsar.shade.javax.ws.rs.POST;
import org.apache.pulsar.shade.javax.ws.rs.PUT;
import org.apache.pulsar.shade.javax.ws.rs.Path;
import org.apache.pulsar.shade.javax.ws.rs.PathParam;
import org.apache.pulsar.shade.javax.ws.rs.QueryParam;
import org.apache.pulsar.shade.javax.ws.rs.container.AsyncResponse;
import org.apache.pulsar.shade.javax.ws.rs.container.Suspended;
import org.apache.pulsar.shade.javax.ws.rs.core.Response;
import org.apache.pulsar.shade.org.apache.commons.lang3.StringUtils;
import org.apache.pulsar.shade.org.apache.pulsar.common.naming.NamedEntity;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.data.TenantInfo;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.data.TenantInfoImpl;
import org.apache.pulsar.shade.org.apache.pulsar.common.util.FutureUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TenantsBase
extends PulsarWebResource {
    private static final Logger log = LoggerFactory.getLogger(TenantsBase.class);

    @GET
    @ApiOperation(value="Get the list of existing tenants.", response=String.class, responseContainer="List")
    @ApiResponses(value={@ApiResponse(code=403, message="The requester doesn't have admin permissions"), @ApiResponse(code=404, message="Tenant doesn't exist")})
    public void getTenants(@Suspended AsyncResponse asyncResponse) {
        String clientAppId = this.clientAppId();
        try {
            this.validateSuperUserAccess();
        }
        catch (Exception e2) {
            asyncResponse.resume(e2);
            return;
        }
        this.tenantResources().listTenantsAsync().whenComplete((tenants, e) -> {
            if (e != null) {
                log.error("[{}] Failed to get tenants list", (Object)clientAppId, e);
                asyncResponse.resume(new RestException((Throwable)e));
                return;
            }
            ArrayList deepCopy = new ArrayList(tenants);
            deepCopy.sort(null);
            asyncResponse.resume(deepCopy);
        });
    }

    @GET
    @Path(value="/{tenant}")
    @ApiOperation(value="Get the admin configuration for a given tenant.")
    @ApiResponses(value={@ApiResponse(code=403, message="The requester doesn't have admin permissions"), @ApiResponse(code=404, message="Tenant does not exist")})
    public void getTenantAdmin(@Suspended AsyncResponse asyncResponse, @ApiParam(value="The tenant name") @PathParam(value="tenant") String tenant) {
        String clientAppId = this.clientAppId();
        try {
            this.validateSuperUserAccess();
        }
        catch (Exception e2) {
            asyncResponse.resume(e2);
        }
        this.tenantResources().getTenantAsync(tenant).whenComplete((tenantInfo, e) -> {
            if (e != null) {
                log.error("[{}] Failed to get Tenant {}", (Object)clientAppId, (Object)e.getMessage());
                asyncResponse.resume(new RestException(Response.Status.INTERNAL_SERVER_ERROR, "Failed to get Tenant"));
                return;
            }
            boolean response = tenantInfo.isPresent() ? asyncResponse.resume(tenantInfo.get()) : asyncResponse.resume(new RestException(Response.Status.NOT_FOUND, "Tenant does not exist"));
        });
    }

    @PUT
    @Path(value="/{tenant}")
    @ApiOperation(value="Create a new tenant.", notes="This operation requires Pulsar super-user privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="The requester doesn't have admin permissions"), @ApiResponse(code=409, message="Tenant already exists"), @ApiResponse(code=412, message="Tenant name is not valid"), @ApiResponse(code=412, message="Clusters can not be empty"), @ApiResponse(code=412, message="Clusters do not exist")})
    public void createTenant(@Suspended AsyncResponse asyncResponse, @ApiParam(value="The tenant name") @PathParam(value="tenant") String tenant, @ApiParam(value="TenantInfo") TenantInfoImpl tenantInfo) {
        String clientAppId = this.clientAppId();
        try {
            this.validateSuperUserAccess();
            this.validatePoliciesReadOnlyAccess();
            this.validateClusters(tenantInfo);
            NamedEntity.checkName(tenant);
        }
        catch (IllegalArgumentException e2) {
            log.warn("[{}] Failed to create tenant with invalid name {}", new Object[]{this.clientAppId(), tenant, e2});
            asyncResponse.resume(new RestException(Response.Status.PRECONDITION_FAILED, "Tenant name is not valid"));
            return;
        }
        catch (Exception e3) {
            asyncResponse.resume(e3);
            return;
        }
        this.tenantResources().listTenantsAsync().whenComplete((tenants, e) -> {
            if (e != null) {
                log.error("[{}] Failed to create tenant ", (Object)clientAppId, (Object)e.getCause());
                asyncResponse.resume(new RestException((Throwable)e));
                return;
            }
            int maxTenants = this.pulsar().getConfiguration().getMaxTenants();
            if (maxTenants > 0 && tenants != null && tenants.size() >= maxTenants) {
                asyncResponse.resume(new RestException(Response.Status.PRECONDITION_FAILED, "Exceed the maximum number of tenants"));
                return;
            }
            ((CompletableFuture)this.tenantResources().tenantExistsAsync(tenant).thenAccept(exist -> {
                if (exist.booleanValue()) {
                    asyncResponse.resume(new RestException(Response.Status.CONFLICT, "Tenant already exist"));
                    return;
                }
                ((CompletableFuture)this.tenantResources().createTenantAsync(tenant, tenantInfo).thenAccept(r -> {
                    log.info("[{}] Created tenant {}", (Object)this.clientAppId(), (Object)tenant);
                    asyncResponse.resume(Response.noContent().build());
                })).exceptionally(ex -> {
                    log.error("[{}] Failed to create tenant {}", new Object[]{clientAppId, tenant, e});
                    asyncResponse.resume(new RestException((Throwable)ex));
                    return null;
                });
            })).exceptionally(ex -> {
                log.error("[{}] Failed to create tenant {}", new Object[]{this.clientAppId(), tenant, e});
                asyncResponse.resume(new RestException((Throwable)ex));
                return null;
            });
        });
    }

    @POST
    @Path(value="/{tenant}")
    @ApiOperation(value="Update the admins for a tenant.", notes="This operation requires Pulsar super-user privileges.")
    @ApiResponses(value={@ApiResponse(code=403, message="The requester doesn't have admin permissions"), @ApiResponse(code=404, message="Tenant does not exist"), @ApiResponse(code=409, message="Tenant already exists"), @ApiResponse(code=412, message="Clusters can not be empty"), @ApiResponse(code=412, message="Clusters do not exist")})
    public void updateTenant(@Suspended AsyncResponse asyncResponse, @ApiParam(value="The tenant name") @PathParam(value="tenant") String tenant, @ApiParam(value="TenantInfo") TenantInfoImpl newTenantAdmin) {
        try {
            this.validateSuperUserAccess();
            this.validatePoliciesReadOnlyAccess();
            this.validateClusters(newTenantAdmin);
        }
        catch (Exception e) {
            asyncResponse.resume(e);
            return;
        }
        String clientAddId = this.clientAppId();
        ((CompletableFuture)this.tenantResources().getTenantAsync(tenant).thenAccept(tenantAdmin -> {
            if (!tenantAdmin.isPresent()) {
                asyncResponse.resume(new RestException(Response.Status.NOT_FOUND, "Tenant " + tenant + " not found"));
                return;
            }
            TenantInfo oldTenantAdmin = (TenantInfo)tenantAdmin.get();
            HashSet<String> newClusters = new HashSet<String>(newTenantAdmin.getAllowedClusters());
            ((CompletableFuture)this.canUpdateCluster(tenant, oldTenantAdmin.getAllowedClusters(), newClusters).thenApply(r -> {
                ((CompletableFuture)this.tenantResources().updateTenantAsync(tenant, old -> newTenantAdmin).thenAccept(done -> {
                    log.info("Successfully updated tenant info {}", (Object)tenant);
                    asyncResponse.resume(Response.noContent().build());
                })).exceptionally(ex -> {
                    log.warn("Failed to update tenant {}", (Object)tenant, (Object)ex.getCause());
                    asyncResponse.resume(new RestException((Throwable)ex));
                    return null;
                });
                return null;
            })).exceptionally(nsEx -> {
                asyncResponse.resume(nsEx.getCause());
                return null;
            });
        })).exceptionally(ex -> {
            log.error("[{}] Failed to get tenant {}", new Object[]{clientAddId, tenant, ex.getCause()});
            asyncResponse.resume(new RestException((Throwable)ex));
            return null;
        });
    }

    @DELETE
    @Path(value="/{tenant}")
    @ApiOperation(value="Delete a tenant and all namespaces and topics under it.")
    @ApiResponses(value={@ApiResponse(code=403, message="The requester doesn't have admin permissions"), @ApiResponse(code=404, message="Tenant does not exist"), @ApiResponse(code=405, message="Broker doesn't allow forced deletion of tenants"), @ApiResponse(code=409, message="The tenant still has active namespaces")})
    public void deleteTenant(@Suspended AsyncResponse asyncResponse, @PathParam(value="tenant") @ApiParam(value="The tenant name") String tenant, @QueryParam(value="force") @DefaultValue(value="false") boolean force) {
        try {
            this.validateSuperUserAccess();
            this.validatePoliciesReadOnlyAccess();
        }
        catch (Exception e) {
            asyncResponse.resume(e);
            return;
        }
        this.internalDeleteTenant(asyncResponse, tenant, force);
    }

    protected void internalDeleteTenant(AsyncResponse asyncResponse, String tenant, boolean force) {
        if (force) {
            this.internalDeleteTenantForcefully(asyncResponse, tenant);
        } else {
            this.internalDeleteTenant(asyncResponse, tenant);
        }
    }

    protected void internalDeleteTenant(AsyncResponse asyncResponse, String tenant) {
        this.tenantResources().tenantExistsAsync(tenant).thenApply(exists -> {
            if (!exists.booleanValue()) {
                asyncResponse.resume(new RestException(Response.Status.NOT_FOUND, "Tenant doesn't exist"));
                return null;
            }
            return ((CompletableFuture)this.hasActiveNamespace(tenant).thenAccept(ns -> {
                try {
                    ((CompletableFuture)this.tenantResources().deleteTenantAsync(tenant).thenAccept(t -> {
                        log.info("[{}] Deleted tenant {}", (Object)this.clientAppId(), (Object)tenant);
                        asyncResponse.resume(Response.noContent().build());
                    })).exceptionally(ex -> {
                        log.error("Failed to delete tenant {}", (Object)tenant, (Object)ex.getCause());
                        asyncResponse.resume(new RestException((Throwable)ex));
                        return null;
                    });
                    log.info("[{}] Deleted tenant {}", (Object)this.clientAppId(), (Object)tenant);
                }
                catch (Exception e) {
                    log.error("[{}] Failed to delete tenant {}", new Object[]{this.clientAppId(), tenant, e});
                    asyncResponse.resume(new RestException(e));
                }
            })).exceptionally(ex -> {
                log.error("Failed to delete tenant due to active namespace {}", (Object)tenant, (Object)ex.getCause());
                asyncResponse.resume(new RestException((Throwable)ex));
                return null;
            });
        });
    }

    protected void internalDeleteTenantForcefully(AsyncResponse asyncResponse, String tenant) {
        List<String> namespaces;
        if (!this.pulsar().getConfiguration().isForceDeleteTenantAllowed()) {
            asyncResponse.resume(new RestException(Response.Status.METHOD_NOT_ALLOWED, "Broker doesn't allow forced deletion of tenants"));
            return;
        }
        try {
            namespaces = this.tenantResources().getListOfNamespaces(tenant);
        }
        catch (Exception e) {
            log.error("[{}] Failed to get namespaces list of {}", new Object[]{this.clientAppId(), tenant, e});
            asyncResponse.resume(new RestException(e));
            return;
        }
        ArrayList<CompletableFuture<Void>> futures = Lists.newArrayList();
        try {
            for (String namespace : namespaces) {
                futures.add(this.pulsar().getAdminClient().namespaces().deleteNamespaceAsync(namespace, true));
            }
        }
        catch (Exception e) {
            log.error("[{}] Failed to force delete namespaces {}", new Object[]{this.clientAppId(), namespaces, e});
            asyncResponse.resume(new RestException(e));
        }
        FutureUtil.waitForAll(futures).handle((result, exception) -> {
            if (exception != null) {
                if (exception.getCause() instanceof PulsarAdminException) {
                    asyncResponse.resume(new RestException((PulsarAdminException)exception.getCause()));
                } else {
                    log.error("[{}] Failed to force delete namespaces {}", new Object[]{this.clientAppId(), namespaces, exception});
                    asyncResponse.resume(new RestException(exception.getCause()));
                }
                return null;
            }
            try {
                this.pulsar().getPulsarResources().getTopicResources().clearTenantPersistence(tenant).get();
            }
            catch (InterruptedException | ExecutionException e) {
                log.warn("[{}] Failed to remove managed-ledger for {}", new Object[]{this.clientAppId(), tenant, e});
            }
            this.internalDeleteTenant(asyncResponse, tenant);
            asyncResponse.resume(Response.noContent().build());
            return null;
        });
    }

    private void validateClusters(TenantInfo info) {
        List nonexistentClusters;
        if (info == null || info.getAllowedClusters().stream().filter(c -> !StringUtils.isBlank(c)).collect(Collectors.toSet()).isEmpty() || info.getAllowedClusters().stream().anyMatch(ac -> StringUtils.isBlank(ac))) {
            log.warn("[{}] Failed to validate due to clusters are empty", (Object)this.clientAppId());
            throw new RestException(Response.Status.PRECONDITION_FAILED, "Clusters can not be empty");
        }
        try {
            Set<String> availableClusters = this.clusterResources().list();
            Set<String> allowedClusters = info.getAllowedClusters();
            nonexistentClusters = allowedClusters.stream().filter(cluster -> !availableClusters.contains(cluster) && !"global".equals(cluster)).collect(Collectors.toList());
        }
        catch (Exception e) {
            log.error("[{}] Failed to get available clusters", (Object)this.clientAppId(), (Object)e);
            throw new RestException(e);
        }
        if (nonexistentClusters.size() > 0) {
            log.warn("[{}] Failed to validate due to clusters {} do not exist", (Object)this.clientAppId(), nonexistentClusters);
            throw new RestException(Response.Status.PRECONDITION_FAILED, "Clusters do not exist");
        }
    }
}

