/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service;

import java.util.ArrayList;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.admin.AdminResource;
import org.apache.pulsar.broker.service.Producer;
import org.apache.pulsar.broker.service.persistent.PersistentTopic;
import org.apache.pulsar.shade.com.google.common.collect.Lists;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.ManagedCursor;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.impl.ManagedLedgerImpl;
import org.apache.pulsar.shade.org.apache.pulsar.common.naming.TopicName;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.data.BacklogQuota;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.data.Policies;
import org.apache.pulsar.shade.org.apache.pulsar.common.policies.data.TopicPolicies;
import org.apache.pulsar.shade.org.apache.pulsar.common.util.FutureUtil;
import org.apache.pulsar.zookeeper.ZooKeeperDataCache;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BacklogQuotaManager {
    private static final Logger log = LoggerFactory.getLogger(BacklogQuotaManager.class);
    private final BacklogQuota defaultQuota;
    private final ZooKeeperDataCache<Policies> zkCache;
    private final PulsarService pulsar;
    private final boolean isTopicLevelPoliciesEnable;

    public BacklogQuotaManager(PulsarService pulsar) {
        this.isTopicLevelPoliciesEnable = pulsar.getConfiguration().isTopicLevelPoliciesEnabled();
        this.defaultQuota = new BacklogQuota(pulsar.getConfiguration().getBacklogQuotaDefaultLimitGB() * 1024L * 1024L * 1024L, pulsar.getConfiguration().getBacklogQuotaDefaultRetentionPolicy());
        this.zkCache = pulsar.getConfigurationCache().policiesCache();
        this.pulsar = pulsar;
    }

    public BacklogQuota getDefaultQuota() {
        return this.defaultQuota;
    }

    public BacklogQuota getBacklogQuota(String namespace, String policyPath) {
        try {
            return this.zkCache.get(policyPath).map(p -> p.backlog_quota_map.getOrDefault((Object)BacklogQuota.BacklogQuotaType.destination_storage, this.defaultQuota)).orElse(this.defaultQuota);
        }
        catch (Exception e) {
            log.warn("Failed to read policies data, will apply the default backlog quota: namespace={}", (Object)namespace, (Object)e);
            return this.defaultQuota;
        }
    }

    public BacklogQuota getBacklogQuota(TopicName topicName) {
        String policyPath = AdminResource.path("policies", topicName.getNamespace());
        if (!this.isTopicLevelPoliciesEnable) {
            return this.getBacklogQuota(topicName.getNamespace(), policyPath);
        }
        try {
            return Optional.ofNullable(this.pulsar.getTopicPoliciesService().getTopicPolicies(topicName)).map(TopicPolicies::getBackLogQuotaMap).map(map -> (BacklogQuota)map.get(BacklogQuota.BacklogQuotaType.destination_storage.name())).orElseGet(() -> this.getBacklogQuota(topicName.getNamespace(), policyPath));
        }
        catch (Exception e) {
            log.warn("Failed to read topic policies data, will apply the namespace backlog quota: topicName={}", (Object)topicName, (Object)e);
            return this.getBacklogQuota(topicName.getNamespace(), policyPath);
        }
    }

    public long getBacklogQuotaLimit(TopicName topicName) {
        return this.getBacklogQuota(topicName).getLimit();
    }

    public void handleExceededBacklogQuota(PersistentTopic persistentTopic) {
        TopicName topicName = TopicName.get(persistentTopic.getName());
        BacklogQuota quota = this.getBacklogQuota(topicName);
        log.info("Backlog quota exceeded for topic [{}]. Applying [{}] policy", (Object)persistentTopic.getName(), (Object)quota.getPolicy());
        switch (quota.getPolicy()) {
            case consumer_backlog_eviction: {
                this.dropBacklog(persistentTopic, quota);
                break;
            }
            case producer_exception: 
            case producer_request_hold: {
                this.disconnectProducers(persistentTopic);
                break;
            }
        }
    }

    private void dropBacklog(PersistentTopic persistentTopic, BacklogQuota quota) {
        double reductionFactor = 0.9;
        double targetSize = reductionFactor * (double)quota.getLimit();
        ManagedLedgerImpl mLedger = (ManagedLedgerImpl)persistentTopic.getManagedLedger();
        long backlogSize = mLedger.getEstimatedBacklogSize();
        if (log.isDebugEnabled()) {
            log.debug("[{}] target size is [{}] for quota limit [{}], backlog size is [{}]", new Object[]{persistentTopic.getName(), targetSize, targetSize / reductionFactor, backlogSize});
        }
        ManagedCursor previousSlowestConsumer = null;
        while ((double)backlogSize > targetSize) {
            ManagedCursor slowestConsumer = mLedger.getSlowestConsumer();
            if (slowestConsumer == null) {
                if (!log.isDebugEnabled()) break;
                log.debug("[{}] slowest consumer null.", (Object)persistentTopic.getName());
                break;
            }
            double messageSkipFactor = ((double)backlogSize - targetSize) / (double)backlogSize;
            if (slowestConsumer == previousSlowestConsumer) {
                log.info("[{}] Cursors not progressing, target size is [{}] for quota limit [{}], backlog size is [{}]", new Object[]{persistentTopic.getName(), targetSize, targetSize / reductionFactor, backlogSize});
                break;
            }
            long entriesInBacklog = slowestConsumer.getNumberOfEntriesInBacklog(false);
            int messagesToSkip = (int)(messageSkipFactor * (double)entriesInBacklog);
            try {
                if (messagesToSkip == 0) {
                    if (!log.isDebugEnabled()) break;
                    log.debug("no messages to skip for [{}]", (Object)slowestConsumer);
                    break;
                }
                if (log.isDebugEnabled()) {
                    log.debug("Skipping [{}] messages on slowest consumer [{}] having backlog entries : [{}]", new Object[]{messagesToSkip, slowestConsumer.getName(), entriesInBacklog});
                }
                slowestConsumer.skipEntries(messagesToSkip, ManagedCursor.IndividualDeletedEntries.Include);
            }
            catch (Exception e) {
                log.error("Error skipping [{}] messages from slowest consumer : [{}]", (Object)messagesToSkip, (Object)slowestConsumer.getName());
            }
            backlogSize = mLedger.getEstimatedBacklogSize();
            previousSlowestConsumer = slowestConsumer;
            if (!log.isDebugEnabled()) continue;
            log.debug("[{}] Updated unconsumed size = [{}]. skipFactor: [{}]", new Object[]{persistentTopic.getName(), backlogSize, messageSkipFactor});
        }
    }

    private void disconnectProducers(PersistentTopic persistentTopic) {
        ArrayList futures = Lists.newArrayList();
        Map<String, Producer> producers = persistentTopic.getProducers();
        producers.values().forEach(producer -> {
            log.info("Producer [{}] has exceeded backlog quota on topic [{}]. Disconnecting producer", (Object)producer.getProducerName(), (Object)persistentTopic.getName());
            futures.add(producer.disconnect());
        });
        ((CompletableFuture)FutureUtil.waitForAll(futures).thenRun(() -> log.info("All producers on topic [{}] are disconnected", (Object)persistentTopic.getName()))).exceptionally(exception -> {
            log.error("Error in disconnecting producers on topic [{}] [{}]", (Object)persistentTopic.getName(), exception);
            return null;
        });
    }
}

