/*
 * Decompiled with CFR 0.152.
 */
package org.apache.pulsar.broker.service.persistent;

import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import org.apache.pulsar.broker.PulsarService;
import org.apache.pulsar.broker.service.Topic;
import org.apache.pulsar.broker.service.persistent.PersistentTopic;
import org.apache.pulsar.shade.com.google.common.annotations.VisibleForTesting;
import org.apache.pulsar.shade.io.netty.buffer.ByteBuf;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.AsyncCallbacks;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.Entry;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.ManagedCursor;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.ManagedLedger;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.ManagedLedgerException;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.Position;
import org.apache.pulsar.shade.org.apache.bookkeeper.mledger.impl.PositionImpl;
import org.apache.pulsar.shade.org.apache.pulsar.common.api.proto.MessageMetadata;
import org.apache.pulsar.shade.org.apache.pulsar.common.protocol.Commands;
import org.apache.pulsar.shade.org.apache.pulsar.common.util.collections.ConcurrentOpenHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class MessageDeduplication {
    private final PulsarService pulsar;
    private final PersistentTopic topic;
    private final ManagedLedger managedLedger;
    private ManagedCursor managedCursor;
    private volatile Status status;
    @VisibleForTesting
    final ConcurrentOpenHashMap<String, Long> highestSequencedPushed = ConcurrentOpenHashMap.newBuilder().expectedItems(16).concurrencyLevel(1).build();
    @VisibleForTesting
    final ConcurrentOpenHashMap<String, Long> highestSequencedPersisted = ConcurrentOpenHashMap.newBuilder().expectedItems(16).concurrencyLevel(1).build();
    private final int snapshotInterval;
    private int snapshotCounter;
    private volatile long lastSnapshotTimestamp = 0L;
    private final int maxNumberOfProducers;
    private final Map<String, Long> inactiveProducers = new ConcurrentHashMap<String, Long>();
    private final String replicatorPrefix;
    private static final Logger log = LoggerFactory.getLogger(MessageDeduplication.class);

    public MessageDeduplication(PulsarService pulsar, PersistentTopic topic, ManagedLedger managedLedger) {
        this.pulsar = pulsar;
        this.topic = topic;
        this.managedLedger = managedLedger;
        this.status = Status.Initialized;
        this.snapshotInterval = pulsar.getConfiguration().getBrokerDeduplicationEntriesInterval();
        this.maxNumberOfProducers = pulsar.getConfiguration().getBrokerDeduplicationMaxNumberOfProducers();
        this.snapshotCounter = 0;
        this.replicatorPrefix = pulsar.getConfiguration().getReplicatorPrefix();
    }

    private CompletableFuture<Void> recoverSequenceIdsMap() {
        this.managedCursor.getProperties().forEach((k, v) -> {
            this.producerRemoved((String)k);
            this.highestSequencedPushed.put((String)k, (Long)v);
            this.highestSequencedPersisted.put((String)k, (Long)v);
        });
        log.info("[{}] Replaying {} entries for deduplication", (Object)this.topic.getName(), (Object)this.managedCursor.getNumberOfEntries());
        CompletableFuture<Void> future = new CompletableFuture<Void>();
        this.replayCursor(future);
        return future;
    }

    private void replayCursor(final CompletableFuture<Void> future) {
        this.managedCursor.asyncReadEntries(100, new AsyncCallbacks.ReadEntriesCallback(){

            @Override
            public void readEntriesComplete(List<Entry> entries, Object ctx) {
                for (Entry entry : entries) {
                    ByteBuf messageMetadataAndPayload = entry.getDataBuffer();
                    MessageMetadata md = Commands.parseMessageMetadata(messageMetadataAndPayload);
                    String producerName = md.getProducerName();
                    long sequenceId = Math.max(md.getHighestSequenceId(), md.getSequenceId());
                    MessageDeduplication.this.highestSequencedPushed.put(producerName, sequenceId);
                    MessageDeduplication.this.highestSequencedPersisted.put(producerName, sequenceId);
                    MessageDeduplication.this.producerRemoved(producerName);
                    entry.release();
                }
                if (MessageDeduplication.this.managedCursor.hasMoreEntries()) {
                    MessageDeduplication.this.pulsar.getExecutor().execute(() -> MessageDeduplication.this.replayCursor(future));
                } else {
                    future.complete(null);
                }
            }

            @Override
            public void readEntriesFailed(ManagedLedgerException exception, Object ctx) {
                future.completeExceptionally(exception);
            }
        }, null, PositionImpl.LATEST);
    }

    public Status getStatus() {
        return this.status;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompletableFuture<Void> checkStatus() {
        boolean shouldBeEnabled = this.isDeduplicationEnabled();
        MessageDeduplication messageDeduplication = this;
        synchronized (messageDeduplication) {
            if (this.status == Status.Recovering || this.status == Status.Removing) {
                this.pulsar.getExecutor().schedule(this::checkStatus, 1L, TimeUnit.MINUTES);
                return CompletableFuture.completedFuture(null);
            }
            if (this.status == Status.Initialized && !shouldBeEnabled) {
                this.status = Status.Removing;
                this.managedLedger.asyncDeleteCursor("pulsar.dedup", new AsyncCallbacks.DeleteCursorCallback(){

                    @Override
                    public void deleteCursorComplete(Object ctx) {
                        MessageDeduplication.this.status = Status.Disabled;
                        log.info("[{}] Deleted deduplication cursor", (Object)MessageDeduplication.this.topic.getName());
                    }

                    @Override
                    public void deleteCursorFailed(ManagedLedgerException exception, Object ctx) {
                        if (exception instanceof ManagedLedgerException.CursorNotFoundException) {
                            MessageDeduplication.this.status = Status.Disabled;
                        } else {
                            log.error("[{}] Deleted deduplication cursor error", (Object)MessageDeduplication.this.topic.getName(), (Object)exception);
                        }
                    }
                }, null);
            }
            if (this.status == Status.Enabled && !shouldBeEnabled) {
                final CompletableFuture<Void> future = new CompletableFuture<Void>();
                this.status = Status.Removing;
                this.managedLedger.asyncDeleteCursor("pulsar.dedup", new AsyncCallbacks.DeleteCursorCallback(){

                    @Override
                    public void deleteCursorComplete(Object ctx) {
                        MessageDeduplication.this.status = Status.Disabled;
                        MessageDeduplication.this.managedCursor = null;
                        MessageDeduplication.this.highestSequencedPushed.clear();
                        MessageDeduplication.this.highestSequencedPersisted.clear();
                        future.complete(null);
                        log.info("[{}] Disabled deduplication", (Object)MessageDeduplication.this.topic.getName());
                    }

                    @Override
                    public void deleteCursorFailed(ManagedLedgerException exception, Object ctx) {
                        if (exception instanceof ManagedLedgerException.CursorNotFoundException) {
                            MessageDeduplication.this.status = Status.Disabled;
                            MessageDeduplication.this.managedCursor = null;
                            MessageDeduplication.this.highestSequencedPushed.clear();
                            MessageDeduplication.this.highestSequencedPersisted.clear();
                            future.complete(null);
                        } else {
                            log.warn("[{}] Failed to disable deduplication: {}", (Object)MessageDeduplication.this.topic.getName(), (Object)exception.getMessage());
                            MessageDeduplication.this.status = Status.Failed;
                            future.completeExceptionally(exception);
                        }
                    }
                }, null);
                return future;
            }
            if ((this.status == Status.Disabled || this.status == Status.Initialized) && shouldBeEnabled) {
                final CompletableFuture<Void> future = new CompletableFuture<Void>();
                this.managedLedger.asyncOpenCursor("pulsar.dedup", new AsyncCallbacks.OpenCursorCallback(){

                    @Override
                    public void openCursorComplete(ManagedCursor cursor, Object ctx) {
                        cursor.setAlwaysInactive();
                        MessageDeduplication.this.managedCursor = cursor;
                        ((CompletableFuture)MessageDeduplication.this.recoverSequenceIdsMap().thenRun(() -> {
                            MessageDeduplication.this.status = Status.Enabled;
                            future.complete(null);
                            log.info("[{}] Enabled deduplication", (Object)MessageDeduplication.this.topic.getName());
                        })).exceptionally(ex -> {
                            MessageDeduplication.this.status = Status.Failed;
                            log.warn("[{}] Failed to enable deduplication: {}", (Object)MessageDeduplication.this.topic.getName(), (Object)ex.getMessage());
                            future.completeExceptionally((Throwable)ex);
                            return null;
                        });
                    }

                    @Override
                    public void openCursorFailed(ManagedLedgerException exception, Object ctx) {
                        log.warn("[{}] Failed to enable deduplication: {}", (Object)MessageDeduplication.this.topic.getName(), (Object)exception.getMessage());
                        future.completeExceptionally(exception);
                    }
                }, null);
                return future;
            }
            return CompletableFuture.completedFuture(null);
        }
    }

    public boolean isEnabled() {
        return this.status == Status.Enabled;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public MessageDupStatus isDuplicate(Topic.PublishContext publishContext, ByteBuf headersAndPayload) {
        if (!this.isEnabled() || publishContext.isMarkerMessage()) {
            return MessageDupStatus.NotDup;
        }
        String producerName = publishContext.getProducerName();
        long sequenceId = publishContext.getSequenceId();
        long highestSequenceId = Math.max(publishContext.getHighestSequenceId(), sequenceId);
        if (producerName.startsWith(this.replicatorPrefix)) {
            int readerIndex = headersAndPayload.readerIndex();
            MessageMetadata md = Commands.parseMessageMetadata(headersAndPayload);
            producerName = md.getProducerName();
            sequenceId = md.getSequenceId();
            highestSequenceId = Math.max(md.getHighestSequenceId(), sequenceId);
            publishContext.setOriginalProducerName(producerName);
            publishContext.setOriginalSequenceId(sequenceId);
            publishContext.setOriginalHighestSequenceId(highestSequenceId);
            headersAndPayload.readerIndex(readerIndex);
        }
        ConcurrentOpenHashMap<String, Long> concurrentOpenHashMap = this.highestSequencedPushed;
        synchronized (concurrentOpenHashMap) {
            Long lastSequenceIdPushed = this.highestSequencedPushed.get(producerName);
            if (lastSequenceIdPushed != null && sequenceId <= lastSequenceIdPushed) {
                Long lastSequenceIdPersisted;
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Message identified as duplicated producer={} seq-id={} -- highest-seq-id={}", new Object[]{this.topic.getName(), producerName, sequenceId, lastSequenceIdPushed});
                }
                if ((lastSequenceIdPersisted = this.highestSequencedPersisted.get(producerName)) != null && sequenceId <= lastSequenceIdPersisted) {
                    return MessageDupStatus.Dup;
                }
                return MessageDupStatus.Unknown;
            }
            this.highestSequencedPushed.put(producerName, highestSequenceId);
        }
        return MessageDupStatus.NotDup;
    }

    public void recordMessagePersisted(Topic.PublishContext publishContext, PositionImpl position) {
        if (!this.isEnabled() || publishContext.isMarkerMessage()) {
            return;
        }
        String producerName = publishContext.getProducerName();
        long sequenceId = publishContext.getSequenceId();
        long highestSequenceId = publishContext.getHighestSequenceId();
        if (publishContext.getOriginalProducerName() != null) {
            producerName = publishContext.getOriginalProducerName();
            sequenceId = publishContext.getOriginalSequenceId();
            highestSequenceId = publishContext.getOriginalHighestSequenceId();
        }
        this.highestSequencedPersisted.put(producerName, Math.max(highestSequenceId, sequenceId));
        if (++this.snapshotCounter >= this.snapshotInterval) {
            this.snapshotCounter = 0;
            this.takeSnapshot(position);
        }
    }

    public void resetHighestSequenceIdPushed() {
        if (!this.isEnabled()) {
            return;
        }
        this.highestSequencedPushed.clear();
        for (String producer : this.highestSequencedPersisted.keys()) {
            this.highestSequencedPushed.put(producer, this.highestSequencedPersisted.get(producer));
        }
    }

    private void takeSnapshot(final Position position) {
        if (log.isDebugEnabled()) {
            log.debug("[{}] Taking snapshot of sequence ids map", (Object)this.topic.getName());
        }
        TreeMap<String, Long> snapshot = new TreeMap<String, Long>();
        this.highestSequencedPersisted.forEach((producerName, sequenceId) -> {
            if (snapshot.size() < this.maxNumberOfProducers) {
                snapshot.put((String)producerName, (Long)sequenceId);
            }
        });
        this.getManagedCursor().asyncMarkDelete(position, snapshot, new AsyncCallbacks.MarkDeleteCallback(){

            @Override
            public void markDeleteComplete(Object ctx) {
                if (log.isDebugEnabled()) {
                    log.debug("[{}] Stored new deduplication snapshot at {}", (Object)MessageDeduplication.this.topic.getName(), (Object)position);
                }
                MessageDeduplication.this.lastSnapshotTimestamp = System.currentTimeMillis();
            }

            @Override
            public void markDeleteFailed(ManagedLedgerException exception, Object ctx) {
                log.warn("[{}] Failed to store new deduplication snapshot at {}", (Object)MessageDeduplication.this.topic.getName(), (Object)position);
            }
        }, null);
    }

    private boolean isDeduplicationEnabled() {
        return this.topic.getHierarchyTopicPolicies().getDeduplicationEnabled().get();
    }

    public void producerAdded(String producerName) {
        this.inactiveProducers.remove(producerName);
    }

    public void producerRemoved(String producerName) {
        this.inactiveProducers.put(producerName, System.currentTimeMillis());
    }

    public synchronized void purgeInactiveProducers() {
        long minimumActiveTimestamp = System.currentTimeMillis() - TimeUnit.MINUTES.toMillis(this.pulsar.getConfiguration().getBrokerDeduplicationProducerInactivityTimeoutMinutes());
        Iterator<Map.Entry<String, Long>> mapIterator = this.inactiveProducers.entrySet().iterator();
        boolean hasInactive = false;
        while (mapIterator.hasNext()) {
            Map.Entry<String, Long> entry = mapIterator.next();
            String producerName = entry.getKey();
            long lastActiveTimestamp = entry.getValue();
            if (lastActiveTimestamp >= minimumActiveTimestamp) continue;
            log.info("[{}] Purging dedup information for producer {}", (Object)this.topic.getName(), (Object)producerName);
            mapIterator.remove();
            this.highestSequencedPushed.remove(producerName);
            this.highestSequencedPersisted.remove(producerName);
            hasInactive = true;
        }
        if (hasInactive && this.isEnabled()) {
            this.takeSnapshot(this.getManagedCursor().getMarkDeletedPosition());
        }
    }

    public long getLastPublishedSequenceId(String producerName) {
        Long sequenceId = this.highestSequencedPushed.get(producerName);
        return sequenceId != null ? sequenceId : -1L;
    }

    public void takeSnapshot() {
        Integer interval = this.topic.getHierarchyTopicPolicies().getDeduplicationSnapshotIntervalSeconds().get();
        long currentTimeStamp = System.currentTimeMillis();
        if (interval == null || interval <= 0 || currentTimeStamp - this.lastSnapshotTimestamp < TimeUnit.SECONDS.toMillis(interval.intValue())) {
            return;
        }
        PositionImpl position = (PositionImpl)this.managedLedger.getLastConfirmedEntry();
        if (position == null) {
            return;
        }
        PositionImpl markDeletedPosition = (PositionImpl)this.managedCursor.getMarkDeletedPosition();
        if (markDeletedPosition != null && position.compareTo(markDeletedPosition) <= 0) {
            return;
        }
        this.takeSnapshot(position);
    }

    @VisibleForTesting
    ManagedCursor getManagedCursor() {
        return this.managedCursor;
    }

    public static class MessageDupUnknownException
    extends RuntimeException {
        public MessageDupUnknownException() {
            super("Cannot determine whether the message is a duplicate at this time");
        }
    }

    @VisibleForTesting
    public static enum MessageDupStatus {
        Unknown,
        NotDup,
        Dup;

    }

    static enum Status {
        Initialized,
        Disabled,
        Recovering,
        Enabled,
        Removing,
        Failed;

    }
}

