/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.connector.pulsar.source.offset;

import org.apache.flink.connector.pulsar.source.AbstractPartition;
import org.apache.flink.connector.pulsar.source.StartOffsetInitializer;

import org.apache.pulsar.client.api.Consumer;
import org.apache.pulsar.client.api.Message;
import org.apache.pulsar.client.api.MessageId;
import org.apache.pulsar.client.api.PulsarClientException;

import java.util.Optional;
import java.util.function.Supplier;

/**
 * An implementation of {@link StartOffsetInitializer} to initialize the offsets
 * based on a timestamp.
 *
 * <p>Should be initialized through {@link StartOffsetInitializer}.
 */
public class TimestampStartOffsetInitializer implements StartOffsetInitializer {
    private static final long serialVersionUID = 2932230571773627233L;
    private final long startingTimestamp;

    public TimestampStartOffsetInitializer(long startingTimestamp) {
        this.startingTimestamp = startingTimestamp;
    }

    @Override
    public void initializeAfterCreation(AbstractPartition partition, Consumer<?> consumer) throws PulsarClientException {
        consumer.seek(startingTimestamp);
    }

    @Override
    public Optional<String> verifyOffset(
            AbstractPartition partition,
            Supplier<Optional<MessageId>> lastMessageIdFetcher,
            Supplier<Optional<Message<byte[]>>> firstMessageFetcher) {
        return firstMessageFetcher.get().isPresent() ?
                Optional.empty() :
                Optional.of(String.format("No data found after timestamp %s", startingTimestamp));
    }
}
