/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.connector.pulsar.source.reader.split;

import org.apache.flink.annotation.Internal;
import org.apache.flink.connector.pulsar.source.config.SourceConfiguration;
import org.apache.flink.connector.pulsar.source.reader.source.PulsarUnorderedSourceReader;
import org.apache.flink.connector.pulsar.source.split.PulsarPartitionSplit;
import org.apache.flink.connector.pulsar.source.split.PulsarPartitionSplitState;

import org.apache.pulsar.client.admin.PulsarAdmin;
import org.apache.pulsar.client.api.Consumer;
import org.apache.pulsar.client.api.CryptoKeyReader;
import org.apache.pulsar.client.api.Message;
import org.apache.pulsar.client.api.PulsarClient;
import org.apache.pulsar.client.api.PulsarClientException;
import org.apache.pulsar.client.api.Schema;
import org.apache.pulsar.client.api.transaction.Transaction;
import org.apache.pulsar.client.api.transaction.TransactionCoordinatorClient;
import org.apache.pulsar.client.api.transaction.TransactionCoordinatorClientException;
import org.apache.pulsar.client.api.transaction.TransactionCoordinatorClientException.TransactionNotFoundException;
import org.apache.pulsar.client.api.transaction.TxnID;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.annotation.Nullable;

import java.time.Duration;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.TimeUnit;

import static org.apache.flink.connector.pulsar.common.utils.PulsarExceptionUtils.sneakyClient;
import static org.apache.flink.connector.pulsar.common.utils.PulsarTransactionUtils.createTransaction;
import static org.apache.flink.connector.pulsar.common.utils.PulsarTransactionUtils.unwrap;

/**
 * The split reader a given {@link PulsarPartitionSplit}, it would be closed once the {@link
 * PulsarUnorderedSourceReader} is closed.
 */
@Internal
public class PulsarUnorderedPartitionSplitReader extends PulsarPartitionSplitReaderBase {
    private static final Logger LOG =
            LoggerFactory.getLogger(PulsarUnorderedPartitionSplitReader.class);

    private final TransactionCoordinatorClient coordinatorClient;

    @Nullable private Transaction uncommittedTransaction;

    public PulsarUnorderedPartitionSplitReader(
            PulsarClient pulsarClient,
            PulsarAdmin pulsarAdmin,
            SourceConfiguration sourceConfiguration,
            Schema<byte[]> schema,
            @Nullable CryptoKeyReader cryptoKeyReader,
            TransactionCoordinatorClient coordinatorClient) {
        super(pulsarClient, pulsarAdmin, sourceConfiguration, schema, cryptoKeyReader);

        this.coordinatorClient = coordinatorClient;
    }

    @Override
    protected Message<byte[]> pollMessage(Duration timeout)
            throws ExecutionException, InterruptedException, PulsarClientException {
        Message<byte[]> message =
                pulsarConsumer.receive(Math.toIntExact(timeout.toMillis()), TimeUnit.MILLISECONDS);

        // Skip the message when receive timeout
        if (message == null) {
            return null;
        }

        if (!sourceConfiguration.isEnableAutoAcknowledgeMessage()) {
            if (uncommittedTransaction == null) {
                // Create a transaction.
                this.uncommittedTransaction = newTransaction();
            }

            try {
                // Add this message into transaction.
                pulsarConsumer
                        .acknowledgeAsync(message.getMessageId(), uncommittedTransaction)
                        .get();
            } catch (InterruptedException e) {
                Thread.currentThread().interrupt();
                throw e;
            }
        }

        return message;
    }

    @Override
    protected void finishedPollMessage(Message<?> message) {
        if (sourceConfiguration.isEnableAutoAcknowledgeMessage()) {
            sneakyClient(() -> pulsarConsumer.acknowledge(message));
        }

        // Release message
        message.release();
    }

    @Override
    protected void startConsumer(PulsarPartitionSplit split, Consumer<?> consumer) {
        TxnID uncommittedTransactionId = split.getUncommittedTransactionId();

        // Abort the uncommitted pulsar transaction.
        if (uncommittedTransactionId != null) {
            if (coordinatorClient != null) {
                try {
                    coordinatorClient.abort(uncommittedTransactionId);
                } catch (TransactionCoordinatorClientException e) {
                    TransactionCoordinatorClientException exception = unwrap(e);
                    // The aborted transaction would return a not found exception.
                    if (!(exception instanceof TransactionNotFoundException)) {
                        LOG.error(
                                "Failed to abort the uncommitted transaction {} when restart the reader",
                                uncommittedTransactionId,
                                e);
                    }
                }
            }

            // Redeliver unacknowledged messages because of the message is out of order.
            consumer.redeliverUnacknowledgedMessages();
        }
    }

    public PulsarPartitionSplitState snapshotState(long checkpointId) {
        PulsarPartitionSplitState state = new PulsarPartitionSplitState(registeredSplit);

        // Avoiding NP problem when Pulsar don't get the message before Flink checkpoint.
        if (uncommittedTransaction != null) {
            TxnID txnID = uncommittedTransaction.getTxnID();
            this.uncommittedTransaction = newTransaction();
            state.setUncommittedTransactionId(txnID);
        }

        return state;
    }

    private Transaction newTransaction() {
        long timeoutMillis = sourceConfiguration.getTransactionTimeoutMillis();
        return createTransaction(pulsarClient, timeoutMillis);
    }
}
