/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.bookkeeper.server.http.service;

import java.util.Iterator;
import lombok.NonNull;
import org.apache.bookkeeper.client.BookKeeperAdmin;
import org.apache.bookkeeper.common.util.JsonUtil;
import org.apache.bookkeeper.http.HttpServer;
import org.apache.bookkeeper.http.service.HttpEndpointService;
import org.apache.bookkeeper.http.service.HttpServiceRequest;
import org.apache.bookkeeper.http.service.HttpServiceResponse;
import org.apache.bookkeeper.meta.LedgerManagerFactory;
import org.apache.bookkeeper.meta.LedgerUnderreplicationManager;
import org.apache.bookkeeper.meta.UnderreplicatedLedger;
import org.apache.bookkeeper.net.BookieId;

/**
 * HttpEndpointService that exposes the current info about the cluster of bookies.
 *
 * <pre>
 * <code>
 * {
 *  "hasAuditorElected" : true,
 *  "auditorId" : "blah",
 *  "hasUnderReplicatedLedgers": false,
 *  "isLedgerReplicationEnabled": true,
 *  "totalBookiesCount": 10,
 *  "writableBookiesCount": 6,
 *  "readonlyBookiesCount": 3,
 *  "unavailableBookiesCount": 1
 * }
 * </code>
 * </pre>
 */
public class ClusterInfoService implements HttpEndpointService {
    private static final org.slf4j.Logger log = org.slf4j.LoggerFactory.getLogger(ClusterInfoService.class);
    @NonNull
    private final BookKeeperAdmin bka;
    @NonNull
    private final LedgerManagerFactory ledgerManagerFactory;


    /**
     * POJO definition for the cluster info response.
     */
    public static class ClusterInfo {
        private boolean auditorElected;
        private String auditorId;
        private boolean clusterUnderReplicated;
        private boolean ledgerReplicationEnabled;
        private int totalBookiesCount;
        private int writableBookiesCount;
        private int readonlyBookiesCount;
        private int unavailableBookiesCount;

        public ClusterInfo() {
        }

        public boolean isAuditorElected() {
            return this.auditorElected;
        }

        public String getAuditorId() {
            return this.auditorId;
        }

        public boolean isClusterUnderReplicated() {
            return this.clusterUnderReplicated;
        }

        public boolean isLedgerReplicationEnabled() {
            return this.ledgerReplicationEnabled;
        }

        public int getTotalBookiesCount() {
            return this.totalBookiesCount;
        }

        public int getWritableBookiesCount() {
            return this.writableBookiesCount;
        }

        public int getReadonlyBookiesCount() {
            return this.readonlyBookiesCount;
        }

        public int getUnavailableBookiesCount() {
            return this.unavailableBookiesCount;
        }

        public void setAuditorElected(final boolean auditorElected) {
            this.auditorElected = auditorElected;
        }

        public void setAuditorId(final String auditorId) {
            this.auditorId = auditorId;
        }

        public void setClusterUnderReplicated(final boolean clusterUnderReplicated) {
            this.clusterUnderReplicated = clusterUnderReplicated;
        }

        public void setLedgerReplicationEnabled(final boolean ledgerReplicationEnabled) {
            this.ledgerReplicationEnabled = ledgerReplicationEnabled;
        }

        public void setTotalBookiesCount(final int totalBookiesCount) {
            this.totalBookiesCount = totalBookiesCount;
        }

        public void setWritableBookiesCount(final int writableBookiesCount) {
            this.writableBookiesCount = writableBookiesCount;
        }

        public void setReadonlyBookiesCount(final int readonlyBookiesCount) {
            this.readonlyBookiesCount = readonlyBookiesCount;
        }

        public void setUnavailableBookiesCount(final int unavailableBookiesCount) {
            this.unavailableBookiesCount = unavailableBookiesCount;
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof ClusterInfoService.ClusterInfo)) return false;
            final ClusterInfoService.ClusterInfo other = (ClusterInfoService.ClusterInfo) o;
            if (!other.canEqual((Object) this)) return false;
            if (this.isAuditorElected() != other.isAuditorElected()) return false;
            if (this.isClusterUnderReplicated() != other.isClusterUnderReplicated()) return false;
            if (this.isLedgerReplicationEnabled() != other.isLedgerReplicationEnabled()) return false;
            if (this.getTotalBookiesCount() != other.getTotalBookiesCount()) return false;
            if (this.getWritableBookiesCount() != other.getWritableBookiesCount()) return false;
            if (this.getReadonlyBookiesCount() != other.getReadonlyBookiesCount()) return false;
            if (this.getUnavailableBookiesCount() != other.getUnavailableBookiesCount()) return false;
            final Object this$auditorId = this.getAuditorId();
            final Object other$auditorId = other.getAuditorId();
            if (this$auditorId == null ? other$auditorId != null : !this$auditorId.equals(other$auditorId)) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof ClusterInfoService.ClusterInfo;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            result = result * PRIME + (this.isAuditorElected() ? 79 : 97);
            result = result * PRIME + (this.isClusterUnderReplicated() ? 79 : 97);
            result = result * PRIME + (this.isLedgerReplicationEnabled() ? 79 : 97);
            result = result * PRIME + this.getTotalBookiesCount();
            result = result * PRIME + this.getWritableBookiesCount();
            result = result * PRIME + this.getReadonlyBookiesCount();
            result = result * PRIME + this.getUnavailableBookiesCount();
            final Object $auditorId = this.getAuditorId();
            result = result * PRIME + ($auditorId == null ? 43 : $auditorId.hashCode());
            return result;
        }

        @Override
        public String toString() {
            return "ClusterInfoService.ClusterInfo(auditorElected=" + this.isAuditorElected() + ", auditorId=" + this.getAuditorId() + ", clusterUnderReplicated=" + this.isClusterUnderReplicated() + ", ledgerReplicationEnabled=" + this.isLedgerReplicationEnabled() + ", totalBookiesCount=" + this.getTotalBookiesCount() + ", writableBookiesCount=" + this.getWritableBookiesCount() + ", readonlyBookiesCount=" + this.getReadonlyBookiesCount() + ", unavailableBookiesCount=" + this.getUnavailableBookiesCount() + ")";
        }
    }

    @Override
    public HttpServiceResponse handle(HttpServiceRequest request) throws Exception {
        final HttpServiceResponse response = new HttpServiceResponse();
        if (HttpServer.Method.GET != request.getMethod()) {
            response.setCode(HttpServer.StatusCode.NOT_FOUND);
            response.setBody("Only GET is supported.");
            return response;
        }
        final ClusterInfo info = new ClusterInfo();
        fillUReplicatedInfo(info);
        fillAuditorInfo(info);
        fillBookiesInfo(info);
        String jsonResponse = JsonUtil.toJson(info);
        response.setBody(jsonResponse);
        response.setCode(HttpServer.StatusCode.OK);
        return response;
    }

    private void fillBookiesInfo(ClusterInfo info) {
        try {
            int totalBookiesCount = bka.getAllBookies().size();
            int writableBookiesCount = bka.getAvailableBookies().size();
            int readonlyBookiesCount = bka.getReadOnlyBookies().size();
            int unavailableBookiesCount = totalBookiesCount - writableBookiesCount - readonlyBookiesCount;
            info.setTotalBookiesCount(totalBookiesCount);
            info.setWritableBookiesCount(writableBookiesCount);
            info.setReadonlyBookiesCount(readonlyBookiesCount);
            info.setUnavailableBookiesCount(unavailableBookiesCount);
        } catch (final java.lang.Throwable $ex) {
            throw lombok.Lombok.sneakyThrow($ex);
        }
    }

    private void fillAuditorInfo(ClusterInfo info) {
        try {
            BookieId currentAuditor = bka.getCurrentAuditor();
            info.setAuditorElected(currentAuditor != null);
            info.setAuditorId(currentAuditor == null ? "" : currentAuditor.getId());
        } catch (Exception e) {
            log.error("Could not get Auditor info", e);
            info.setAuditorElected(false);
            info.setAuditorId("");
        }
    }

    private void fillUReplicatedInfo(ClusterInfo info) {
        try {
            try (LedgerUnderreplicationManager underreplicationManager = ledgerManagerFactory.newLedgerUnderreplicationManager()) {
                Iterator<UnderreplicatedLedger> iter = underreplicationManager.listLedgersToRereplicate(null);
                info.setClusterUnderReplicated(iter.hasNext());
                info.setLedgerReplicationEnabled(underreplicationManager.isLedgerReplicationEnabled());
            }
        } catch (final java.lang.Throwable $ex) {
            throw lombok.Lombok.sneakyThrow($ex);
        }
    }

    public ClusterInfoService(@NonNull final BookKeeperAdmin bka, @NonNull final LedgerManagerFactory ledgerManagerFactory) {
        if (bka == null) {
            throw new NullPointerException("bka is marked non-null but is null");
        }
        if (ledgerManagerFactory == null) {
            throw new NullPointerException("ledgerManagerFactory is marked non-null but is null");
        }
        this.bka = bka;
        this.ledgerManagerFactory = ledgerManagerFactory;
    }
}
