/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.bookkeeper.server.http.service;

import static com.google.common.base.Preconditions.checkNotNull;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.StateManager;
import org.apache.bookkeeper.common.util.JsonUtil;
import org.apache.bookkeeper.http.HttpServer;
import org.apache.bookkeeper.http.service.HttpEndpointService;
import org.apache.bookkeeper.http.service.HttpServiceRequest;
import org.apache.bookkeeper.http.service.HttpServiceResponse;

/**
 * HttpEndpointService that exposes the current state of the bookie.
 *
 * <p>Get the current bookie status:
 *
 * <pre>
 * <code>
 * {
 *  "running" : true,
 *  "readOnly" : false,
 *  "shuttingDown" : false,
 *  "availableForHighPriorityWrites" : true
 * }
 * </code>
 * </pre>
 */
public class BookieStateService implements HttpEndpointService {
    private final Bookie bookie;

    public BookieStateService(Bookie bookie) {
        this.bookie = checkNotNull(bookie);
    }


    /**
     * POJO definition for the bookie state response.
     */
    public static class BookieState {
        private boolean running;
        private boolean readOnly;
        private boolean shuttingDown;
        private boolean availableForHighPriorityWrites;

        public boolean isRunning() {
            return this.running;
        }

        public boolean isReadOnly() {
            return this.readOnly;
        }

        public boolean isShuttingDown() {
            return this.shuttingDown;
        }

        public boolean isAvailableForHighPriorityWrites() {
            return this.availableForHighPriorityWrites;
        }

        public void setRunning(final boolean running) {
            this.running = running;
        }

        public void setReadOnly(final boolean readOnly) {
            this.readOnly = readOnly;
        }

        public void setShuttingDown(final boolean shuttingDown) {
            this.shuttingDown = shuttingDown;
        }

        public void setAvailableForHighPriorityWrites(final boolean availableForHighPriorityWrites) {
            this.availableForHighPriorityWrites = availableForHighPriorityWrites;
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof BookieStateService.BookieState)) return false;
            final BookieStateService.BookieState other = (BookieStateService.BookieState) o;
            if (!other.canEqual((Object) this)) return false;
            if (this.isRunning() != other.isRunning()) return false;
            if (this.isReadOnly() != other.isReadOnly()) return false;
            if (this.isShuttingDown() != other.isShuttingDown()) return false;
            if (this.isAvailableForHighPriorityWrites() != other.isAvailableForHighPriorityWrites()) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof BookieStateService.BookieState;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            result = result * PRIME + (this.isRunning() ? 79 : 97);
            result = result * PRIME + (this.isReadOnly() ? 79 : 97);
            result = result * PRIME + (this.isShuttingDown() ? 79 : 97);
            result = result * PRIME + (this.isAvailableForHighPriorityWrites() ? 79 : 97);
            return result;
        }

        @Override
        public String toString() {
            return "BookieStateService.BookieState(running=" + this.isRunning() + ", readOnly=" + this.isReadOnly() + ", shuttingDown=" + this.isShuttingDown() + ", availableForHighPriorityWrites=" + this.isAvailableForHighPriorityWrites() + ")";
        }

        public BookieState() {
        }
    }

    @Override
    public HttpServiceResponse handle(HttpServiceRequest request) throws Exception {
        HttpServiceResponse response = new HttpServiceResponse();
        if (HttpServer.Method.GET != request.getMethod()) {
            response.setCode(HttpServer.StatusCode.NOT_FOUND);
            response.setBody("Only support GET method to retrieve bookie state.");
            return response;
        }
        StateManager sm = bookie.getStateManager();
        BookieState bs = new BookieState();
        bs.running = sm.isRunning();
        bs.readOnly = sm.isReadOnly();
        bs.shuttingDown = sm.isShuttingDown();
        bs.availableForHighPriorityWrites = sm.isAvailableForHighPriorityWrites();
        String jsonResponse = JsonUtil.toJson(bs);
        response.setBody(jsonResponse);
        response.setCode(HttpServer.StatusCode.OK);
        return response;
    }
}
