/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.bookkeeper.server.http.service;

import lombok.NonNull;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.common.util.JsonUtil;
import org.apache.bookkeeper.http.HttpServer;
import org.apache.bookkeeper.http.service.HttpEndpointService;
import org.apache.bookkeeper.http.service.HttpServiceRequest;
import org.apache.bookkeeper.http.service.HttpServiceResponse;

/**
 * HttpEndpointService that exposes the current info of the bookie.
 *
 * <pre>
 * <code>
 * {
 *  "freeSpace" : 0,
 *  "totalSpace" : 0
 * }
 * </code>
 * </pre>
 */
public class BookieInfoService implements HttpEndpointService {
    @NonNull
    private final Bookie bookie;


    /**
     * POJO definition for the bookie info response.
     */
    public static class BookieInfo {
        private long freeSpace;
        private long totalSpace;

        public long getFreeSpace() {
            return this.freeSpace;
        }

        public long getTotalSpace() {
            return this.totalSpace;
        }

        public void setFreeSpace(final long freeSpace) {
            this.freeSpace = freeSpace;
        }

        public void setTotalSpace(final long totalSpace) {
            this.totalSpace = totalSpace;
        }

        @Override
        public boolean equals(final Object o) {
            if (o == this) return true;
            if (!(o instanceof BookieInfoService.BookieInfo)) return false;
            final BookieInfoService.BookieInfo other = (BookieInfoService.BookieInfo) o;
            if (!other.canEqual((Object) this)) return false;
            if (this.getFreeSpace() != other.getFreeSpace()) return false;
            if (this.getTotalSpace() != other.getTotalSpace()) return false;
            return true;
        }

        protected boolean canEqual(final Object other) {
            return other instanceof BookieInfoService.BookieInfo;
        }

        @Override
        public int hashCode() {
            final int PRIME = 59;
            int result = 1;
            final long $freeSpace = this.getFreeSpace();
            result = result * PRIME + (int) ($freeSpace >>> 32 ^ $freeSpace);
            final long $totalSpace = this.getTotalSpace();
            result = result * PRIME + (int) ($totalSpace >>> 32 ^ $totalSpace);
            return result;
        }

        @Override
        public String toString() {
            return "BookieInfoService.BookieInfo(freeSpace=" + this.getFreeSpace() + ", totalSpace=" + this.getTotalSpace() + ")";
        }

        public BookieInfo() {
        }

        public BookieInfo(final long freeSpace, final long totalSpace) {
            this.freeSpace = freeSpace;
            this.totalSpace = totalSpace;
        }
    }

    @Override
    public HttpServiceResponse handle(HttpServiceRequest request) throws Exception {
        HttpServiceResponse response = new HttpServiceResponse();
        if (HttpServer.Method.GET != request.getMethod()) {
            response.setCode(HttpServer.StatusCode.NOT_FOUND);
            response.setBody("Only GET is supported.");
            return response;
        }
        BookieInfo bi = new BookieInfo(bookie.getTotalFreeSpace(), bookie.getTotalDiskSpace());
        String jsonResponse = JsonUtil.toJson(bi);
        response.setBody(jsonResponse);
        response.setCode(HttpServer.StatusCode.OK);
        return response;
    }

    public BookieInfoService(@NonNull final Bookie bookie) {
        if (bookie == null) {
            throw new NullPointerException("bookie is marked non-null but is null");
        }
        this.bookie = bookie;
    }
}
