/*
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 *
 */
package org.apache.bookkeeper.bookie;

/**
 * This is the garbage collection thread status.
 * It includes what phase GarbageCollection (major/minor), gc counters, last gc time, etc.
 */
public class GarbageCollectionStatus {
    // whether the GC thread is in force GC.
    private boolean forceCompacting;
    // whether the GC thread is in major compacting.
    private boolean majorCompacting;
    // whether the GC thread is in minor compacting.
    private boolean minorCompacting;
    private long lastMajorCompactionTime;
    private long lastMinorCompactionTime;
    private long majorCompactionCounter;
    private long minorCompactionCounter;

    GarbageCollectionStatus(final boolean forceCompacting, final boolean majorCompacting, final boolean minorCompacting, final long lastMajorCompactionTime, final long lastMinorCompactionTime, final long majorCompactionCounter, final long minorCompactionCounter) {
        this.forceCompacting = forceCompacting;
        this.majorCompacting = majorCompacting;
        this.minorCompacting = minorCompacting;
        this.lastMajorCompactionTime = lastMajorCompactionTime;
        this.lastMinorCompactionTime = lastMinorCompactionTime;
        this.majorCompactionCounter = majorCompactionCounter;
        this.minorCompactionCounter = minorCompactionCounter;
    }


    public static class GarbageCollectionStatusBuilder {
        private boolean forceCompacting;
        private boolean majorCompacting;
        private boolean minorCompacting;
        private long lastMajorCompactionTime;
        private long lastMinorCompactionTime;
        private long majorCompactionCounter;
        private long minorCompactionCounter;

        GarbageCollectionStatusBuilder() {
        }

        /**
         * @return {@code this}.
         */
        public GarbageCollectionStatus.GarbageCollectionStatusBuilder forceCompacting(final boolean forceCompacting) {
            this.forceCompacting = forceCompacting;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public GarbageCollectionStatus.GarbageCollectionStatusBuilder majorCompacting(final boolean majorCompacting) {
            this.majorCompacting = majorCompacting;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public GarbageCollectionStatus.GarbageCollectionStatusBuilder minorCompacting(final boolean minorCompacting) {
            this.minorCompacting = minorCompacting;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public GarbageCollectionStatus.GarbageCollectionStatusBuilder lastMajorCompactionTime(final long lastMajorCompactionTime) {
            this.lastMajorCompactionTime = lastMajorCompactionTime;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public GarbageCollectionStatus.GarbageCollectionStatusBuilder lastMinorCompactionTime(final long lastMinorCompactionTime) {
            this.lastMinorCompactionTime = lastMinorCompactionTime;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public GarbageCollectionStatus.GarbageCollectionStatusBuilder majorCompactionCounter(final long majorCompactionCounter) {
            this.majorCompactionCounter = majorCompactionCounter;
            return this;
        }

        /**
         * @return {@code this}.
         */
        public GarbageCollectionStatus.GarbageCollectionStatusBuilder minorCompactionCounter(final long minorCompactionCounter) {
            this.minorCompactionCounter = minorCompactionCounter;
            return this;
        }

        public GarbageCollectionStatus build() {
            return new GarbageCollectionStatus(this.forceCompacting, this.majorCompacting, this.minorCompacting, this.lastMajorCompactionTime, this.lastMinorCompactionTime, this.majorCompactionCounter, this.minorCompactionCounter);
        }

        @Override
        public String toString() {
            return "GarbageCollectionStatus.GarbageCollectionStatusBuilder(forceCompacting=" + this.forceCompacting + ", majorCompacting=" + this.majorCompacting + ", minorCompacting=" + this.minorCompacting + ", lastMajorCompactionTime=" + this.lastMajorCompactionTime + ", lastMinorCompactionTime=" + this.lastMinorCompactionTime + ", majorCompactionCounter=" + this.majorCompactionCounter + ", minorCompactionCounter=" + this.minorCompactionCounter + ")";
        }
    }

    public static GarbageCollectionStatus.GarbageCollectionStatusBuilder builder() {
        return new GarbageCollectionStatus.GarbageCollectionStatusBuilder();
    }

    public void setForceCompacting(final boolean forceCompacting) {
        this.forceCompacting = forceCompacting;
    }

    public void setMajorCompacting(final boolean majorCompacting) {
        this.majorCompacting = majorCompacting;
    }

    public void setMinorCompacting(final boolean minorCompacting) {
        this.minorCompacting = minorCompacting;
    }

    public void setLastMajorCompactionTime(final long lastMajorCompactionTime) {
        this.lastMajorCompactionTime = lastMajorCompactionTime;
    }

    public void setLastMinorCompactionTime(final long lastMinorCompactionTime) {
        this.lastMinorCompactionTime = lastMinorCompactionTime;
    }

    public void setMajorCompactionCounter(final long majorCompactionCounter) {
        this.majorCompactionCounter = majorCompactionCounter;
    }

    public void setMinorCompactionCounter(final long minorCompactionCounter) {
        this.minorCompactionCounter = minorCompactionCounter;
    }

    public boolean isForceCompacting() {
        return this.forceCompacting;
    }

    public boolean isMajorCompacting() {
        return this.majorCompacting;
    }

    public boolean isMinorCompacting() {
        return this.minorCompacting;
    }

    public long getLastMajorCompactionTime() {
        return this.lastMajorCompactionTime;
    }

    public long getLastMinorCompactionTime() {
        return this.lastMinorCompactionTime;
    }

    public long getMajorCompactionCounter() {
        return this.majorCompactionCounter;
    }

    public long getMinorCompactionCounter() {
        return this.minorCompactionCounter;
    }
}
