/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import java.io.File;
import java.io.IOException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadFactory;
import org.apache.bookkeeper.bookie.BookieImpl;
import org.apache.bookkeeper.bookie.LedgerDirsManager;
import org.apache.bookkeeper.bookie.LedgerDirsMonitor;
import org.apache.bookkeeper.common.testing.executors.MockExecutorController;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.conf.TestBKConfiguration;
import org.apache.bookkeeper.stats.Gauge;
import org.apache.bookkeeper.stats.StatsLogger;
import org.apache.bookkeeper.test.TestStatsProvider;
import org.apache.bookkeeper.util.DiskChecker;
import org.apache.bookkeeper.util.IOUtils;
import org.apache.commons.io.FileUtils;
import org.hamcrest.Matcher;
import org.hamcrest.Matchers;
import org.junit.After;
import org.junit.Assert;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.ArgumentMatchers;
import org.mockito.MockedStatic;
import org.mockito.Mockito;
import org.mockito.junit.MockitoJUnitRunner;

@RunWith(value=MockitoJUnitRunner.class)
public class LedgerDirsManagerTest {
    ServerConfiguration conf;
    File curDir;
    LedgerDirsManager dirsManager;
    LedgerDirsMonitor ledgerMonitor;
    MockDiskChecker mockDiskChecker;
    private TestStatsProvider statsProvider;
    private TestStatsProvider.TestStatsLogger statsLogger;
    int diskCheckInterval = 1000;
    float threshold = 0.5f;
    float warnThreshold = 0.5f;
    final List<File> tempDirs = new ArrayList<File>();
    ScheduledExecutorService executor;
    MockExecutorController executorController;
    MockedStatic<Executors> executorsMockedStatic;

    File createTempDir(String prefix, String suffix) throws IOException {
        File dir = IOUtils.createTempDir((String)prefix, (String)suffix);
        this.tempDirs.add(dir);
        return dir;
    }

    @Before
    public void setUp() throws Exception {
        this.executorsMockedStatic = Mockito.mockStatic(Executors.class);
        File tmpDir = this.createTempDir("bkTest", ".dir");
        this.curDir = BookieImpl.getCurrentDirectory((File)tmpDir);
        BookieImpl.checkDirectoryStructure((File)this.curDir);
        this.conf = TestBKConfiguration.newServerConfiguration();
        this.conf.setLedgerDirNames(new String[]{tmpDir.toString()});
        this.conf.setDiskLowWaterMarkUsageThreshold(this.conf.getDiskUsageThreshold());
        this.conf.setDiskCheckInterval(this.diskCheckInterval);
        this.conf.setIsForceGCAllowWhenNoSpace(true);
        this.conf.setMinUsableSizeForEntryLogCreation(Long.MIN_VALUE);
        this.executor = (ScheduledExecutorService)Mockito.mock(ScheduledExecutorService.class);
        this.executorController = new MockExecutorController().controlScheduleAtFixedRate(this.executor, 10);
        this.executorsMockedStatic.when(() -> Executors.newSingleThreadScheduledExecutor((ThreadFactory)ArgumentMatchers.any())).thenReturn((Object)this.executor);
        this.mockDiskChecker = new MockDiskChecker(this.threshold, this.warnThreshold);
        this.statsProvider = new TestStatsProvider();
        this.statsLogger = this.statsProvider.getStatsLogger("test");
        this.dirsManager = new LedgerDirsManager(this.conf, this.conf.getLedgerDirs(), new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()), (StatsLogger)this.statsLogger);
        this.ledgerMonitor = new LedgerDirsMonitor(this.conf, (DiskChecker)this.mockDiskChecker, Collections.singletonList(this.dirsManager));
        this.ledgerMonitor.init();
    }

    @After
    public void tearDown() throws Exception {
        this.executorsMockedStatic.close();
        this.ledgerMonitor.shutdown();
        for (File dir : this.tempDirs) {
            FileUtils.deleteDirectory((File)dir);
        }
        this.tempDirs.clear();
    }

    @Test
    public void testGetWritableDir() throws Exception {
        try {
            List writeDirs = this.dirsManager.getWritableLedgerDirs();
            Assert.assertTrue((String)"Must have a writable ledgerDir", (writeDirs.size() > 0 ? 1 : 0) != 0);
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException nwlde) {
            Assert.fail((String)"We should have a writable ledgerDir");
        }
    }

    @Test
    public void testPickWritableDirExclusive() throws Exception {
        try {
            this.dirsManager.pickRandomWritableDir(this.curDir);
            Assert.fail((String)"Should not reach here due to there is no writable ledger dir.");
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException nwlde) {
            Assert.assertTrue((boolean)true);
        }
    }

    @Test
    public void testNoWritableDir() throws Exception {
        try {
            this.dirsManager.addToFilledDirs(this.curDir);
            this.dirsManager.pickRandomWritableDir();
            Assert.fail((String)"Should not reach here due to there is no writable ledger dir.");
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException nwlde) {
            Assert.assertEquals((String)"Should got NoWritableLedgerDirException w/ 'All ledger directories are non writable'.", (Object)"All ledger directories are non writable", (Object)nwlde.getMessage());
        }
    }

    @Test
    public void testGetWritableDirForLog() throws Exception {
        try {
            this.dirsManager.addToFilledDirs(this.curDir);
            this.dirsManager.getWritableLedgerDirs();
            Assert.fail((String)"Should not reach here due to there is no writable ledger dir.");
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException nwlde) {
            try {
                List writeDirs = this.dirsManager.getWritableLedgerDirsForNewLog();
                Assert.assertTrue((String)"Must have a writable ledgerDir", (writeDirs.size() > 0 ? 1 : 0) != 0);
            }
            catch (LedgerDirsManager.NoWritableLedgerDirException e) {
                Assert.fail((String)"We should have a writeble ledgerDir");
            }
        }
    }

    @Test
    public void testGetWritableDirForLogNoEnoughDiskSpace() throws Exception {
        this.conf.setMinUsableSizeForEntryLogCreation(this.curDir.getUsableSpace() + 1024L);
        this.dirsManager = new LedgerDirsManager(this.conf, this.conf.getLedgerDirs(), new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()), (StatsLogger)this.statsLogger);
        try {
            this.dirsManager.addToFilledDirs(this.curDir);
            this.dirsManager.getWritableLedgerDirs();
            Assert.fail((String)"Should not reach here due to there is no writable ledger dir.");
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException nwlde) {
            try {
                this.dirsManager.getWritableLedgerDirsForNewLog();
                Assert.fail((String)"Should not reach here due to there is no enough disk space left");
            }
            catch (LedgerDirsManager.NoWritableLedgerDirException noWritableLedgerDirException) {
                // empty catch block
            }
        }
    }

    @Test
    public void testLedgerDirsMonitorDuringTransition() throws Exception {
        this.testLedgerDirsMonitorDuringTransition(true);
    }

    @Test
    public void testHighPriorityWritesDisallowedDuringTransition() throws Exception {
        this.testLedgerDirsMonitorDuringTransition(false);
    }

    private void testLedgerDirsMonitorDuringTransition(boolean highPriorityWritesAllowed) throws Exception {
        if (!highPriorityWritesAllowed) {
            this.ledgerMonitor.shutdown();
            this.conf.setMinUsableSizeForHighPriorityWrites(this.curDir.getUsableSpace() + 1024L);
            this.dirsManager = new LedgerDirsManager(this.conf, this.conf.getLedgerDirs(), new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()), (StatsLogger)this.statsLogger);
            this.ledgerMonitor = new LedgerDirsMonitor(this.conf, (DiskChecker)this.mockDiskChecker, Collections.singletonList(this.dirsManager));
            this.ledgerMonitor.init();
        }
        MockLedgerDirsListener mockLedgerDirsListener = new MockLedgerDirsListener();
        this.dirsManager.addLedgerDirsListener((LedgerDirsManager.LedgerDirsListener)mockLedgerDirsListener);
        this.ledgerMonitor.start();
        Assert.assertFalse((boolean)mockLedgerDirsListener.readOnly);
        Assert.assertTrue((boolean)mockLedgerDirsListener.highPriorityWritesAllowed);
        this.mockDiskChecker.setUsage(this.threshold + 0.05f);
        this.executorController.advance(Duration.ofMillis(this.diskCheckInterval));
        Assert.assertTrue((boolean)mockLedgerDirsListener.readOnly);
        Assert.assertEquals((Object)highPriorityWritesAllowed, (Object)mockLedgerDirsListener.highPriorityWritesAllowed);
        this.mockDiskChecker.setUsage(this.threshold - 0.05f);
        this.executorController.advance(Duration.ofMillis(this.diskCheckInterval));
        Assert.assertFalse((boolean)mockLedgerDirsListener.readOnly);
        Assert.assertTrue((boolean)mockLedgerDirsListener.highPriorityWritesAllowed);
    }

    @Test
    public void testIsReadOnlyModeOnAnyDiskFullEnabled() throws Exception {
        this.testAnyLedgerFullTransitToReadOnly(true);
        this.testAnyLedgerFullTransitToReadOnly(false);
    }

    public void testAnyLedgerFullTransitToReadOnly(boolean isReadOnlyModeOnAnyDiskFullEnabled) throws Exception {
        this.ledgerMonitor.shutdown();
        float nospace = 0.9f;
        float lwm = 0.8f;
        File tmpDir1 = this.createTempDir("bkTest", ".dir");
        File curDir1 = BookieImpl.getCurrentDirectory((File)tmpDir1);
        BookieImpl.checkDirectoryStructure((File)curDir1);
        File tmpDir2 = this.createTempDir("bkTest", ".dir");
        File curDir2 = BookieImpl.getCurrentDirectory((File)tmpDir2);
        BookieImpl.checkDirectoryStructure((File)curDir2);
        this.conf.setDiskUsageThreshold(0.9f);
        this.conf.setDiskLowWaterMarkUsageThreshold(0.8f);
        this.conf.setDiskUsageWarnThreshold(0.9f);
        this.conf.setReadOnlyModeOnAnyDiskFullEnabled(isReadOnlyModeOnAnyDiskFullEnabled);
        this.conf.setLedgerDirNames(new String[]{tmpDir1.toString(), tmpDir2.toString()});
        this.mockDiskChecker = new MockDiskChecker(0.9f, this.warnThreshold);
        this.dirsManager = new LedgerDirsManager(this.conf, this.conf.getLedgerDirs(), new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()), (StatsLogger)this.statsLogger);
        this.ledgerMonitor = new LedgerDirsMonitor(this.conf, (DiskChecker)this.mockDiskChecker, Collections.singletonList(this.dirsManager));
        HashMap<File, Float> usageMap = new HashMap<File, Float>();
        usageMap.put(curDir1, Float.valueOf(0.1f));
        usageMap.put(curDir2, Float.valueOf(0.1f));
        this.mockDiskChecker.setUsageMap(usageMap);
        this.ledgerMonitor.init();
        MockLedgerDirsListener mockLedgerDirsListener = new MockLedgerDirsListener();
        this.dirsManager.addLedgerDirsListener((LedgerDirsManager.LedgerDirsListener)mockLedgerDirsListener);
        this.ledgerMonitor.start();
        Thread.sleep(this.diskCheckInterval * 2 + 100);
        Assert.assertFalse((boolean)mockLedgerDirsListener.readOnly);
        if (isReadOnlyModeOnAnyDiskFullEnabled) {
            this.setUsageAndThenVerify(curDir1, 0.1f, curDir2, 0.95f, this.mockDiskChecker, mockLedgerDirsListener, true);
            this.setUsageAndThenVerify(curDir1, 0.95f, curDir2, 0.1f, this.mockDiskChecker, mockLedgerDirsListener, true);
            this.setUsageAndThenVerify(curDir1, 0.95f, curDir2, 0.95f, this.mockDiskChecker, mockLedgerDirsListener, true);
            this.setUsageAndThenVerify(curDir1, 0.59999996f, curDir2, 0.95f, this.mockDiskChecker, mockLedgerDirsListener, true);
            this.setUsageAndThenVerify(curDir1, 0.7f, curDir2, 0.7f, this.mockDiskChecker, mockLedgerDirsListener, false);
        } else {
            this.setUsageAndThenVerify(curDir1, 0.1f, curDir2, 0.1f, this.mockDiskChecker, mockLedgerDirsListener, false);
            this.setUsageAndThenVerify(curDir1, 0.1f, curDir2, 0.95f, this.mockDiskChecker, mockLedgerDirsListener, false);
            this.setUsageAndThenVerify(curDir1, 0.95f, curDir2, 0.1f, this.mockDiskChecker, mockLedgerDirsListener, false);
            this.setUsageAndThenVerify(curDir1, 0.95f, curDir2, 0.95f, this.mockDiskChecker, mockLedgerDirsListener, true);
            this.setUsageAndThenVerify(curDir1, 0.59999996f, curDir2, 0.95f, this.mockDiskChecker, mockLedgerDirsListener, false);
            this.setUsageAndThenVerify(curDir1, 0.7f, curDir2, 0.7f, this.mockDiskChecker, mockLedgerDirsListener, false);
        }
    }

    @Test
    public void testLedgerDirsMonitorHandlingLowWaterMark() throws Exception {
        this.ledgerMonitor.shutdown();
        float warn = 0.9f;
        float nospace = 0.98f;
        float lwm = 0.94f;
        float lwm2warn = 0.91999996f;
        float lwm2nospace = 0.96000004f;
        float nospaceExceeded = 0.985f;
        this.conf.setDiskUsageThreshold(0.98f);
        this.conf.setDiskLowWaterMarkUsageThreshold(0.94f);
        this.conf.setDiskUsageWarnThreshold(0.9f);
        this.mockDiskChecker = new MockDiskChecker(0.98f, this.warnThreshold);
        this.dirsManager = new LedgerDirsManager(this.conf, this.conf.getLedgerDirs(), new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()));
        this.ledgerMonitor = new LedgerDirsMonitor(this.conf, (DiskChecker)this.mockDiskChecker, Collections.singletonList(this.dirsManager));
        this.ledgerMonitor.init();
        MockLedgerDirsListener mockLedgerDirsListener = new MockLedgerDirsListener();
        this.dirsManager.addLedgerDirsListener((LedgerDirsManager.LedgerDirsListener)mockLedgerDirsListener);
        this.ledgerMonitor.start();
        this.executorController.advance(Duration.ofMillis(this.diskCheckInterval));
        Assert.assertFalse((boolean)mockLedgerDirsListener.readOnly);
        this.mockDiskChecker.setUsage(0.96000004f);
        this.executorController.advance(Duration.ofMillis(this.diskCheckInterval));
        Assert.assertFalse((boolean)mockLedgerDirsListener.readOnly);
        this.mockDiskChecker.setUsage(0.985f);
        this.executorController.advance(Duration.ofMillis(this.diskCheckInterval));
        Assert.assertTrue((boolean)mockLedgerDirsListener.readOnly);
        this.mockDiskChecker.setUsage(0.96000004f);
        this.executorController.advance(Duration.ofMillis(this.diskCheckInterval));
        Assert.assertTrue((boolean)mockLedgerDirsListener.readOnly);
        this.mockDiskChecker.setUsage(0.91999996f);
        this.executorController.advance(Duration.ofMillis(this.diskCheckInterval));
        Assert.assertFalse((boolean)mockLedgerDirsListener.readOnly);
        this.mockDiskChecker.setUsage(0.96000004f);
        this.executorController.advance(Duration.ofMillis(this.diskCheckInterval));
        Assert.assertFalse((boolean)mockLedgerDirsListener.readOnly);
    }

    @Test
    public void testLedgerDirsMonitorHandlingWithMultipleLedgerDirectories() throws Exception {
        this.ledgerMonitor.shutdown();
        float nospace = 0.9f;
        float lwm = 0.8f;
        File tmpDir1 = this.createTempDir("bkTest", ".dir");
        File curDir1 = BookieImpl.getCurrentDirectory((File)tmpDir1);
        BookieImpl.checkDirectoryStructure((File)curDir1);
        File tmpDir2 = this.createTempDir("bkTest", ".dir");
        File curDir2 = BookieImpl.getCurrentDirectory((File)tmpDir2);
        BookieImpl.checkDirectoryStructure((File)curDir2);
        this.conf.setDiskUsageThreshold(0.9f);
        this.conf.setDiskLowWaterMarkUsageThreshold(0.8f);
        this.conf.setDiskUsageWarnThreshold(0.9f);
        this.conf.setLedgerDirNames(new String[]{tmpDir1.toString(), tmpDir2.toString()});
        this.mockDiskChecker = new MockDiskChecker(0.9f, this.warnThreshold);
        this.dirsManager = new LedgerDirsManager(this.conf, this.conf.getLedgerDirs(), new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()), (StatsLogger)this.statsLogger);
        this.ledgerMonitor = new LedgerDirsMonitor(this.conf, (DiskChecker)this.mockDiskChecker, Collections.singletonList(this.dirsManager));
        HashMap<File, Float> usageMap = new HashMap<File, Float>();
        usageMap.put(curDir1, Float.valueOf(0.1f));
        usageMap.put(curDir2, Float.valueOf(0.1f));
        this.mockDiskChecker.setUsageMap(usageMap);
        this.ledgerMonitor.init();
        MockLedgerDirsListener mockLedgerDirsListener = new MockLedgerDirsListener();
        this.dirsManager.addLedgerDirsListener((LedgerDirsManager.LedgerDirsListener)mockLedgerDirsListener);
        this.ledgerMonitor.start();
        Thread.sleep(this.diskCheckInterval * 2 + 100);
        Assert.assertFalse((boolean)mockLedgerDirsListener.readOnly);
        this.setUsageAndThenVerify(curDir1, 0.85f, curDir2, 0.85f, this.mockDiskChecker, mockLedgerDirsListener, false);
        this.setUsageAndThenVerify(curDir1, 0.91999996f, curDir2, 0.84999996f, this.mockDiskChecker, mockLedgerDirsListener, false);
        this.setUsageAndThenVerify(curDir1, 0.95f, curDir2, 0.91999996f, this.mockDiskChecker, mockLedgerDirsListener, true);
        this.setUsageAndThenVerify(curDir1, 0.84999996f, curDir2, 0.84999996f, this.mockDiskChecker, mockLedgerDirsListener, true);
        this.setUsageAndThenVerify(curDir1, 0.77000004f, curDir2, 0.87f, this.mockDiskChecker, mockLedgerDirsListener, true);
        this.setUsageAndThenVerify(curDir1, 0.63f, curDir2, 0.92999995f, this.mockDiskChecker, mockLedgerDirsListener, false);
        Assert.assertEquals((String)"Only one LedgerDir should be writable", (long)1L, (long)this.dirsManager.getWritableLedgerDirs().size());
        this.setUsageAndThenVerify(curDir1, 0.77000004f, curDir2, 0.78000003f, this.mockDiskChecker, mockLedgerDirsListener, false);
        Assert.assertEquals((String)"Both the LedgerDirs should be writable", (long)2L, (long)this.dirsManager.getWritableLedgerDirs().size());
        this.setUsageAndThenVerify(curDir1, 0.82f, curDir2, 0.88f, this.mockDiskChecker, mockLedgerDirsListener, false);
    }

    @Test
    public void testLedgerDirsMonitorStartReadOnly() throws Exception {
        this.ledgerMonitor.shutdown();
        float nospace = 0.9f;
        float lwm = 0.8f;
        File tmpDir1 = this.createTempDir("bkTest", ".dir");
        File curDir1 = BookieImpl.getCurrentDirectory((File)tmpDir1);
        BookieImpl.checkDirectoryStructure((File)curDir1);
        File tmpDir2 = this.createTempDir("bkTest", ".dir");
        File curDir2 = BookieImpl.getCurrentDirectory((File)tmpDir2);
        BookieImpl.checkDirectoryStructure((File)curDir2);
        this.conf.setDiskUsageThreshold(0.9f);
        this.conf.setDiskLowWaterMarkUsageThreshold(0.8f);
        this.conf.setDiskUsageWarnThreshold(0.9f);
        this.conf.setLedgerDirNames(new String[]{tmpDir1.toString(), tmpDir2.toString()});
        HashMap<File, Float> usageMap = new HashMap<File, Float>();
        usageMap.put(curDir1, Float.valueOf(0.95f));
        usageMap.put(curDir2, Float.valueOf(0.95f));
        this.mockDiskChecker = new MockDiskChecker(0.9f, this.warnThreshold);
        this.mockDiskChecker.setUsageMap(usageMap);
        this.dirsManager = new LedgerDirsManager(this.conf, this.conf.getLedgerDirs(), new DiskChecker(this.conf.getDiskUsageThreshold(), this.conf.getDiskUsageWarnThreshold()), (StatsLogger)this.statsLogger);
        this.ledgerMonitor = new LedgerDirsMonitor(this.conf, (DiskChecker)this.mockDiskChecker, Collections.singletonList(this.dirsManager));
        try {
            this.ledgerMonitor.init();
            Assert.fail((String)"NoWritableLedgerDirException expected");
        }
        catch (LedgerDirsManager.NoWritableLedgerDirException noWritableLedgerDirException) {
            // empty catch block
        }
        MockLedgerDirsListener mockLedgerDirsListener = new MockLedgerDirsListener();
        this.dirsManager.addLedgerDirsListener((LedgerDirsManager.LedgerDirsListener)mockLedgerDirsListener);
        this.ledgerMonitor.start();
        Thread.sleep(this.diskCheckInterval * 2 + 100);
        this.verifyUsage(curDir1, 0.95f, curDir2, 0.95f, mockLedgerDirsListener, true);
    }

    @Test
    public void testValidateLwmThreshold() {
        ServerConfiguration configuration = TestBKConfiguration.newServerConfiguration();
        configuration.setDiskUsageThreshold(0.65f);
        configuration.setDiskLowWaterMarkUsageThreshold(0.9f);
        try {
            new LedgerDirsMonitor(configuration, (DiskChecker)this.mockDiskChecker, Collections.singletonList(this.dirsManager));
            Assert.fail((String)"diskSpaceThreshold < diskSpaceLwmThreshold, should be failed.");
        }
        catch (Exception e) {
            Assert.assertTrue((boolean)e.getMessage().contains("diskSpaceThreshold >= diskSpaceLwmThreshold"));
        }
        configuration.setDiskUsageThreshold(0.0f);
        configuration.setDiskLowWaterMarkUsageThreshold(1.0f);
        try {
            new LedgerDirsMonitor(configuration, (DiskChecker)this.mockDiskChecker, Collections.singletonList(this.dirsManager));
            Assert.fail((String)"diskSpaceThreshold = 0 and diskUsageLwmThreshold = 1, should be failed.");
        }
        catch (Exception e) {
            Assert.assertTrue((boolean)e.getMessage().contains("Should be > 0 and < 1"));
        }
        configuration.setDiskUsageThreshold(0.95f);
        configuration.setDiskLowWaterMarkUsageThreshold(0.9f);
        new LedgerDirsMonitor(configuration, (DiskChecker)this.mockDiskChecker, Collections.singletonList(this.dirsManager));
    }

    private void setUsageAndThenVerify(File dir1, float dir1Usage, File dir2, float dir2Usage, MockDiskChecker mockDiskChecker, MockLedgerDirsListener mockLedgerDirsListener, boolean verifyReadOnly) throws InterruptedException {
        HashMap<File, Float> usageMap = new HashMap<File, Float>();
        usageMap.put(dir1, Float.valueOf(dir1Usage));
        usageMap.put(dir2, Float.valueOf(dir2Usage));
        mockDiskChecker.setUsageMap(usageMap);
        this.verifyUsage(dir1, dir1Usage, dir2, dir2Usage, mockLedgerDirsListener, verifyReadOnly);
    }

    private void verifyUsage(File dir1, float dir1Usage, File dir2, float dir2Usage, MockLedgerDirsListener mockLedgerDirsListener, boolean verifyReadOnly) {
        this.executorController.advance(Duration.ofMillis(this.diskCheckInterval));
        float sample1 = this.getGauge(dir1.getParent()).getSample().floatValue();
        float sample2 = this.getGauge(dir2.getParent()).getSample().floatValue();
        Assert.assertEquals((Object)mockLedgerDirsListener.readOnly, (Object)verifyReadOnly);
        Assert.assertThat((Object)Float.valueOf(sample1), (Matcher)Matchers.equalTo((Object)Float.valueOf(dir1Usage * 100.0f)));
        Assert.assertThat((Object)Float.valueOf(sample2), (Matcher)Matchers.equalTo((Object)Float.valueOf(dir2Usage * 100.0f)));
    }

    private Gauge<? extends Number> getGauge(String path) {
        String gaugeName = String.format("test.dir_%s_usage", path.replace('/', '_'));
        return this.statsProvider.getGauge(gaugeName);
    }

    private class MockLedgerDirsListener
    implements LedgerDirsManager.LedgerDirsListener {
        public volatile boolean highPriorityWritesAllowed;
        public volatile boolean readOnly;

        public MockLedgerDirsListener() {
            this.reset();
        }

        public void diskWritable(File disk) {
            if (LedgerDirsManagerTest.this.conf.isReadOnlyModeOnAnyDiskFullEnabled()) {
                return;
            }
            this.readOnly = false;
            this.highPriorityWritesAllowed = true;
        }

        public void diskJustWritable(File disk) {
            if (LedgerDirsManagerTest.this.conf.isReadOnlyModeOnAnyDiskFullEnabled()) {
                return;
            }
            this.readOnly = false;
            this.highPriorityWritesAllowed = true;
        }

        public void allDisksFull(boolean highPriorityWritesAllowed) {
            this.readOnly = true;
            this.highPriorityWritesAllowed = highPriorityWritesAllowed;
        }

        public void anyDiskFull(boolean highPriorityWritesAllowed) {
            if (LedgerDirsManagerTest.this.conf.isReadOnlyModeOnAnyDiskFullEnabled()) {
                this.readOnly = true;
                this.highPriorityWritesAllowed = highPriorityWritesAllowed;
            }
        }

        public void allDisksWritable() {
            this.readOnly = false;
            this.highPriorityWritesAllowed = true;
        }

        public void reset() {
            this.readOnly = false;
            this.highPriorityWritesAllowed = true;
        }
    }

    private class MockDiskChecker
    extends DiskChecker {
        private volatile float used;
        private volatile Map<File, Float> usageMap;

        public MockDiskChecker(float threshold, float warnThreshold) {
            super(threshold, warnThreshold);
            this.usageMap = null;
            this.used = 0.0f;
        }

        public float checkDir(File dir) throws DiskChecker.DiskErrorException, DiskChecker.DiskOutOfSpaceException, DiskChecker.DiskWarnThresholdException {
            float dirUsage = this.getDirUsage(dir);
            if (dirUsage > this.getDiskUsageThreshold()) {
                throw new DiskChecker.DiskOutOfSpaceException("", dirUsage);
            }
            if (dirUsage > this.getDiskUsageWarnThreshold()) {
                throw new DiskChecker.DiskWarnThresholdException("", dirUsage);
            }
            return dirUsage;
        }

        public float getTotalDiskUsage(List<File> dirs) {
            float accumulatedDiskUsage = 0.0f;
            for (File dir : dirs) {
                accumulatedDiskUsage += this.getDirUsage(dir);
            }
            return accumulatedDiskUsage / (float)dirs.size();
        }

        public float getDirUsage(File dir) {
            float dirUsage = this.usageMap == null || !this.usageMap.containsKey(dir) ? this.used : this.usageMap.get(dir).floatValue();
            return dirUsage;
        }

        public void setUsage(float usage) {
            this.used = usage;
        }

        public void setUsageMap(Map<File, Float> usageMap) {
            this.usageMap = usageMap;
        }
    }
}

