/*
 * Copyright (c) 2010-2015 Pivotal Software, Inc. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you
 * may not use this file except in compliance with the License. You
 * may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or
 * implied. See the License for the specific language governing
 * permissions and limitations under the License. See accompanying
 * LICENSE file.
 */

package com.pivotal.gemfirexd.internal.engine.sql.execute;

import java.sql.Timestamp;
import java.util.ArrayList;

import com.gemstone.gemfire.internal.cache.TXState;
import com.gemstone.gnu.trove.TIntArrayList;
import com.gemstone.gnu.trove.TIntIntHashMap;
import com.gemstone.gnu.trove.TLongArrayList;
import com.pivotal.gemfirexd.internal.catalog.UUID;
import com.pivotal.gemfirexd.internal.engine.access.GemFireTransaction;
import com.pivotal.gemfirexd.internal.engine.sql.catalog.ExtraTableInfo;
import com.pivotal.gemfirexd.internal.engine.store.GemFireContainer;
import com.pivotal.gemfirexd.internal.iapi.error.StandardException;
import com.pivotal.gemfirexd.internal.iapi.reference.SQLState;
import com.pivotal.gemfirexd.internal.iapi.services.cache.ClassSize;
import com.pivotal.gemfirexd.internal.iapi.services.io.FormatableBitSet;
import com.pivotal.gemfirexd.internal.iapi.services.sanity.SanityManager;
import com.pivotal.gemfirexd.internal.iapi.sql.Activation;
import com.pivotal.gemfirexd.internal.iapi.sql.ResultSet;
import com.pivotal.gemfirexd.internal.iapi.sql.Row;
import com.pivotal.gemfirexd.internal.iapi.sql.dictionary.ColumnDescriptor;
import com.pivotal.gemfirexd.internal.iapi.sql.execute.CursorResultSet;
import com.pivotal.gemfirexd.internal.iapi.sql.execute.ExecRow;
import com.pivotal.gemfirexd.internal.iapi.sql.execute.NoPutResultSet;
import com.pivotal.gemfirexd.internal.iapi.sql.execute.TargetResultSet;
import com.pivotal.gemfirexd.internal.iapi.store.access.RowUtil;
import com.pivotal.gemfirexd.internal.iapi.store.access.TransactionController;
import com.pivotal.gemfirexd.internal.iapi.types.DataValueDescriptor;
import com.pivotal.gemfirexd.internal.iapi.types.RowLocation;
import com.pivotal.gemfirexd.internal.iapi.types.SQLInteger;
import com.pivotal.gemfirexd.internal.iapi.types.SQLLongint;
import com.pivotal.gemfirexd.internal.impl.sql.execute.PlanUtils;
import com.pivotal.gemfirexd.internal.impl.sql.execute.PlanUtils.Context;
import com.pivotal.gemfirexd.internal.impl.sql.execute.ResultSetStatisticsVisitor;
import com.pivotal.gemfirexd.internal.impl.sql.execute.ValueRow;
import com.pivotal.gemfirexd.internal.shared.common.StoredFormatIds;

/**
 * Temporary holder for autogenerated keys.
 * 
 * @author swale
 * @since 7.0
 */
public final class AutogenKeysResultSet implements ResultSet, NoPutResultSet {

  private final Activation activation;

  private final TIntIntHashMap columnMap;

  private final TLongArrayList singleUUIDList;

  private final ArrayList<long[]> uuidList;

  private final TIntArrayList singleShortUUIDList;

  private final ArrayList<int[]> shortUUIDList;

  private final int nUUIDs;

  private final int nShortUUIDs;

  private boolean isOpen;

  private int iterPos;

  private int numRows;

  private final ValueRow currentRow;

  static final int BASE_MEMORY_USAGE = ClassSize
      .estimateBaseFromCatalog(AutogenKeysResultSet.class);

  public AutogenKeysResultSet(final Activation activation,
      final int[] autoGenColumnPositions, final ExtraTableInfo tableInfo) {
    this.activation = activation;
    final int numAutoGenKeys = autoGenColumnPositions.length;
    this.columnMap = new TIntIntHashMap(numAutoGenKeys);
    this.currentRow = new ValueRow(numAutoGenKeys);
    ColumnDescriptor cd;
    int rowPos = 0;
    int numUUIDs = 0;
    int numShortUUIDs = 0;
    int typeId;
    for (int pos : autoGenColumnPositions) {
      cd = tableInfo.getAutoGeneratedColumn(pos);
      rowPos++;
      if ((typeId = cd.getType().getDVDTypeFormatId()) ==
          StoredFormatIds.SQL_LONGINT_ID) {
        numUUIDs++;
        this.columnMap.put(rowPos, numUUIDs);
        this.currentRow.setColumn(rowPos, new SQLLongint());
      }
      else {
        if (typeId != StoredFormatIds.SQL_INTEGER_ID) {
          SanityManager.THROWASSERT("unexpected type of autogenerated column "
              + "(expected BIGINT or INT): " + cd);
        }
        numShortUUIDs++;
        this.columnMap.put(rowPos, -numShortUUIDs);
        this.currentRow.setColumn(rowPos, new SQLInteger());
      }
    }
    if (numUUIDs > 0) {
      if (numUUIDs == 1) {
        this.singleUUIDList = new TLongArrayList();
        this.uuidList = null;
      }
      else {
        this.uuidList = new ArrayList<long[]>();
        this.singleUUIDList = null;
      }
    }
    else {
      this.singleUUIDList = null;
      this.uuidList = null;
    }
    if (numShortUUIDs > 0) {
      if (numShortUUIDs == 1) {
        this.singleShortUUIDList = new TIntArrayList();
        this.shortUUIDList = null;
      }
      else {
        this.shortUUIDList = new ArrayList<int[]>();
        this.singleShortUUIDList = null;
      }
    }
    else {
      this.singleShortUUIDList = null;
      this.shortUUIDList = null;
    }
    this.nUUIDs = numUUIDs;
    this.nShortUUIDs = numShortUUIDs;
  }

  public void insertRow(final ExecRow row, final int[] autoGenColumnPositions)
      throws StandardException {
    final long[] uuids;
    final int[] shortUUIDs;
    if (this.nUUIDs <= 1) {
      uuids = null;
    }
    else {
      uuids = new long[this.nUUIDs];
      this.uuidList.add(uuids);
    }
    if (this.nShortUUIDs <= 1) {
      shortUUIDs = null;
    }
    else {
      shortUUIDs = new int[this.nShortUUIDs];
      this.shortUUIDList.add(shortUUIDs);
    }

    int uuidIndex = 0;
    int shortUUIDIndex = 0;
    DataValueDescriptor dvd;
    int typeId;
    for (int pos : autoGenColumnPositions) {
      dvd = row.getColumn(pos);
      if ((typeId = dvd.getTypeFormatId()) == StoredFormatIds.SQL_LONGINT_ID) {
        if (uuids == null) {
          this.singleUUIDList.add(dvd.getLong());
        }
        else {
          uuids[uuidIndex++] = dvd.getLong();
        }
      }
      else {
        assert typeId == StoredFormatIds.SQL_INTEGER_ID: "unexpected type of "
            + "autogenerated column (expected BIGINT or INT) " + typeId + ": "
            + dvd;
        if (shortUUIDs == null) {
          this.singleShortUUIDList.add(dvd.getInt());
        }
        else {
          shortUUIDs[shortUUIDIndex++] = dvd.getInt();
        }
      }
    }
    this.numRows++;
  }

  public long estimateMemoryUsage() {
    long memory = BASE_MEMORY_USAGE;
    if (this.numRows > 0) {
      int rowSize = this.nUUIDs > 1 ? (4 * this.nUUIDs)
          + ClassSize.estimateArrayOverhead() : (this.nUUIDs == 1 ? 4 : 0);
      rowSize += this.nShortUUIDs > 1 ? (2 * this.nShortUUIDs)
          + ClassSize.estimateArrayOverhead() : (this.nShortUUIDs == 1 ? 2 : 0);
      memory += (this.numRows * rowSize);
    }
    return memory;
  }

  public int numRows() {
    return this.numRows;
  }

  /////////////////////////////////////////////////////////
  //
  // NoPutResultSet
  //
  /////////////////////////////////////////////////////////

  /**
   * Open the scan and evaluate qualifiers and the like.
   */
  public void openCore() throws StandardException {
    this.isOpen = true;
  }

  /**
   * @see NoPutResultSet#reopenCore()
   */
  public void reopenCore() throws StandardException {
    this.isOpen = true;
  }

  /**
   * Get the next row.
   * 
   * @return the next row, or null if none
   * 
   * @exception StandardException
   *              on error
   */
  public ExecRow getNextRowCore() throws StandardException {
    if (this.isOpen) {
      if (this.iterPos < this.numRows) {
        int listPos;
        final long[] uuids = this.uuidList == null ? null : this.uuidList
            .get(this.iterPos);
        final int[] shortUUIDs = this.shortUUIDList == null ? null
            : this.shortUUIDList.get(this.iterPos);
        for (int colPos = 1; colPos <= this.currentRow.nColumns(); colPos++) {
          listPos = this.columnMap.get(colPos);
          if (listPos > 0) {
            if (this.singleUUIDList != null) {
              this.currentRow.getColumn(colPos).setValue(
                  this.singleUUIDList.get(this.iterPos));
            }
            else {
              this.currentRow.getColumn(colPos).setValue(uuids[listPos - 1]);
            }
          }
          else {
            if (this.singleShortUUIDList != null) {
              this.currentRow.getColumn(colPos).setValue(
                  this.singleShortUUIDList.get(this.iterPos));
            }
            else {
              this.currentRow.getColumn(colPos).setValue(
                  shortUUIDs[-listPos - 1]);
            }
          }
        }
        this.iterPos++;
        return this.currentRow;
      }
    }
    return null;
  }

  /**
   * @see NoPutResultSet#markAsTopResultSet()
   */
  public void markAsTopResultSet() {
  }

  /**
   * Return the point of attachment for this subquery. (Only meaningful for Any
   * and Once ResultSets, which can and will only be at the top of a ResultSet
   * for a subquery.)
   * 
   * @return int Point of attachment (result set number) for this subquery. (-1
   *         if not a subquery - also Sanity violation)
   */
  public int getPointOfAttachment() {
    return -1;
  }

  /**
   * Return the isolation level of the scan in the result set. Only expected to
   * be called for those ResultSets that contain a scan.
   * 
   * @return The isolation level of the scan (in TransactionController
   *         constants).
   */
  public int getScanIsolationLevel() {
    return TransactionController.ISOLATION_READ_UNCOMMITTED;
  }

  /**
   * Notify a NPRS that it is the source for the specified TargetResultSet. This
   * is useful when doing bulk insert.
   * 
   * @param trs
   *          The TargetResultSet.
   */
  public void setTargetResultSet(TargetResultSet trs) {
  }

  /**
   * Set whether or not the NPRS need the row location when acting as a row
   * source. (The target result set determines this.)
   * 
   */
  public void setNeedsRowLocation(boolean needsRowLocation) {
  }

  /**
   * Get the estimated row count from this result set.
   * 
   * @return The estimated row count (as a double) from this result set.
   */
  public double getEstimatedRowCount() {
    return 0d;
  }

  /**
   * Get the number of this ResultSet, which is guaranteed to be unique within a
   * statement.
   */
  public int resultSetNumber() {
    return 0;
  }

  /**
   * Set the current row to the row passed in.
   * 
   * @param row
   *          the new current row
   * 
   */
  public void setCurrentRow(ExecRow row) {
    SanityManager.THROWASSERT("setCurrentRow() not expected to be called for "
        + getClass().getName());
  }

  /**
   * Clear the current row
   * 
   */
  public void clearCurrentRow() {
    for (int pos = 1; pos <= this.currentRow.nColumns(); pos++) {
      this.currentRow.getColumn(pos).restoreToNull();
    }
  }

  /**
   * This result set has its row from the last fetch done. If the cursor is
   * closed, a null is returned.
   * 
   * @see CursorResultSet
   * 
   * @return the last row returned;
   * @exception StandardException
   *              thrown on failure.
   */
  public ExecRow getCurrentRow() throws StandardException {
    if (SanityManager.DEBUG) {
      SanityManager.ASSERT(this.isOpen, "resultSet expected to be open");
    }
    return this.currentRow;
  }

  /**
   * Returns the row location of the current base table row of the cursor. If
   * this cursor's row is composed of multiple base tables' rows, i.e. due to a
   * join, then a null is returned. For a temporary row holder, we always return
   * null.
   * 
   * @return the row location of the current cursor row.
   */
  public RowLocation getRowLocation() {
    SanityManager.THROWASSERT("getRowLocation() not expected to be called for "
        + getClass().getName());
    return null;
  }

  /**
   * Clean up
   * 
   * @exception StandardException
   *              thrown on error
   */
  public void close(boolean cleanupOnError) throws StandardException {
    this.isOpen = false;
  }

  @Override
  public void setGfKeysForNCJoin(ArrayList<DataValueDescriptor> keys)
      throws StandardException {
    throw StandardException.newException(SQLState.NOT_IMPLEMENTED,
        " Currently this method is not implemented or overridden for class "
            + this.getClass().getSimpleName());
  }
  //////////////////////////////////////////////////////////////////////////
  //
  // MISC FROM RESULT SET
  //
  /////////////////////////////////////////////////////////////////////////

  /**
   * Returns TRUE if the statement returns rows (i.e. is a SELECT or FETCH
   * statement), FALSE if it returns no rows.
   * 
   * @return TRUE if the statement returns rows, FALSE if not.
   */
  public boolean returnsRows() {
    return true;
  }

  public int modifiedRowCount() {
    return 0;
  };

  /**
   * Tells the system that there will be calls to getNextRow().
   * 
   * @exception StandardException
   *              Thrown on failure
   */
  public void open() throws StandardException {
    openCore();
  }

  /**
   * Returns the row at the absolute position from the query, and returns NULL
   * when there is no such position. (Negative position means from the end of
   * the result set.) Moving the cursor to an invalid position leaves the cursor
   * positioned either before the first row (negative position) or after the
   * last row (positive position). NOTE: An exception will be thrown on 0.
   * 
   * @param row
   *          The position.
   * @return The row at the absolute position, or NULL if no such position.
   * 
   * @exception StandardException
   *              Thrown on failure
   * @see Row
   */
  public ExecRow getAbsoluteRow(int row) throws StandardException {
    SanityManager.THROWASSERT("getAbsoluteRow() not expected to be called for "
        + getClass().getName());
    return null;
  }

  /**
   * Returns the row at the relative position from the current cursor position,
   * and returns NULL when there is no such position. (Negative position means
   * toward the beginning of the result set.) Moving the cursor to an invalid
   * position leaves the cursor positioned either before the first row (negative
   * position) or after the last row (positive position). NOTE: 0 is valid.
   * NOTE: An exception is thrown if the cursor is not currently positioned on a
   * row.
   * 
   * @param row
   *          The position.
   * @return The row at the relative position, or NULL if no such position.
   * 
   * @exception StandardException
   *              Thrown on failure
   * @see Row
   */
  public ExecRow getRelativeRow(int row) throws StandardException {
    SanityManager.THROWASSERT("getRelativeRow() not expected to be called for "
        + getClass().getName());
    return null;
  }

  /**
   * Sets the current position to before the first row and returns NULL because
   * there is no current row.
   * 
   * @return NULL.
   * 
   * @exception StandardException
   *              Thrown on failure
   * @see Row
   */
  public ExecRow setBeforeFirstRow() throws StandardException {
    SanityManager.THROWASSERT("setBeforeFirstRow() not expected "
        + "to be called for " + getClass().getName());
    return null;
  }

  /**
   * Returns the first row from the query, and returns NULL when there are no
   * rows.
   * 
   * @return The first row, or NULL if no rows.
   * 
   * @exception StandardException
   *              Thrown on failure
   * @see Row
   */
  public ExecRow getFirstRow() throws StandardException {
    SanityManager.THROWASSERT("getFirstRow() not expected to be called for "
        + getClass().getName());
    return null;
  }

  /**
   * Returns the next row from the query, and returns NULL when there are no
   * more rows.
   * 
   * @return The next row, or NULL if no more rows.
   * 
   * @exception StandardException
   *              Thrown on failure
   * @see Row
   */
  public ExecRow getNextRow() throws StandardException {
    return getNextRowCore();
  }

  /**
   * Returns the previous row from the query, and returns NULL when there are no
   * more previous rows.
   * 
   * @return The previous row, or NULL if no more previous rows.
   * 
   * @exception StandardException
   *              Thrown on failure
   * @see Row
   */
  public ExecRow getPreviousRow() throws StandardException {
    SanityManager.THROWASSERT("getPreviousRow() not expected to be called for "
        + getClass().getName());
    return null;
  }

  /**
   * Returns the last row from the query, and returns NULL when there are no
   * rows.
   * 
   * @return The last row, or NULL if no rows.
   * 
   * @exception StandardException
   *              Thrown on failure
   * @see Row
   */
  public ExecRow getLastRow() throws StandardException {
    SanityManager.THROWASSERT("getLastRow() not expected to be called for "
        + getClass().getName());
    return null;
  }

  /**
   * Sets the current position to after the last row and returns NULL because
   * there is no current row.
   * 
   * @return NULL.
   * 
   * @exception StandardException
   *              Thrown on failure
   * @see Row
   */
  public ExecRow setAfterLastRow() throws StandardException {
    SanityManager.THROWASSERT("getLastRow() not expected to be called for "
        + getClass().getName());
    return null;
  }

  /**
   * Determine if the cursor is before the first row in the result set.
   * 
   * @return true if before the first row, false otherwise. Returns false when
   *         the result set contains no rows.
   */
  public boolean checkRowPosition(int isType) {
    return false;
  }

  /**
   * Returns the row number of the current row. Row numbers start from 1 and go
   * to 'n'. Corresponds to row numbering used to position current row in the
   * result set (as per JDBC).
   * 
   * @return the row number, or 0 if not on a row
   * 
   */
  public int getRowNumber() {
    return 0;
  }

  /**
   * Tells the system to clean up on an error.
   * 
   * @exception StandardException
   *              Thrown on error.
   */
  public void cleanUp(boolean cleanupOnError) throws StandardException {
    close(cleanupOnError);
  }

  /**
   * Find out if the ResultSet is closed or not. Will report true for result
   * sets that do not return rows.
   * 
   * @return true if the ResultSet has been closed.
   */
  public boolean isClosed() {
    return !this.isOpen;
  }

  /**
   * Tells the system that there will be no more access to any database
   * information via this result set; in particular, no more calls to open().
   * Will close the result set if it is not already closed.
   * 
   * @exception StandardException
   *              on error
   */
  public void finish() throws StandardException {
    close(false);
  }

  /**
   * Get the execution time in milliseconds.
   * 
   * @return long The execution time in milliseconds.
   */
  public long getExecuteTime() {
    return 0L;
  }

  /**
   * @see ResultSet#getAutoGeneratedKeysResultset
   */
  public ResultSet getAutoGeneratedKeysResultset() {
    // A non-null resultset would be returned only for an insert statement
    return (ResultSet)null;
  }

  /**
   * Get the Timestamp for the beginning of execution.
   * 
   * @return Timestamp The Timestamp for the beginning of execution.
   */
  public Timestamp getBeginExecutionTimestamp() {
    return (Timestamp)null;
  }

  /**
   * Get the Timestamp for the end of execution.
   * 
   * @return Timestamp The Timestamp for the end of execution.
   */
  public Timestamp getEndExecutionTimestamp() {
    return (Timestamp)null;
  }

  /**
   * Return the total amount of time spent in this ResultSet
   * 
   * @param type
   *          CURRENT_RESULTSET_ONLY - time spent only in this ResultSet
   *          ENTIRE_RESULTSET_TREE - time spent in this ResultSet and below.
   * 
   * @return long The total amount of time spent (in milliseconds).
   */
  public final long getTimeSpent(int type, int timeType) {
    return 0L;
  }

  /**
   * Get the subquery ResultSet tracking array from the top ResultSet. (Used for
   * tracking open subqueries when closing down on an error.)
   * 
   * @param numSubqueries
   *          The size of the array (For allocation on demand.)
   * 
   * @return NoPutResultSet[] Array of NoPutResultSets for subqueries.
   */
  public NoPutResultSet[] getSubqueryTrackingArray(int numSubqueries) {
    return (NoPutResultSet[])null;
  }

  /**
   * Returns the name of the cursor, if this is cursor statement of some type
   * (declare, open, fetch, positioned update, positioned delete, close).
   * 
   * @return A String with the name of the cursor, if any. Returns NULL if this
   *         is not a cursor statement.
   */
  public String getCursorName() {
    return (String)null;
  }

  /**
   * @see NoPutResultSet#requiresRelocking
   */
  public boolean requiresRelocking() {
    SanityManager.THROWASSERT("requiresRelocking() not expected "
        + "to be called for " + getClass().getName());
    return false;
  }

  /////////////////////////////////////////////////////////
  //
  // Access/RowSource -- not implemented
  //
  /////////////////////////////////////////////////////////
  /**
   * Get the next row as an array of column objects. The column objects can be a
   * JBMS Storable or any Serializable/Externalizable/Formattable/Streaming
   * type. <BR>
   * A return of null indicates that the complete set of rows has been read.
   * 
   * <p>
   * A null column can be specified by leaving the object null, or indicated by
   * returning a non-null getValidColumns. On streaming columns, it can be
   * indicated by returning a non-null get FieldStates.
   * 
   * <p>
   * If RowSource.needToClone() is true then the returned row (the
   * DataValueDescriptor[]) is guaranteed not to be modified by drainer of the
   * RowSource (except that the input stream will be read, of course) and
   * drainer will keep no reference to it before making the subsequent nextRow
   * call. So it is safe to return the same DataValueDescriptor[] in subsequent
   * nextRow calls if that is desirable for performance reasons.
   * <p>
   * If RowSource.needToClone() is false then the returned row (the
   * DataValueDescriptor[]) may be be modified by drainer of the RowSource, and
   * the drainer may keep a reference to it after making the subsequent nextRow
   * call. In this case the client should severe all references to the row after
   * returning it from getNextRowFromRowSource().
   * 
   * @exception StandardException
   *              Standard Derby Error Policy
   */
  public ExecRow getNextRowFromRowSource() throws StandardException {
    return null;
  }

  /**
   * Does the caller of getNextRowFromRowSource() need to clone the row in order
   * to keep a reference to the row past the getNextRowFromRowSource() call
   * which returned the row. This call must always return the same for all rows
   * in a RowSource (ie. the caller will call this once per scan from a
   * RowSource and assume the behavior is true for all rows in the RowSource).
   */
  public boolean needsToClone() {
    return false;
  }

  /**
   * getValidColumns describes the DataValueDescriptor[] returned by all calls
   * to the getNextRowFromRowSource() call.
   * 
   * If getValidColumns returns null, the number of columns is given by the
   * DataValueDescriptor.length where DataValueDescriptor[] is returned by the
   * preceeding getNextRowFromRowSource() call. Column N maps to
   * DataValueDescriptor[N], where column numbers start at zero.
   * 
   * If getValidColumns return a non null validColumns FormatableBitSet the
   * number of columns is given by the number of bits set in validColumns.
   * Column N is not in the partial row if validColumns.get(N) returns false.
   * Column N is in the partial row if validColumns.get(N) returns true. If
   * column N is in the partial row then it maps to DataValueDescriptor[M] where
   * M is the count of calls to validColumns.get(i) that return true where i <
   * N. If DataValueDescriptor.length is greater than the number of columns
   * indicated by validColumns the extra entries are ignored.
   */
  public FormatableBitSet getValidColumns() {
    return null;
  }

  /**
   * closeRowSource tells the RowSource that it will no longer need to return
   * any rows and it can release any resource it may have. Subsequent call to
   * any method on the RowSource will result in undefined behavior. A closed
   * rowSource can be closed again.
   */
  public void closeRowSource() {
  }

  /////////////////////////////////////////////////////////
  //
  // Access/RowLocationRetRowSource -- not implemented
  //
  /////////////////////////////////////////////////////////
  /**
   * needsRowLocation returns true iff this the row source expects the drainer
   * of the row source to call rowLocation after getting a row from
   * getNextRowFromRowSource.
   * 
   * @return true iff this row source expects some row location to be returned
   * @see #rowLocation
   */
  public boolean needsRowLocation() {
    return false;
  }

  /**
   * rowLocation is a callback for the drainer of the row source to return the
   * rowLocation of the current row, i.e, the row that is being returned by
   * getNextRowFromRowSource. This interface is for the purpose of loading a
   * base table with index. In that case, the indices can be built at the same
   * time the base table is laid down once the row location of the base row is
   * known. This is an example pseudo code on how this call is expected to be
   * used:
   * 
   * <BR>
   * 
   * <pre>
   * boolean needsRL = rowSource.needsRowLocation();
   * DataValueDescriptor[] row;
   * while ((row = rowSource.getNextRowFromRowSource()) != null) {
   *   RowLocation rl = heapConglomerate.insertRow(row);
   *   if (needsRL)
   *     rowSource.rowLocation(rl);
   * }
   * </pre>
   * 
   * <BR>
   * 
   * NeedsRowLocation and rowLocation will ONLY be called by a drainer of the
   * row source which CAN return a row location. Drainer of row source which
   * cannot return rowLocation will guarentee to not call either callbacks.
   * Conversely, if NeedsRowLocation is called and it returns true, then for
   * every row return by getNextRowFromRowSource, a rowLocation callback must
   * also be issued with the row location of the row. Implementor of both the
   * source and the drain of the row source must be aware of this protocol.
   * 
   * <BR>
   * The RowLocation object is own by the caller of rowLocation, in other words,
   * the drainer of the RowSource. This is so that we don't need to new a row
   * location for every row. If the Row Source wants to keep the row location,
   * it needs to clone it (RowLocation is a ClonableObject).
   * 
   * @exception StandardException
   *              on error
   */
  public void rowLocation(RowLocation rl) throws StandardException {
  }

  /**
   * @see NoPutResultSet#positionScanAtRowLocation
   * 
   *      This method is result sets used for scroll insensitive updatable
   *      result sets for other result set it is a no-op.
   */
  public void positionScanAtRowLocation(RowLocation rl)
      throws StandardException {
    // Only used for Scrollable insensitive result sets otherwise no-op
  }

  // GemFireXD additions -- not implemented

  /**
   * Is this ResultSet or it's source result set for update This method will be
   * overriden in the inherited Classes if it is true
   * 
   * @return Whether or not the result set is for update.
   */
  public boolean isForUpdate() {
    return false;
  }

  public java.sql.SQLWarning getWarnings() {
    return null;
  }

  /**
   * @see NoPutResultSet#updateRow
   * 
   *      This method is result sets used for scroll insensitive updatable
   *      result sets for other result set it is a no-op.
   */
  public void updateRow(ExecRow row) throws StandardException {
    // Only ResultSets of type Scroll Insensitive implement
    // detectability, so for other result sets this method
    // is a no-op
  }

  @Override
  public void deleteRowDirectly() throws StandardException {
    SanityManager.THROWASSERT("deleteRowDirectly() not expected "
        + "to be called for " + getClass().getName());
  }

  /**
   * @see NoPutResultSet#markRowAsDeleted
   * 
   *      This method is result sets used for scroll insensitive updatable
   *      result sets for other result set it is a no-op.
   */
  public void markRowAsDeleted() throws StandardException {
    // Only ResultSets of type Scroll Insensitive implement
    // detectability, so for other result sets this method
    // is a no-op
  }

  /**
   * Return the <code>Activation</code> for this result set.
   * 
   * @return activation
   */
  public final Activation getActivation() {
    return this.activation;
  }

  @Override
  public final boolean canUpdateInPlace() {
    return false;
  }

  @Override
  public TXState initLocalTXState() {
    SanityManager.THROWASSERT("initLocalTXState() not expected "
        + "to be called for " + getClass().getName());
    // never reached
    return null;
  }

  @Override
  public void upgradeReadLockToWrite(final RowLocation rl,
      final GemFireContainer container) throws StandardException {
    SanityManager.THROWASSERT("upgradeReadLockToWrite() not expected "
        + "to be called for " + getClass().getName());
  }

  @Override
  public void updateRowLocationPostRead() throws StandardException {
    SanityManager.THROWASSERT("updateRowLocationPostRead() not expected "
        + "to be called for " + getClass().getName());
  }

  @Override
  public void filteredRowLocationPostRead(TXState localTXState) throws StandardException {
    
    SanityManager.THROWASSERT("filteredRowLocationPostRead() not expected "
        + "to be called for " + getClass().getName());
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean supportsMoveToNextKey() {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public int getScanKeyGroupID() {
    throw new UnsupportedOperationException("not expected to be invoked");
  }

  /**
   * @see ResultSet#hasAutoGeneratedKeysResultSet
   */
  @Override
  public boolean hasAutoGeneratedKeysResultSet() {
    return false;
  }

  /**
   * @see ResultSet#flushBatch
   */
  @Override
  public void flushBatch() {
  }

  @Override
  public void accept(ResultSetStatisticsVisitor visitor) {
    SanityManager.THROWASSERT("accept() not expected to be called for "
        + getClass().getName());
  }

  @Override
  public UUID getExecutionPlanID() {
    SanityManager.THROWASSERT("getExecutionPlanID() not expected to be called for "
        + getClass().getName());
    return null;
  }

  @Override
  public void markLocallyExecuted() {
    // nothing to do.
  }

  @Override
  public void resetStatistics() {
    // TODO Auto-generated method stub
    
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isDistributedResultSet() {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean addLockReference(GemFireTransaction tran) {
    return false;
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean releaseLocks(GemFireTransaction tran) {
    return false;
  }

  @Override
  public void closeBatch() throws StandardException {
  }
  
  @Override
  public RowLocation fetch(final RowLocation loc, ExecRow destRow,
      FormatableBitSet validColumns, boolean faultIn, GemFireContainer container)
      throws StandardException {
    return RowUtil.fetch(loc, destRow, validColumns, faultIn, container, null,
        null, 0, null);
  }

  public Context getNewPlanContext() {
    return new PlanUtils.Context();
  }

  @Override
  public StringBuilder buildQueryPlan(final StringBuilder builder, final PlanUtils.Context context) {
    
    PlanUtils.xmlBeginTag(builder, context, this);
    PlanUtils.xmlTermTag(builder, context, PlanUtils.OP_AUTO_GEN_KEYS);
    
    PlanUtils.xmlCloseTag(builder, context, this);
    
    return builder;
  }

  @Override
  public void releasePreviousByteSource() {
  }

  @Override
  public void setMaxSortingLimit(long limit) {
  }

  @Override
  public void checkCancellationFlag() throws StandardException {
    final Activation act = this.activation;
    if (act != null && act.isQueryCancelled()) {
      act.checkCancellationFlag();
    }
  }
  
  @Override
  public void forceReOpenCore() throws StandardException { 
  }
}
