package s2.adapi.sdk.offerwall.impl;

import android.content.Context;
import android.content.SharedPreferences;
import android.content.pm.ApplicationInfo;
import android.content.pm.PackageInfo;
import android.content.pm.PackageManager;
import android.os.Build;
import android.os.Bundle;

import androidx.lifecycle.LiveData;
import androidx.lifecycle.MutableLiveData;

import com.google.android.gms.ads.identifier.AdvertisingIdClient;
import com.google.android.gms.appset.AppSet;
import com.google.android.gms.appset.AppSetIdClient;
import com.google.android.gms.appset.AppSetIdInfo;
import com.google.android.gms.tasks.OnSuccessListener;
import com.google.android.gms.tasks.Task;

import java.util.HashSet;
import java.util.Set;

import s2.adapi.sdk.offerwall.S2Log;

public class SdkCore {

    private static final String OFFERWALL_URL = "https://ads.snapplay.io/api/main/offerwall/reqlist";
    private static final String MAINWALL_URL = "https://ads.snapplay.io/api/main/offerwall/reqmain";

    //private static final String MAINWALL_URL = "https://adm.snapplay.io/adm/main/offerwall/reqmain";
    //private static final String OFFERWALL_URL = "https://adm.snapplay.io/adm/main/offerwall/reqlist";

    private static final String META_S2_APP_ID = "s2_app_id";
    private String packageName = null;
    private String appVersion = null;
    private String installMarket = null;

    private String appId = null;

    private String adid = null;
    private String asid = null;

    private String deviceModel = null;
    private String deviceOs = "Android";
    private String osVersion = null;
    private String osVersionCode = null;

    private MutableLiveData<String> pubUserName;

    // SdkCore 객체를 생성하는 Factory 메소드. 생성 및 초기화에 성공하면 해당 객체를 반환하고 실패하면 null 을 반환한다.
    public static SdkCore getInstance(Context context) {
        SdkCore sdkCore = new SdkCore();
        boolean flag = sdkCore.initInstance(context);

        if (flag) {
            return sdkCore;
        }
        else {
            return null;
        }
    }

    // S2OfferwallView 에서 startActivity 한 S2OfferwallActivity 가 finish() 되면 이것을 offerwall view 에
    // 이벤트를 발생시키기위하여 LiveData 를 사용한 observer pattern 을 적용한다.
    // 이것은 offerwallView 내에 있는 offerwall 의 onShow() javascript 를 호출하기 위한 용도이다.

    private final MutableLiveData<Set<String>> activityKeys = new MutableLiveData<>(new HashSet<>());

    public LiveData<Set<String>> getActivityKeys() {
        return activityKeys;
    }

    public void addActivityKey(String key) {
        if (activityKeys.getValue() == null) {
            activityKeys.setValue(new HashSet<>());
        }

        // LiveData는 동일 객체일 경우 onChanged() 호출 안됨, 항상 new HashSet<>(...)으로 새 Set을 만들어서 setValue() 하세요.
        Set<String> cloneKeys = new HashSet<>(activityKeys.getValue()); // hashset 복제하기.
        cloneKeys.add(key);
        activityKeys.setValue(cloneKeys);
    }

    public void removeActivityKey(String key) {
        if (activityKeys.getValue() == null) {
            return;
        }

        Set<String> cloneKeys = new HashSet<>(activityKeys.getValue());
        cloneKeys.remove(key);
        activityKeys.setValue(cloneKeys);
    }

    // pubUserName observer pattern 적용을 위한 LiveData
    public MutableLiveData<String> getCurrentUserName() {
        if (pubUserName == null) {
            pubUserName = new MutableLiveData<>();
        }

        return pubUserName;
    }

    public String getOfferwallUrl(boolean isMainwall) {
        String offerwallUrl = (isMainwall? MAINWALL_URL : OFFERWALL_URL) + "?app_id=" + appId;

        if (adid != null) {
            offerwallUrl += "&adid=" + adid;
        }

        if (asid != null) {
            offerwallUrl += "&asid=" + asid;
        }

        if (packageName != null) {
            offerwallUrl += "&app_pkg=" + packageName;
        }

        if (appVersion != null) {
            offerwallUrl += "&app_ver=" + appVersion;
        }

        if (installMarket != null) {
            offerwallUrl += "&inst_mkt=" + installMarket;
        }

        if (deviceModel != null) {
            offerwallUrl += "&ph_mdl=" + deviceModel;
        }

        if (osVersion != null) {
            offerwallUrl += "&os_ver=" + osVersion;
        }

        offerwallUrl += "&ph_os=" + deviceOs + "&sdk_ver=" + Constants.SDK_VER;

        if (Constants.IS_DEBUG) {
            offerwallUrl += "&debug=Y";
        }

        return offerwallUrl;
    }

    public String getADID() {
        return adid;
    }
    
    private boolean initInstance(final Context context) {

        // App Package 명, App version, 설치시 사용된 마켓앱 정보 가져오기
        packageName = context.getPackageName();

        PackageManager packageManager = context.getPackageManager();

        try {
            PackageInfo packageInfo = packageManager.getPackageInfo(packageName, 0);
            appVersion = packageInfo.versionName;
            installMarket = packageManager.getInstallerPackageName(packageName);

            // app_id (pub_id, pub_key)
            // API 로 저장되어 있는 값을 우선 적용한다.
            appId = Constants.getPreferences(context).getString(Constants.PREF_APP_ID, "");
            if (Utils.isEmpty(appId)) {
                // API 로 저장한 값이 없는 경우에는 AndroidManifest.xml 파일에 설정된 값을 가져온다.
                ApplicationInfo info = packageManager.getApplicationInfo(packageName, PackageManager.GET_META_DATA);
                Bundle bundle = info.metaData;
                if (bundle != null) {
                    appId = bundle.getString(META_S2_APP_ID);
                }
            }

            if (appId == null) {
                S2Log.e("## Init failed. No <meta-data android:name=\"s2_app_id\" android:value=\"YOUR_VALUE\"/> in AndroidManifest.xml.");
                return false;
            }

            S2Log.e("## s2_app_id = " + appId);

        }
        catch (Exception e) {
            S2Log.e("init error.", e);
            return false;
        }

        // appset_id
        // preference 에 저장된 값을 먼저 읽어 온다.
        asid = Constants.getPreferences(context).getString(Constants.PREF_APPSET_ID,"");

        AppSetIdClient client = AppSet.getClient(context);
        Task<AppSetIdInfo> task = client.getAppSetIdInfo();
        task.addOnSuccessListener(new OnSuccessListener<AppSetIdInfo>() {
            @Override
            public void onSuccess(AppSetIdInfo info) {
                // Determine current scope of app set ID.
                int scope = info.getScope();

                // Read app set ID value, which uses version 4 of the
                // universally unique identifier (UUID) format.
                String asid2 = info.getId();
                if (!asid.equals(asid2)) {
                    // value has changed (or first acquired) so store the value into the pref again.
                    asid = asid2;
                    SharedPreferences pref = Constants.getPreferences(context);
                    pref.edit().putString(Constants.PREF_APPSET_ID, asid2).apply();
                }

                S2Log.d("## ASID = " + asid);
            }
        });

        // advertising id
        updateAdid(context);

        // 기기 정보
        // {ph_os=Android, os_ver=14.null, ph_mdl=Samsung SM-F711N, os_ver_m=14}
        deviceModel = Build.MODEL;   // SM-F711N
        osVersion = Build.VERSION.RELEASE;   // 14
        osVersionCode = String.valueOf(Build.VERSION.SDK_INT);   // 34

        S2Log.d("## " + deviceModel + " , " + osVersion + " , " + osVersionCode);
        return true;
    }

    public void updateAdid(Context context) {
        // advertising id
        try {
            AdvertisingIdClient.Info adInfo = AdvertisingIdClient.getAdvertisingIdInfo(context);
            if (!adInfo.isLimitAdTrackingEnabled()) {
                adid = adInfo.getId();
                S2Log.d("## ADID = " + adid);
            }
        }
        catch (Exception e) {
            S2Log.e("## failed to get ADID.", e);
        }
    }
}
