package s2.adapi.sdk.offerwall;

import static android.app.Activity.RESULT_OK;

import android.annotation.SuppressLint;
import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.content.ContextWrapper;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.pm.PackageManager;
import android.net.Uri;
import android.os.Build;
import android.os.Message;
import android.provider.Settings;
import android.util.AttributeSet;
import android.view.ViewGroup;
import android.webkit.ConsoleMessage;
import android.webkit.JsResult;
import android.webkit.ValueCallback;
import android.webkit.WebChromeClient;
import android.webkit.WebResourceRequest;
import android.webkit.WebView;
import android.webkit.WebViewClient;
import android.widget.RelativeLayout;

import androidx.annotation.NonNull;
import androidx.core.content.ContextCompat;
import androidx.lifecycle.LiveData;
import androidx.lifecycle.Observer;

import java.util.Set;
import java.util.UUID;

import s2.adapi.sdk.offerwall.impl.Constants;
import s2.adapi.sdk.offerwall.impl.SdkCore;
import s2.adapi.sdk.offerwall.impl.Utils;

/**
 * A view that displays an offerwall.
 *
 * <p><b>XML Sample</b></p>
 * <pre>
 * {@code
 * <s2.adapi.sdk.offerwall.S2OfferwallView
 *     android:id="@+id/s2_offerwall_view"
 *     android:layout_width="match_parent"
 *     android:layout_height="match_parent"
 *     />
 * }
 * </pre>
 *
 * <p><b>Activity/Fragment Sample</b></p>
 * <pre>
 * {@code
 * S2OfferwallView offerwallView = (S2OfferwallView) findViewById(R.id.s2_offerwall_view);
 * offerwallView.setEventListener(new S2Offerwall.EventListener() {
 *     @Override
 *     public void onCloseRequested(String param) {
 *         // This event is invoked when the offerwall close is requested.
 *         // You should implement the code to close the offerwall view or activity.
 *     }
 *
 *     @Override
 *     public void onLoginRequested(String param) {
 *         // This event is invoked when the login is requested.
 *         // You should implement the code to login and then call S2Offerwall.setUserName().
 *     }
 *
 *     @Override
 *     public void onPermissionRequested(String permission) {
 *         // This event is invoked when the permission is requested.
 *         // You should implement the code to request the permission.
 *     }
 * });
 * offerwallView.load("your_placement_name", false); // load and display the offerwall
 * }
 * </pre>
 */
public class S2OfferwallView extends RelativeLayout {

    private S2Offerwall.EventListener eventListener = null;

    private WebView webView = null;
    private String pubUserName = "";

    private boolean offerwallLoaded = false;

    private final String activityKey = UUID.randomUUID().toString();

    // 전체 Flow 설명
    // 로그인이 안된 상태에서 오퍼월을 띄운 경우 광고 참여시 로그인이 필요하게되는데
    // 이때 매체에서 onLoginRequest() 이벤트를 받아서 로그인 처리를 하도록 구현해야한다.
    // 로그인이 완료되면 로그인 ID 를 S2Offerwall.setUserName() 을 호출하여 전달한다.
    // 이때 설정된 로그인 ID(pub_user_nm) 을 Observer 패턴(Android LiveData)을 사용하여 S2OfferwallView 에
    // 전달하고 다시 현재 로딩되어 있는 offerwall page 의 S2Offerwall.setPubUserName() javascript 호출하여
    // 해당 페이지에 pub_user_nm 을 설정하도록 처리하였다. 이후 offerwall page 에서 서버 호출 시 pub_user_nm 값을
    // HTTP header 를 통하여 서버에 전달하도록 구현하였다.

    // Observer object to be informed an event when S2Offerwall.setUserName() is called.
    private final Observer<String> userNameObserver = userName -> {
        //S2Log.d("#### userName set to " + userName);
        pubUserName = userName; // keep the changed userName
        invokeJavascript_setPubUserName(userName);   // apply to offerwall page
    };

    // Observer object to be informed an event when S2OfferwallActivity is finished.
    private final Observer<Set<String>> activityKeyObserver = keys -> {
        if (!keys.contains(activityKey)) {
            // activityKeys 가 존재하지 않는다. -> startActivity 된 activity 가 종료되었다는 뜻.
            // 결국 이 offerwall view 가 다시 화면에 나타났다는 뜻이다.
            //S2Log.d("#### view onShow " + activityKey);
            invokeJavascript_onShow(); // 룰렛이벤트 페이지가 갱신을 위하여 호출한다. (이것을 위하여 만든 기능임)

            removeActivityKeyObserver();
        }
    };

    private void removeActivityKeyObserver() {
        SdkCore sdkCore = S2Offerwall.getCoreInstance();
        if (sdkCore != null) {
            LiveData<Set<String>> activityKeys = sdkCore.getActivityKeys();
            if (activityKeys != null) {
                activityKeys.removeObserver(activityKeyObserver);
            }
        }
    }

    private void invokeJavascript_setPubUserName(String userName) {
        S2Log.d("#### invoke javascript  " + userName + ", " + offerwallLoaded);
        if (webView != null && offerwallLoaded) {
            // call the javascript function in the offerwall page.
            webView.loadUrl("javascript:S2Offerwall.setPubUserName('" + userName + "')");
        }
    }

    private void invokeJavascript_onShow() {
        if (webView != null && offerwallLoaded) {
            webView.loadUrl("javascript:S2Sdk.onShow()");
        }
    }

    private void invokeJavascript_requestInstallPay(String actionCode, String installCheckKey, boolean isInstalled) {
        // 설치확인 결과를 전달받은 파라메터와 함께 webview 로 전달한다.
        if (webView != null && offerwallLoaded) {
            // call the javascript function in the offerwall page.
            String jsCall = String.format("javascript:S2Offerwall.requestInstallPay('%s',%s,'%s')",
                                                actionCode, installCheckKey, isInstalled?"Y":"N");
            //S2Log.d("#### invoke javascript  " + jsCall);
            webView.loadUrl(jsCall);
        }
    }

    private void onCloseRequested(String param) {
        if (eventListener != null) {
            eventListener.onCloseRequested(param);
        }
        else if (S2Offerwall.eventListener != null) {
            S2Offerwall.eventListener.onCloseRequested(param);
        }
    }

    private void onLoginRequested(String param) {
        if (eventListener != null) {
            eventListener.onLoginRequested(param);
        }
        else if (S2Offerwall.eventListener != null) {
            S2Offerwall.eventListener.onLoginRequested(param);
        }
    }

    @Override
    protected void onAttachedToWindow() {
        super.onAttachedToWindow();

        //S2Log.d("### on attached window");
        SdkCore sdkCore = S2Offerwall.getCoreInstance();
        if (sdkCore != null) {
            sdkCore.getCurrentUserName().observeForever(userNameObserver);
        }
    }

    @Override
    protected void onDetachedFromWindow() {
        super.onDetachedFromWindow();

        //S2Log.d("### on detached window");
        SdkCore sdkCore = S2Offerwall.getCoreInstance();
        if (sdkCore != null) {
            sdkCore.getCurrentUserName().removeObserver(userNameObserver);
        }

        removeActivityKeyObserver(); // 방어용 코드
    }

    /**
     * Constructor to create an S2OfferwallView instance.
     * @param context Context
     * @param attrs AttributeSet
     */
    public S2OfferwallView(Context context, AttributeSet attrs) {
        super(context, attrs);

        init(context);
    }

    /**
     * Set an EventListener for this view.
     * @param listener EventListener
     */
    public void setEventListener(S2Offerwall.EventListener listener) {
        eventListener = listener;
    }

    /**
     * Load and display an offerwall contents.
     * @param placementName name of placement to be displayed in the offerwall. Use "main" to display the main offerwall.
     * @param isPopupStyle set true or false.
     */
    public void load(final String placementName, boolean isPopupStyle) {
        // S2Offerwall.setConsentAgreed(getContext(), false); // for test
        // 2025.8.1 개인정보 수집동의 창 띄우기
        if (S2Offerwall.isConsentRequired(getContext())) {
            // 동의 skip 이 안되고 사용자가 미동의 상태라면 동의 창을 띄워야한다.
            S2Offerwall.showConsentDialog(getContext(),
                    () -> {
                        loadInternal(placementName, isPopupStyle);
                    },
                    () -> {
                        onCloseRequested(null);
                    });
        }
        else {
            loadInternal(placementName, isPopupStyle);
        }
    }

    private void loadInternal(final String placementName, boolean isPopupStyle) {

        post(() -> {
            String offerwallUrl = S2Offerwall.getOfferwallUrl(S2OfferwallView.this.getContext(), placementName, isPopupStyle);

            if (offerwallUrl == null) {
                // If SDK has not initialized, then show error message on webview using predefined HTML.
                webView.loadData(ERROR_HTML_NOT_INITIALIZED, "text/html", Constants.UTF_8);
            }
            else {
                webView.loadUrl(offerwallUrl);
            }
        });
    }

    /**
     * This is used in S2OfferwallActivity
     * @param urlToLoad url to be loaded
     */
    void loadUrl(final String urlToLoad) {
        post(() -> webView.loadUrl(urlToLoad));
    }

    @SuppressLint("SetJavaScriptEnabled")
    private void init(Context context) {
        WebView.setWebContentsDebuggingEnabled(true); // 디버그 모드 설정
        
        RelativeLayout.LayoutParams layoutParams =
                new RelativeLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);
        layoutParams.addRule(RelativeLayout.CENTER_IN_PARENT);
        layoutParams.setMargins(0,0,0,0);

        webView = new WebView(context);
        webView.setPadding(0, 0,0,0);
        webView.getSettings().setJavaScriptEnabled(true);
        //webView.getSettings().setJavaScriptCanOpenWindowsAutomatically(true);
        webView.getSettings().setSupportMultipleWindows(true);  // window.open() -> onCreateWindow() 호출되도록 설정
        webView.getSettings().setDefaultTextEncodingName("utf-8");
        webView.setScrollBarStyle(WebView.SCROLLBARS_INSIDE_OVERLAY);

        webView.setWebChromeClient(new WebChromeClient() {
            // window.open() 을 시스템 browser 로 처리 하기 위한 구현
            @Override
            public boolean onCreateWindow(WebView view, boolean isDialog, boolean isUserGesture, Message resultMsg) {
                //S2Log.d("#### onCreateWindow " + isDialog + ", " + isUserGesture);

                // window.open() 처리를 위한 WebView 를 생성하고 내부에서 이동할 URL을 브라우저로 전달한다.
                WebView popupWebView = getWebViewForPopup(view);

                // 부모 WebView 와 연결 하기
                WebView.WebViewTransport transport = (WebView.WebViewTransport) resultMsg.obj;
                transport.setWebView(popupWebView);
                resultMsg.sendToTarget();

                return true;
            }

            public void onCloseWindow(WebView view) {
                //S2Log.d("#### onCloseWindow ");

                // window.close() 가 호출되었다. mainwall 에서 띄운 offerwall 의 back() 이 수행된 경우이다.
                // mainwall -> offerwall 을 activity 로 띄울 경우 eventListener.onCloseRequested() 에
                // offerwall activity 를 finish() 하도록 event listner를 설정하였다.
                onCloseRequested(null);
            }

            @Override
            public boolean onJsAlert(WebView view, String url, String message, final JsResult result) {
                new AlertDialog.Builder(view.getContext())
                        .setMessage(message)
                        .setPositiveButton(android.R.string.ok, (dialogInterface, i) -> result.confirm())
                        .setCancelable(false)
                        .create()
                        .show();
                return true;
            }

            @Override
            public boolean onJsConfirm(WebView view, String url, String message, final JsResult result) {
                new AlertDialog.Builder(view.getContext())
                        .setMessage(message)
                        .setPositiveButton(android.R.string.ok, (dialogInterface, i) -> result.confirm())
                        .setNegativeButton(android.R.string.cancel, (dialogInterface, i) -> result.cancel())
                        .setCancelable(false)
                        .create()
                        .show();
                return true;
            }

            @Override
            public boolean onConsoleMessage(ConsoleMessage consoleMessage) {
                // 로그를 무시하고 logcat에 출력되지 않도록 처리
                return true;  // true를 반환하면 시스템 로그에 출력되지 않음
            }

            @Override
            public boolean onShowFileChooser(WebView webView, ValueCallback<Uri[]> filePathCallback,
                                             FileChooserParams fileChooserParams) {
                // 이용문의에서 이미지 첨부하기 위한 구현
                return openFileChooserActivity(filePathCallback);
            }
        });

        webView.setWebViewClient(new WebViewClient() {
            @Override
            public boolean shouldOverrideUrlLoading(WebView view, WebResourceRequest request) {
                if (request == null) {
                    return false;
                }

                Uri requestUrl = request.getUrl();
                String url = requestUrl.toString();
                String host = nullToEmpty(requestUrl.getHost());
                String path = nullToEmpty(requestUrl.getPath());

                S2Log.d("#### overrideUrlLoading " + url);

                if (url.equals("about:blank")) {
                    return false;
                }
                else if (url.startsWith("http://") || url.startsWith("https://")) {
                    // mainwall, offerwall, adinfo 화면이동인 경우 Activity 또는 popup dialog 로 처리하자.
                    if (isOfferwallUrl(host, path)) {
                        // offerwall 화면이 이동의 경우 새로운 Activity 를 띄운다.
                        if (url.contains("embed_yn=M")) {
                            // main activity 로 띄운다.
                            startOfferwallActivityWithUrl(url, false);
                            return true;
                        }
                        else if (url.contains("embed_yn=P")) {
                            // embed_yn=P => embed_yn=S 로 변경 후 popup activity 로 띄운다.
                            String url2 = url.replace("embed_yn=P", "embed_yn=S");
                            startOfferwallActivityWithUrl(url2, true);
                            return true;
                        }
                        else {
                            return false;
                        }
                    }

                    // 위의 경우를 제외하면 그대로 link navigation 이동처리한다.
                    return false;
                }
                else if (url.startsWith("intent://")) {
                    try {
                        Intent intent = Intent.parseUri(url, Intent.URI_INTENT_SCHEME);
                        view.getContext().startActivity(intent);

                        return true;
                    }
                    catch (Exception e) {
                        return false;
                    }
                }
                else if (url.startsWith("close://")) {
                    onCloseRequested(host);

                    return true;
                }
                else if (url.startsWith("login://")) {
                    onLoginRequested(host);

                    return true;
                }
                else if (url.startsWith("payinst://")) {
                    // 설치형 광고의 설치 여부는 SDK 가 판단 한다.
                    //S2Log.d(url);
                    // payinst://<광고앱 package>/<action code>/<install_key>/<install market>
                    String[] split = url.substring(10).split("/");

                    if (split.length > 3) {
                        String appPackage = split[0];
                        String actionCode = split[1];
                        String installCheckKey = split[2];
                        String installMarket = split[3];

                        post(() -> {
                            boolean isInstalled = Utils.isInstalled(getContext(), appPackage, installMarket);
                            //S2Log.d("## isInstalled " + isInstalled);
                            invokeJavascript_requestInstallPay(actionCode, installCheckKey, isInstalled);
                        });
                    }

                    return true;
                }
                else if (url.startsWith("adid://")) {
                    // adid 필수 광고인데 adid 정보가 없다.
                    // 광고 ID 설정 화면으로 이동시킨다.
                    showAdidAlert(getContext());
                    return true;
                }
                else {
                    return false;
                }

            }

            @Override
            public void onPageFinished(WebView view, String url) {
                //S2Log.d("## on page finished " + url);
                // Once a page loading has finished,
                // invoke setPubUserName() script to inject userName to the offerwall page.
                boolean isOfferwallUrl = isOfferwallUrl(url); // false;
                if (isOfferwallUrl) {
                    offerwallLoaded = true;
                    invokeJavascript_setPubUserName(pubUserName);
                }
            }

        });

        addView(webView);

        pubUserName = S2Offerwall.getUserName(context);
    }

    // window.open() 등으로 새창이 뜨는 경우를 처리하기위한 WebView 를 생성한다.
    // 외부 브라우저를 띄우도록 동작한다.
    @NonNull
    private WebView getWebViewForPopup(WebView view) {
        WebView popupWebView = new WebView(view.getContext());

        // 이동할 URL 처리를 위하여 WebViewClient 설정
        popupWebView.setWebViewClient(new WebViewClient() {
            @Override
            public boolean shouldOverrideUrlLoading(WebView view, WebResourceRequest request) {
                //S2Log.d("#### onCreateWindow - overrideUrlLoading " + request.getUrl().toString());

                // System browser 로 Intent 를 전달 한다.
                Intent intent = new Intent(Intent.ACTION_VIEW);
                intent.setData(request.getUrl());

                view.getContext().startActivity(intent);

                return true;
            }
        });

        return popupWebView;
    }

    private void startOfferwallActivityWithUrl(String urlToLoad, boolean isPopup) {
        Activity activity = getActivity();
        if (activity == null) {
            S2Log.e("## cannot get activity from offerwall.");
            return;
        }

        SdkCore sdkCore = S2Offerwall.getCoreInstance();
        if (sdkCore != null) {
            sdkCore.addActivityKey(activityKey);

            // start observation
            LiveData<Set<String>> activityKeys = sdkCore.getActivityKeys();
            if (activityKeys != null) {
                activityKeys.observeForever(activityKeyObserver);
            }
        }

        Intent intent = new Intent(activity, S2OfferwallActivity.class);
        intent.putExtra(Constants.INTENT_PARENT_KEY, activityKey); // activity finish 시점을 확인하기 위한 observer key
        intent.putExtra(Constants.INTENT_URL_TO_LOAD, urlToLoad);

        if (isPopup) {
            intent.putExtra(Constants.INTENT_PRESENT_TYPE, Constants.EXTRA_PRESENT_POPUP);

            activity.startActivity(intent);
            activity.overridePendingTransition(R.anim.slide_from_bottom, 0);
        }
        else {
            intent.putExtra(Constants.INTENT_PRESENT_TYPE, Constants.EXTRA_PRESENT_MAIN);

            activity.startActivity(intent);
            activity.overridePendingTransition(R.anim.slide_from_right, 0);
        }
    }

    private Activity getActivity() {
        return findActivity(getContext());
    }

    private Activity findActivity(Context context) {
        if (context instanceof Activity) {
            return (Activity)context;
        }
        else if (context instanceof ContextWrapper) {
            ContextWrapper contextWrapper = (ContextWrapper) context;
            Context baseContext = contextWrapper.getBaseContext();
            if (baseContext == null) {
                return null;
            }
            else {
                return findActivity(baseContext);
            }
        }
        else {
            return null;
        }
    }

    private String nullToEmpty(String str) {
        return (str == null) ? "" : str;
    }

    private final String[] offerwallPaths = { "/offerwall/list",
                                                "/offerwall/info",
                                                "/offerwall/help",
                                                "/offerwall/main",
                                                "event/page/"
                                                };
    private boolean isOfferwallUrl(String host, String path) {
        if (host == null || !host.endsWith("snapplay.io")) {
            return false;
        }

        for (String p : offerwallPaths) {
            if (path.contains(p)) {
                return true;
            }
        }

        return false;
    }

    private boolean isOfferwallUrl(String url) {
        Uri uri = Uri.parse(url);
        String host = uri.getHost();
        String path = uri.getPath();

        return isOfferwallUrl(host, path);
    }

    private static final String ERROR_HTML_NOT_INITIALIZED = "<html><body>SDK not initialized</body></html>";

    private final int FILE_CHOOSER_REQUEST_CODE = 1127;
    private ValueCallback<Uri[]> filePathCallbackGlobal = null;

    private boolean openFileChooserActivity(ValueCallback<Uri[]> filePathCallback) {
        Activity activity = findActivity(getContext());
        if (activity == null) {
            S2Log.e("## no activity found. cannot choose image file.");
            return false;
        }

        if (filePathCallbackGlobal != null) {
            // 중복 실행 방지
            S2Log.e("## file choser is already open.");
            return false;
        }

        String permission;
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.TIRAMISU) { // Android 13+
            permission = android.Manifest.permission.READ_MEDIA_IMAGES;
        }
        else { // Android 12 이하
            permission = android.Manifest.permission.READ_EXTERNAL_STORAGE;
        }

        if (ContextCompat.checkSelfPermission(activity, permission) != PackageManager.PERMISSION_GRANTED) {
            // 권한이 없으면 요청
            // 이벤트 발생시키고 false return
            if (eventListener != null) {
                eventListener.onPermissionRequested(permission);
            }
            return false;
        }

        filePathCallbackGlobal = filePathCallback;  // 전달 받은 callback 저장해둔다.

        // 파일 선택 Intent 생성
        Intent intent = new Intent(Intent.ACTION_GET_CONTENT);
        intent.addCategory(Intent.CATEGORY_OPENABLE);
        intent.setType("image/*");

        activity.startActivityForResult(Intent.createChooser(intent, "이미지 선택"), FILE_CHOOSER_REQUEST_CODE);

        return true;
    }

    // onActivityResult 호출한다.
    // return : true 내가 처리함, false 내가 처리할 것이 아님
    boolean handleFileChooserActivityResult(int requestCode, int resultCode, Intent data) {
        if (requestCode != FILE_CHOOSER_REQUEST_CODE) {
            // 이것은 내가 요청한 것이 아니다.
            return false;
        }

        if (filePathCallbackGlobal != null) {
            Uri[] results = null;
            if (resultCode == RESULT_OK && data != null) {
                results = new Uri[] {data.getData()};
            }

            filePathCallbackGlobal.onReceiveValue(results);
            filePathCallbackGlobal = null;
        }

        return true;
    }

    private void showAdidAlert(Context context) {
        new AlertDialog.Builder(context)
                //.setTitle("광고 ID 허용 안내")  // 제목
                .setMessage("광고 참여를 위해서는 광고 ID 가 필요합니다. 설정 후 앱을 재실행 해주세요.\n\n[설정 > 개인정보 보호 > 개인정보 설정 > 광고 > 광고 ID 받기]")              // 내용
                .setPositiveButton("설정하기", new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        openAdidSetting(context);
                        dialog.dismiss();         // 닫기
                    }
                })
                .setNegativeButton("취소", new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                        dialog.dismiss();         // 취소 클릭 시
                    }
                })
                .show();
    }

    private void openAdidSetting(Context context) {
        try {
            Intent intent = new Intent(Settings.ACTION_PRIVACY_SETTINGS);

            // 인텐트를 처리할 앱이 있는지 확인
            if (intent.resolveActivity(context.getPackageManager()) != null) {
                context.startActivity(intent);
            } else {
                // 없으면 일반 설정 화면으로 fallback
                context.startActivity(new Intent(Settings.ACTION_SETTINGS));
            }

        }
        catch (Exception e) {
            // 예외 발생 시 일반 설정 화면으로 이동
            context.startActivity(new Intent(Settings.ACTION_SETTINGS));
        }
    }
}
