package s2.adapi.sdk.offerwall;

import android.content.Intent;
import android.content.pm.PackageManager;
import android.os.Bundle;
import android.view.ViewGroup;
import android.widget.Toast;

import androidx.activity.EdgeToEdge;
import androidx.annotation.NonNull;
import androidx.appcompat.app.AppCompatActivity;
import androidx.core.graphics.Insets;
import androidx.core.view.ViewCompat;
import androidx.core.view.WindowInsetsCompat;

import s2.adapi.sdk.offerwall.impl.Constants;
import s2.adapi.sdk.offerwall.impl.SdkCore;

public class S2OfferwallActivity extends AppCompatActivity {

    private final int OFFERWALL_VIEW_ID = 127;

    private final int PERMISSION_REQUEST_CODE = 1128;

    private boolean isPopupStyle = false;

    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);

        // 2027.7.31 Android 15 EdgeToEdge 적용
        EdgeToEdge.enable(this);

        // 인텐트 데이터 추출
        Intent intent = getIntent();

        // Get intent parameters
        final String urlToLoad = intent.getStringExtra(Constants.INTENT_URL_TO_LOAD);
        final String placementName = intent.getStringExtra(Constants.INTENT_PLACEMENT_NAME);
        final String presentType = intent.getStringExtra(Constants.INTENT_PRESENT_TYPE);

        if (Constants.EXTRA_PRESENT_POPUP.equals(presentType)) {
            isPopupStyle = true;
            setContentView(R.layout.offerwall_main); // transparent activity 가 어렵다. 좀 더 고민해보자.
        }
        else {
            setContentView(R.layout.offerwall_main);
        }

        // 2027.7.31 EdgeToEdge 적용
        // 하단은 버튼 네비인 경우에만 Padding 적용한다.
        ViewCompat.setOnApplyWindowInsetsListener(findViewById(R.id.main), (v, insets) -> {
            Insets systemBars = insets.getInsets(WindowInsetsCompat.Type.systemBars());
            boolean isGestureBar = systemBars.bottom < 80;
            v.setPadding(systemBars.left, systemBars.top, systemBars.right,
                         isGestureBar? 0 : systemBars.bottom);
            return insets;
        });

        if (urlToLoad != null) {
            initOfferwallViewWithUrl(urlToLoad);
        }
        else {
            // S2Offerwall initialized
            S2Offerwall.initSdk(this, new S2Offerwall.InitializeListener() {
                @Override
                public void onSuccess() {
                    // Create S2OfferwallView and load in UI thread
                    runOnUiThread(() -> initOfferwallViewWithPlacement(placementName));
                }

                @Override
                public void onFailure() {

                }
            });
        }

    }

    @Override
    public void finish() {
        super.finish();

        final String presentType = getIntent().getStringExtra(Constants.INTENT_PRESENT_TYPE);

        if (Constants.EXTRA_PRESENT_POPUP.equals(presentType)) {
            // animation slide out to bottom
            overridePendingTransition(0, R.anim.slide_to_bottom);
        }
        else if (Constants.EXTRA_PRESENT_MAIN.equals(presentType)) {
            // animation slide out to right
            overridePendingTransition(0, R.anim.slide_to_right);
        }
    }

    @Override
    protected void onDestroy() {
        super.onDestroy();

        // 2025.07.24
        String parentKey = getIntent().getStringExtra(Constants.INTENT_PARENT_KEY);
        SdkCore sdkCore = S2Offerwall.getCoreInstance();
        if (parentKey != null && sdkCore != null) {
            sdkCore.removeActivityKey(parentKey); // LiveData 에서 parentKey 를 제거하여 이 activity 가 종료되었음을 알린다.
        }
    }

    // offerwall view 에서 이미지 첨부를 위한 파일 선택 결과 처리
    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        // requestCode가 파일 선택 요청 코드인지 확인
        S2OfferwallView view = getOfferwallView();
        if (view == null || !view.handleFileChooserActivityResult(requestCode, resultCode, data)) {
            super.onActivityResult(requestCode, resultCode, data);
        }
    }

    private S2OfferwallView getOfferwallView() {
        return (S2OfferwallView) findViewById(OFFERWALL_VIEW_ID);
    }

    private void initOfferwallViewWithPlacement(final String placementName) {

        // placement 를 지정하여 뜨는 경우는 오퍼월 최초 화면으로 띄우는 경우이다.
        // 이경우에도 닫기 이벤트 처리는 필요하므로 나중에 추가하자.
        S2OfferwallView offerwallView = createOfferwallView();
        offerwallView.load(placementName, isPopupStyle);
    }

    private void initOfferwallViewWithUrl(final String urlToLoad) {
        // url 로 offerwall 을 띄우는 경우는 mainwall 에서 offerwall click 하여 activity로 띄워주는 경우이다.
        // 이때 close 처리는 activity 를 닫는 것이고, login event 처리는 parent 의 login event 처리로직을 호출하는 것이다.
        S2OfferwallView offerwallView = createOfferwallView();
        offerwallView.loadUrl(urlToLoad);
    }

    private S2OfferwallView createOfferwallView() {
        S2OfferwallView offerwallView = new S2OfferwallView(this, null);
        ViewGroup.LayoutParams layoutParams = new ViewGroup.LayoutParams(
                ViewGroup.LayoutParams.MATCH_PARENT,
                ViewGroup.LayoutParams.MATCH_PARENT);

        offerwallView.setLayoutParams(layoutParams);
        offerwallView.setId(OFFERWALL_VIEW_ID);

        //setContentView(offerwallView);
        ViewGroup offerwallLayout = findViewById(R.id.offerwall_layout);
        offerwallLayout.addView(offerwallView);

        offerwallView.setEventListener(new S2Offerwall.EventListener() {
            @Override
            public void onLoginRequested(String param) {
                // 최초에 offerwall 을 activity 로 띄우는 경우는 global eventlistener 만 처리할 수 있는 구조이다.
                // 그러므로 이 경우에는 global event listner 를 호출해준다.
                S2Offerwall.eventListener.onLoginRequested(param);
            }

            @Override
            public void onCloseRequested(String param) {
                // Activity 로 뜬 webview 가 닫히는 것이므로 activity 를 닫아준다.
                S2OfferwallActivity.this.finish();
            }

            @Override
            public void onPermissionRequested(String permission) {
                requestPermissions(new String[]{permission}, PERMISSION_REQUEST_CODE);
            }
        });

        return offerwallView;
    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        if (requestCode == PERMISSION_REQUEST_CODE) {
            if (grantResults.length > 0 &&
                    grantResults[0] == PackageManager.PERMISSION_GRANTED) {
                // Permission is granted. Continue the action or workflow in your app.
            }
            else {
                // Explain to the user that the feature is unavailable because
                // the feature requires a permission that the user has denied.
                // At the same time, respect the user's decision. Don't link to
                // system settings in an effort to convince the user to change
                // their decision.
                S2Log.e("#### Permission not granted. ");
                Toast.makeText(this, "권한이 없습니다. 설정 > 애플리케이션 에서 해당 앱의 권한을 승인해주세요.", Toast.LENGTH_LONG).show();
            }
        }
        else {
            super.onRequestPermissionsResult(requestCode, permissions, grantResults);
        }

    }

}
