package s2.adapi.sdk.offerwall;

import android.app.Activity;
import android.app.AlertDialog;
import android.content.Context;
import android.content.Intent;
import android.content.SharedPreferences;
import android.view.LayoutInflater;
import android.view.View;

import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;

import s2.adapi.sdk.offerwall.impl.Constants;
import s2.adapi.sdk.offerwall.impl.SdkCore;

/**
 * The S2Offerwall class provides methods for interacting with the S2 Offerwall SDK.
 * It allows you to initialize the SDK, set user information, display offerwalls,
 * and manage user consent for data collection.
 *
 * <p>Key functionalities include:
 * <ul>
 *     <li>Initializing the SDK using {@link #initSdk(Context)} or {@link #initSdk(Context, InitializeListener)}.</li>
 *     <li>Setting and retrieving the user's unique identifier (username) using {@link #setUserName(Context, String)} and {@link #getUserName(Context)}.</li>
 *     <li>Starting the offerwall activity using {@link #startActivity(Context, String)} or {@link #startActivity(Context, String, boolean)}.</li>
 *     <li>Managing user consent for personalized ads through methods like {@link #setConsentDialogRequired(Context, boolean)},
 *         {@link #isConsentDialogRequired(Context)}, {@link #setConsentAgreed(Context, boolean)},
 *         {@link #isConsentAgreed(Context)}, and {@link #showConsentDialog(Context, Runnable)}.</li>
 *     <li>Setting a global event listener for offerwall events using {@link #setEventListener(EventListener)}.</li>
 *     <li>Retrieving the current Advertising ID using {@link #getADID()}.</li>
 *     <li>Setting a custom App ID using {@link #setAppId(Context, String)}.</li>
 * </ul>
 *
 * <p>The SDK requires initialization before most of its features can be used.
 * It's recommended to initialize the SDK early in your application's lifecycle,
 * typically in the {@code onCreate} method of your Application class or main Activity.
 *
 * <p>User consent is an important aspect. The SDK provides mechanisms to display a
 * consent dialog to users and record their preferences. You can control whether
 * the consent dialog is shown and customize its appearance.
 *
 * <p>The offerwall can be displayed in different ways, including as a standard activity
 * or as a popup-style activity. You can also specify a placement name to display
 * a specific offerwall configuration. The special placement name {@link #MAIN} refers
 * to the main offerwall.
 */
public class S2Offerwall {

    public static final String MAIN = "main";   // Mainwall 을 지칭하는 플레이스먼트 명칭

    private static SdkCore sdkCore = null;

    // Global Event listener
    // 개별 S2OfferwallView 에 설정된 EventListener 가 없는 경우 여기에 설정된 eventListener 를 호출 한다.
    static EventListener eventListener = null;

    // SdkCore 싱글턴 객체를 반환한다. SDK 내부에서만 접근 가능하다.
    static SdkCore getCoreInstance() {
        return sdkCore;
    }

    // 매체 정보와 기기 정보가 담겨있는 URL 에 pub_user_nm 을 추가하여 반환한다.
    static String getOfferwallUrl(Context context, String placementName, boolean isPopupStyle) {
        if (sdkCore == null) {
            return null;
        }

        boolean isMainwall = S2Offerwall.MAIN.equals(placementName);
        String offerwallUrl = sdkCore.getOfferwallUrl(isMainwall);

        String pubUserName = getUserName(context);
        if (!pubUserName.isEmpty()) {
            offerwallUrl = offerwallUrl + "&pub_user_nm=" + pubUserName;
        }

        if (placementName != null && !isMainwall) {
            try {
                placementName = URLEncoder.encode(placementName, "UTF-8");
            }
            catch (Exception ex) {
                //
            }
            offerwallUrl += "&plcmt_nm=" + placementName;
        }

        if (isPopupStyle) {
            offerwallUrl += "&embed_yn=P";
        }
        //S2Log.d("## offerwall url = " + offerwallUrl);

        return offerwallUrl;
    }

    /**
     * Return current Advertising ID
     * @return adid
     */
    public static String getADID() {
        if (sdkCore == null) {
            return null;
        }

        return sdkCore.getADID();
    }

    /**
     * Set Default EventListener
     * @param listener EventListner
     */
    public static void setEventListener(EventListener listener) {
        eventListener = listener;
    }

    // pub_user_nm 값을 SharedPreference 에 저장 한다. 저장 할 때 URL Encoding 하여 저장 한다.
    // 서버에서 URL decoding 이 안되어서 encoding 안하고 저장하자.

    /**
     * Set Unique Id of an user who is using the offerwall
     * @param context Activity or Context
     * @param userName unique Id of an user
     */
    public static void setUserName(Context context, String userName) {
//        String userNameEncoded = userName;
//        try {
//            //noinspection CharsetObjectCanBeUsed
//            userNameEncoded = URLEncoder.encode(userName, Constants.UTF_8);
//        }
//        catch (Exception ex) {
//            // encoding error
//            Log.e(Constants.LOG_TAG, "## setUserName() failed encoding utf-8.");
//            return false;
//        }

        SharedPreferences pref = Constants.getPreferences(context);
        SharedPreferences.Editor editor = pref.edit();
        editor.putString(Constants.PREF_PUB_USER_NAME, userName).apply();

        // userName 이 변경 되었음을 S2OfferwallView 에게 알린다. (observer pattern)
        if (sdkCore != null) {
            sdkCore.getCurrentUserName().setValue(userName);
        }
    }

    /**
     * Clear the user name
     * @param context Activity or Context
     */
    public static void resetUserName(Context context) {
        setUserName(context, "");
    }

    /**
     * return the current user name
     * @param context Activity or Context
     * @return current user name
     */
    public static String getUserName(Context context) {
        SharedPreferences pref = Constants.getPreferences(context);
        return pref.getString(Constants.PREF_PUB_USER_NAME, "");
    }

    /**
     * Set App ID.
     * App ID is an unique identifier for each app, issued by S2 platform.
     * The s2_app_id value set in AndroidManifest.xml is applied by default,
     * but if this method is called, the value set here takes precedence.
     * @param context Application context or Activity
     * @param appId App ID issued by S2 Server
     */
    public static void setAppId(Context context, String appId) {
        SharedPreferences pref = Constants.getPreferences(context);
        SharedPreferences.Editor editor = pref.edit();
        editor.putString(Constants.PREF_APP_ID, appId).apply();
    }

    /**
     * Set whether the consent dialog is required.
     * If set to true and the user has not explicitly agreed with it,
     * the consent dialog will be shown before displaying the offerwall.
     * If set to false, the consent dialog will be skipped.
     *
     * @param context Activity or Context
     * @param required true if the consent dialog is required, false otherwise.
     */
    public static void setConsentDialogRequired(Context context, boolean required) {
        SharedPreferences pref = Constants.getPreferences(context);
        SharedPreferences.Editor editor = pref.edit();
        editor.putString(Constants.PREF_CONSENT_SKIP, required ? "N" : "Y").apply();
    }

    /**
     * Checks if the consent dialog is required.
     * The consent dialog is required if the user has not explicitly agreed with it.
     *
     * @param context The context.
     * @return True if the consent dialog is required, false otherwise.
     */
    public static boolean isConsentDialogRequired(Context context) {
        SharedPreferences pref = Constants.getPreferences(context);
        return "N".equals(pref.getString(Constants.PREF_CONSENT_SKIP, "N"));
    }

    public static void setConsentAgreed(Context context, boolean agreed) {
        SharedPreferences pref = Constants.getPreferences(context);
        SharedPreferences.Editor editor = pref.edit();
        editor.putString(Constants.PREF_CONSENT_YN, agreed ? "Y" : "N").apply();
    }

    public static boolean isConsentAgreed(Context context) {
        SharedPreferences pref = Constants.getPreferences(context);
        return "Y".equals(pref.getString(Constants.PREF_CONSENT_YN, "N"));
    }

    public static boolean isConsentRequired(Context context) {
        // 동의 skip 이 안되고 사용자가 미동의 상태라면 동의 창을 띄워야한다.
        return (isConsentDialogRequired(context) && !isConsentAgreed(context));
    }

    /**
     * Start a S2OfferwallActivity with isPoppupStyle is false.
     * @param context Activity or Context
     * @param placementName name of a placement to be displayed in the offerwall. Use "main" to display the main offerwall.
     */
    public static void startActivity(Context context, String placementName) {
        startActivity(context, placementName, false);
    }

    /**
     * Start a S2OfferwallActivity.
     * @param context Activity or Context
     * @param placementName name of a placement to be displayed in the offerwall. Use "main" to display the main offerwall.
     * @param isPopupStyle if true, activity appears from bottom, otherwise the activity appears from right to left.
     */
    public static void startActivity(Context context, String placementName, boolean isPopupStyle) {
        if (isConsentRequired(context)) {
            // 동의 skip 이 안되고 사용자가 미동의 상태라면 동의 창을 띄워야한다.
            showConsentDialog(context, () -> {
                S2Offerwall.startActivityInternal(context, placementName, isPopupStyle);
            });
        }
        else {
            startActivityInternal(context, placementName, isPopupStyle);
        }
    }

    private static void startActivityInternal(Context context, String placementName, boolean isPopupStyle) {
        Intent intent = new Intent(context, S2OfferwallActivity.class);
        if (placementName != null) {
            intent.putExtra(Constants.INTENT_PLACEMENT_NAME, placementName);
        }

        if (context instanceof Activity && isPopupStyle) {
            // popup style 은 activity 에서 띄우는 경우만 가능하다.
            intent.putExtra(Constants.INTENT_PRESENT_TYPE, Constants.EXTRA_PRESENT_POPUP);

            Activity activity = (Activity) context;
            activity.startActivity(intent);
            activity.overridePendingTransition(R.anim.slide_from_bottom, 0);
        }
        else {
            context.startActivity(intent);
        }
    }

    public static void initSdk(Context context) {
        initSdk(context, null);
    }

    public static boolean isInitialized() {
        return (sdkCore != null);
    }

    public static void initSdk(Context context, final InitializeListener listener) {
        //final String userName = getUserName(context);

        Thread t = new Thread(new Runnable() {
            public void run() {
                if (sdkCore != null) {
                    // already initialized.
                    // adid update
                    sdkCore.updateAdid(context);

                    if (listener != null) {
                        listener.onSuccess();
                    }

                    return;
                }

                sdkCore = SdkCore.getInstance(context);

                if (sdkCore != null) {
                    // 성공
                    if (listener != null) {
                        listener.onSuccess();
                    }
                }
                else {
                    // 초기화 실패
                    if (listener != null) {
                        listener.onFailure();
                    }
                }

            }
        });

        t.start();
    }

    public static void showConsentDialog(Context context, int resIdDialogLayout, int resIdButtonAgree, int resIdButtonDeny,
                                         Runnable onAgree) {
        showConsentDialog(context, resIdDialogLayout, resIdButtonAgree, resIdButtonDeny, onAgree, null);
    }

    public static void showConsentDialog(Context context, int resIdDialogLayout, int resIdButtonAgree, int resIdButtonDeny,
                                         Runnable onAgree, Runnable onDeny) {
        // 이미 동의했으면 띄우지 않는다.
        if (!isConsentRequired(context)) {
            if (onAgree != null) {
                onAgree.run();
            }

            return;
        }

        // 동의창 띄우기
        View dialogView = LayoutInflater.from(context).inflate(resIdDialogLayout, null);

        AlertDialog dialog = new AlertDialog.Builder(context)
                .setView(dialogView)
                .setCancelable(false)
                .create();

        dialogView.findViewById(resIdButtonAgree).setOnClickListener(v -> {
            // 동의 처리
            setConsentAgreed(context, true);
            dialog.dismiss();

            if (onAgree != null) {
                onAgree.run();
            }
        });

        dialogView.findViewById(resIdButtonDeny).setOnClickListener(v -> {
            // 거부 처리
            setConsentAgreed(context, false);
            dialog.dismiss();

            if (onDeny != null) {
                onDeny.run();
            }
        });

        dialog.show();
    }

    public static void showConsentDialog(Context context, Runnable onAgree, Runnable onDeny) {
        showConsentDialog(context, R.layout.offerwall_consent_dialog, R.id.consentAgree, R.id.consentDeny, onAgree, onDeny);
    }

    public static void showConsentDialog(Context context, Runnable onAgree) {
        showConsentDialog(context, R.layout.offerwall_consent_dialog, R.id.consentAgree, R.id.consentDeny, onAgree, null);
    }

    public interface InitializeListener {
        void onSuccess();

        void onFailure();
    }

    public interface EventListener {
        // login://param
        void onLoginRequested(String param);

        // close://param
        void onCloseRequested(String param);

        // 이용문의 등에서 이미지 파일 첨부 등 실행시 필요한 권한을 요청한다.
        void onPermissionRequested(String permission);
    }
}
