/*
 * Copyright 2019 Skullabs Contributors (https://github.com/skullabs)
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kos.core.client;

import com.fasterxml.jackson.core.type.TypeReference;
import io.vertx.core.Future;
import io.vertx.core.MultiMap;
import io.vertx.core.http.HttpMethod;
import lombok.val;

/**
 * Abstract class used as base class for generated rest clients.
 */
public class AutoGeneratedRestClient {

    protected final RestClientConfiguration configuration;

    public AutoGeneratedRestClient(RestClientConfiguration configuration) {
        this.configuration = configuration;
    }

    protected MultiMap createEmptyHeaders(){
        return MultiMap.caseInsensitiveMultiMap()
            .addAll(configuration.getHeaders());
    }

    public <T> Future<T> execute(
        HttpMethod httpMethod, String uri,
        MultiMap headers,
        Class<T> expectedResponseType
    ){
        val handler = ResponseHandler.<T>create(configuration.getRestClientSerializer(), expectedResponseType);
        return execute(httpMethod, uri, headers, handler);
    }

    public <T> Future<T> execute(
        HttpMethod httpMethod, String uri,
        MultiMap headers,
        TypeReference<T> expectedResponseType
    ){
        val handler = ResponseHandler.<T>create(configuration.getRestClientSerializer(), expectedResponseType);
        return execute(httpMethod, uri, headers, handler);
    }

    private <T> Future<T> execute(
            HttpMethod httpMethod, String uri,
            MultiMap headers,
            ResponseHandler<T> handler
    ){
        val baseUri = configuration.getBaseUrl() + uri;
        configuration.getClient()
            .request(httpMethod, configuration.getPort(), configuration.getHost(), baseUri)
            .ssl(configuration.isHttps())
            .putHeaders(headers)
                .send(handler);

        return handler.future();
    }

    public <T> Future<T> send(
        HttpMethod httpMethod, String uri,
        MultiMap headers, Object payload,
        TypeReference<T> expectedResponseType
    ){
        val handler = ResponseHandler.<T>create(configuration.getRestClientSerializer(), expectedResponseType);
        return send(httpMethod, uri, headers, payload, handler);
    }

    public <T> Future<T> send(
        HttpMethod httpMethod, String uri,
        MultiMap headers, Object payload,
        Class<T> expectedResponseType
    ){
        val handler = ResponseHandler.<T>create(configuration.getRestClientSerializer(), expectedResponseType);
        return send(httpMethod, uri, headers, payload, handler);
    }

    protected <T> Future<T> send(
        HttpMethod httpMethod, String uri,
        MultiMap headers, Object payload,
        ResponseHandler<T> handler
    ){
        val serializer = configuration.getRestClientSerializer();
        headers.add("content-type", serializer.contentType());

        val baseUri = configuration.getBaseUrl() + uri;
        configuration.getClient()
            .request(httpMethod, configuration.getPort(), configuration.getHost(), baseUri)
            .ssl(configuration.isHttps())
            .putHeaders(headers)
            .sendBuffer(serializer.serialize(payload), handler);

        return handler.future();
    }
}
