/*
 * Copyright (C) 2022-2022 Huawei Technologies Co., Ltd. All rights reserved.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 */

package io.sermant.flowcontrol.res4j.handler;

import io.github.resilience4j.circuitbreaker.CircuitBreaker;
import io.github.resilience4j.circuitbreaker.CircuitBreakerConfig;
import io.github.resilience4j.circuitbreaker.CircuitBreakerConfig.SlidingWindowType;
import io.github.resilience4j.circuitbreaker.CircuitBreakerRegistry;
import io.sermant.flowcontrol.common.core.resolver.CircuitBreakerRuleResolver;
import io.sermant.flowcontrol.common.core.rule.CircuitBreakerRule;
import io.sermant.flowcontrol.common.entity.MetricEntity;
import io.sermant.flowcontrol.common.handler.AbstractRequestHandler;
import io.sermant.flowcontrol.common.util.StringUtils;
import io.sermant.flowcontrol.res4j.adaptor.CircuitBreakerAdaptor;
import io.sermant.flowcontrol.res4j.service.ServiceCollectorService;
import io.sermant.flowcontrol.res4j.util.MonitorUtils;

import java.time.Duration;
import java.util.Map;
import java.util.Optional;

/**
 * CircuitBreakerHandler
 *
 * @author zhouss
 * @since 2022-01-24
 */
public class CircuitBreakerHandler extends AbstractRequestHandler<CircuitBreaker, CircuitBreakerRule> {
    @Override
    protected final Optional<CircuitBreaker> createHandler(String businessName, CircuitBreakerRule rule) {
        final SlidingWindowType slidingWindowType = getSlidingWindowType(rule.getSlidingWindowType());
        CircuitBreaker circuitBreaker = CircuitBreakerRegistry.of(CircuitBreakerConfig.custom()
                .failureRateThreshold(rule.getFailureRateThreshold())
                .slowCallRateThreshold(rule.getSlowCallRateThreshold())
                .waitDurationInOpenState(Duration.ofMillis(rule.getParsedWaitDurationInOpenState()))
                .slowCallDurationThreshold(Duration.ofMillis(rule.getParsedSlowCallDurationThreshold()))
                .permittedNumberOfCallsInHalfOpenState(rule.getPermittedNumberOfCallsInHalfOpenState())
                .minimumNumberOfCalls(rule.getMinimumNumberOfCalls()).slidingWindowType(slidingWindowType)
                .slidingWindowSize(getWindowSize(slidingWindowType, rule.getParsedSlidingWindowSize())).build())
                .circuitBreaker(businessName);
        if (MonitorUtils.isStartMonitor()) {
            addEventConsumers(circuitBreaker);
            ServiceCollectorService.CIRCUIT_BREAKER_MAP.putIfAbsent(businessName, circuitBreaker);
        }
        return Optional.of(new CircuitBreakerAdaptor(circuitBreaker, rule));
    }

    /**
     * increased event consumption processing
     *
     * @param circuitBreaker circuitBreaker
     */
    private static void addEventConsumers(CircuitBreaker circuitBreaker) {
        Map<String, MetricEntity> monitors = ServiceCollectorService.MONITORS;
        MetricEntity metricEntity = monitors.computeIfAbsent(circuitBreaker.getName(), str -> new MetricEntity());
        metricEntity.setName(circuitBreaker.getName());
        circuitBreaker.getEventPublisher().onError(event -> {
            metricEntity.getFailedFuseRequest().getAndIncrement();
            metricEntity.getFuseRequest().getAndIncrement();
            metricEntity.getFuseTime().getAndAdd(event.getElapsedDuration().toMillis());
        }).onSuccess(event -> {
            metricEntity.getFuseRequest().getAndIncrement();
            metricEntity.getSuccessFulFuseRequest().getAndIncrement();
            metricEntity.getFuseTime().getAndAdd(event.getElapsedDuration().toMillis());
        }).onCallNotPermitted(event -> {
            metricEntity.getPermittedFulFuseRequest().getAndIncrement();
        }).onIgnoredError(event -> {
            metricEntity.getIgnoreFulFuseRequest().getAndIncrement();
            metricEntity.getFuseRequest().getAndIncrement();
            metricEntity.getFuseTime().getAndAdd(event.getElapsedDuration().toMillis());
        }).onSlowCallRateExceeded(event -> metricEntity.getSlowFuseRequest().getAndIncrement());
    }

    private int getWindowSize(SlidingWindowType slidingWindowType, long parsedSlidingWindowSize) {
        if (slidingWindowType == SlidingWindowType.COUNT_BASED) {
            return (int) parsedSlidingWindowSize;
        }

        // For the time being, rest4j only supports seconds as a time window,
        // where the parsedSlidingWindowSize is milliseconds, so convert to seconds here
        return (int) Duration.ofMillis(parsedSlidingWindowSize).getSeconds();
    }

    private CircuitBreakerConfig.SlidingWindowType getSlidingWindowType(String type) {
        if (StringUtils.equalIgnoreCase(type, CircuitBreakerRule.SLIDE_WINDOW_COUNT_TYPE)) {
            return CircuitBreakerConfig.SlidingWindowType.COUNT_BASED;
        }
        return CircuitBreakerConfig.SlidingWindowType.TIME_BASED;
    }

    @Override
    protected String configKey() {
        return CircuitBreakerRuleResolver.CONFIG_KEY;
    }
}
