package io.sendon.sms;

import java.io.File;
import java.util.List;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import io.sendon.base.SendonClient;
import io.sendon.base.SendonJsonResponse;
import io.sendon.sms.request.LmsBuilder;
import io.sendon.sms.request.MmsBuilder;
import io.sendon.sms.request.Reservation;
import io.sendon.sms.request.SmsBuilder;
import io.sendon.sms.request.ToWithBucket;
import io.sendon.sms.request.ToWithName;
import io.sendon.sms.response.CancelGroup;
import io.sendon.sms.response.CancelRepeat;
import io.sendon.sms.response.GenerateCsvPresignedUrl;
import io.sendon.sms.response.GetGroup;
import io.sendon.sms.response.GetImage;
import io.sendon.sms.response.GetRepeats;
import io.sendon.sms.response.SendGroup;
import io.sendon.sms.response.SendRepeat;
import io.sendon.sms.response.SendSms;
import io.sendon.sms.response.UploadImage;

public class SendonSms extends SendonClient {

  public enum MessageType {
    SMS("SMS"),
    LMS("LMS"),
    MMS("MMS");

    public final String value;

    MessageType(String messageType) {
      this.value = messageType;
    }
  }

  public SendonSms(String userId, String apiKey) {
    super(userId, apiKey);
  }

  public SendonSms(String userId, String apiKey, boolean useOkHttp) {
    super(userId, apiKey, useOkHttp);
  }

  /**
   * Sends an SMS message.
   *
   * @param from        The sender's phone number.
   * @param to          The list of recipients. Can be {@code List<String>}, {@code List<ToWithName>}, or {@code List<ToWithBucket>}.
   * @param message     The message content.
   * @param isAd        Whether the message is an advertisement.
   * @param reservation The reservation details for scheduling.
   * @return The response of the SMS send operation.
   * @see ToWithName
   * @see ToWithBucket
   */
  public SendSms sendSms(String from, List<? extends Object> to, String message, boolean isAd, Reservation reservation) {
    return send(MessageType.SMS, from, to, null, message, isAd, reservation, null);
  }

  /**
   * Sends an LMS message.
   *
   * @param from        The sender's phone number.
   * @param to          The list of recipients. Can be {@code List<String>}, {@code List<ToWithName>}, or {@code List<ToWithBucket>}.
   * @param title       The title of the message.
   * @param message     The message content.
   * @param isAd        Whether the message is an advertisement.
   * @param reservation The reservation details for scheduling.
   * @return The response of the LMS send operation.
   * @see ToWithName
   * @see ToWithBucket
   */
  public SendSms sendLms(String from, List<? extends Object> to, String title, String message, boolean isAd, Reservation reservation) {
    return send(MessageType.LMS, from, to, title, message, isAd, reservation, null);
  }

  /**
   * Sends an MMS message.
   *
   * @param from        The sender's phone number.
   * @param to          The list of recipients. Can be {@code List<String>}, {@code List<ToWithName>}, or {@code List<ToWithBucket>}.
   * @param title       The title of the message.
   * @param message     The message content.
   * @param isAd        Whether the message is an advertisement.
   * @param reservation The reservation details for scheduling.
   * @param images      The list of image URLs to include in the MMS.
   * @return The response of the MMS send operation.
   * @see ToWithName
   * @see ToWithBucket
   */
  public SendSms sendMms(String from, List<? extends Object> to, String title, String message, boolean isAd, Reservation reservation, List<String> images) {
    return send(MessageType.MMS, from, to, title, message, isAd, reservation, images);
  }

  /**
   * Sends an SMS message using a builder.
   *
   * @param builder The {@link SmsBuilder} containing the SMS message details.
   * @return The response of the SMS send operation.
   * @see SmsBuilder
   */
  public SendSms sendSms(SmsBuilder builder) {
    return send(MessageType.SMS, builder.from, builder.to, null, builder.message, builder.isAd, builder.reservation, null);
  }

  /**
   * Sends an LMS message using a builder.
   *
   * @param builder The {@link LmsBuilder} containing the LMS message details.
   * @return The response of the LMS send operation.
   * @see LmsBuilder
   */
  public SendSms sendLms(LmsBuilder builder) {
    return send(MessageType.LMS, builder.from, builder.to, builder.title, builder.message, builder.isAd, builder.reservation, null);
  }

  /**
   * Sends an MMS message using a builder.
   *
   * @param builder The {@link MmsBuilder} containing the MMS message details.
   * @return The response of the MMS send operation.
   * @see MmsBuilder
   */
  public SendSms sendMms(MmsBuilder builder) {
    return send(MessageType.MMS, builder.from, builder.to, builder.title, builder.message, builder.isAd, builder.reservation, builder.images);
  }

  private <T> SendSms send(MessageType type, String from, List<T> to, String title, String message, Boolean isAd, Reservation reservation, List<String> images) {
    if (type == null) {
        throw new IllegalArgumentException("MessageType cannot be null");
    }
    if (from == null || from.isEmpty()) {
        throw new IllegalArgumentException("Sender 'from' cannot be null or empty");
    }
    if (to == null || to.isEmpty()) {
        throw new IllegalArgumentException("Recipient list 'to' cannot be null or empty");
    }
    if (message == null || message.isEmpty()) {
        throw new IllegalArgumentException("Message content cannot be null or empty");
    }
    if (isAd == null) {
        throw new IllegalArgumentException("'isAd' flag cannot be null");
    }

    JsonObject bodyJson = new JsonObject();
    bodyJson.addProperty("type", type.value);
    bodyJson.addProperty("from", from);

    JsonArray toArray = new JsonArray();
    for (T recipient : to) {
        if (recipient instanceof String) {
            toArray.add((String) recipient);
        } else if (recipient instanceof ToWithName) {
            JsonObject toObject = new JsonObject();
            toObject.addProperty("phone", ((ToWithName) recipient).phone);
            toObject.addProperty("name", ((ToWithName) recipient).name);
            toArray.add(toObject);
        } else if (recipient instanceof ToWithBucket) {
            JsonObject toObject = new JsonObject();
            toObject.addProperty("bucket", ((ToWithBucket) recipient).bucket);
            toObject.addProperty("key", ((ToWithBucket) recipient).key);
            toArray.add(toObject);
        } else {
            throw new IllegalArgumentException("Unsupported recipient type: " + recipient.getClass().getName());
        }
    }

    bodyJson.add("to", toArray);
    if (title != null) bodyJson.addProperty("title", title);
    bodyJson.addProperty("message", message);
    bodyJson.addProperty("isAd", isAd);

    if (images != null && !images.isEmpty()) {
        JsonArray imagesArray = new JsonArray();
        for (String image : images) {
            imagesArray.add(image);
        }
        bodyJson.add("images", imagesArray);
    }

    if (reservation != null) {
        JsonObject reservationJson = new JsonObject();
        reservationJson.addProperty("datetime", reservation.datetime);

        if (reservation.repeat != null) {
            JsonObject repeatJson = new JsonObject();
            repeatJson.addProperty("limit", reservation.repeat.messageCount);
            repeatJson.addProperty("unit", reservation.repeat.unit.value);
            repeatJson.addProperty("interval", reservation.repeat.interval);
            repeatJson.addProperty("denyNightTime", reservation.repeat.denyNightTime);
            reservationJson.add("repeat", repeatJson);
        }

        bodyJson.add("reservation", reservationJson);
    }

    try {
        SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms", bodyJson.toString()));
        return new SendSms(sendonJsonResponse);
    } catch (Exception e) {
        e.printStackTrace();
    }
    return null;
}

  /**
   * Retrieves details of a message group.
   *
   * @param groupId The ID of the message group to retrieve.
   * @return The details of the message group.
   */
  public GetGroup getGroup(String groupId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(get("/v2/messages/sms/groups/" + groupId));
      return new GetGroup(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Sends a message group.
   *
   * @param groupId The ID of the message group to send.
   * @return The response of the send operation.
   */
  public SendGroup sendGroup(String groupId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms/groups/" + groupId + "/send", null));
      return new SendGroup(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;

  }

  /**
   * Cancels a message group.
   *
   * @param groupId The ID of the message group to cancel.
   * @return The response of the cancel operation.
   */
  public CancelGroup cancelGroup(String groupId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms/groups/" + groupId + "/cancel", null));
      return new CancelGroup(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Retrieves repeat messages for a group.
   *
   * @param groupId The ID of the message group.
   * @param limit   The maximum number of repeat messages to retrieve.
   * @param cursor  The cursor for pagination (optional).
   * @return The repeat messages for the group.
   */
  public GetRepeats getRepeats(String groupId, int limit, String cursor) {
    if (cursor == null) cursor = "";

    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(get("/v2/messages/sms/groups/" + groupId + "/repeats?limit=" + limit + "&cursor=" + cursor));
      return new GetRepeats(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Sends a repeat message for a group.
   *
   * @param groupId  The ID of the message group.
   * @param repeatId The ID of the repeat message to send.
   * @return The response of the send operation.
   */
  public SendRepeat sendRepeat(String groupId, String repeatId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms/groups/" + groupId + "/repeats/" + repeatId + "/send", null));
      return new SendRepeat(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Cancels a repeat message for a group.
   *
   * @param groupId  The ID of the message group.
   * @param repeatId The ID of the repeat message to cancel.
   * @return The response of the cancel operation.
   */
  public CancelRepeat cancelRepeat(String groupId, String repeatId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms/groups/" + groupId + "/repeats/" + repeatId + "/cancel", null));
      return new CancelRepeat(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Generates a presigned URL for uploading a CSV file.
   *
   * @return The presigned URL for uploading a CSV file.
   */
  public GenerateCsvPresignedUrl generateCsvPresignedUrl() {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/sms/csv/presigned", null));
      return new GenerateCsvPresignedUrl(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Uploads images for use in MMS messages.
   *
   * @param images The list of image files to upload.
   * @return The response of the upload operation.
   */
  public UploadImage uploadImages(List<File> images) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(postImagesWithMultipartFormData("/v2/messages/sms/image/upload", images));
      return new UploadImage(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Retrieves URLs for images.
   *
   * @param images The list of image IDs to retrieve URLs for.
   * @return The response containing the image URLs.
   */
  public GetImage getImages(List<String> images) {
    try {
      int index = 1;
      String query = "";
      for(String image : images) {
        query += "imageId" + index + "=" + image + "&";
        index++;
      }
      query+="&expireIn=60";

      SendonJsonResponse sendonJsonResponse = parseJsonResponse(get("/v2/messages/sms/image/url?" + query));
      return new GetImage(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }
}