package io.sendon.rcs;

import java.util.List;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import io.sendon.base.SendonClient;
import io.sendon.base.SendonJsonResponse;
import io.sendon.rcs.request.Fallback;
import io.sendon.rcs.request.RbcConfig;
import io.sendon.rcs.request.RcsBuilder;
import io.sendon.rcs.response.CreateCsvUploadUrl;
import io.sendon.rcs.response.GetRcs;
import io.sendon.rcs.response.SendRcs;

public class SendonRcs extends SendonClient {

  public enum MessageType {
    RCS_TSM("RcsTSM"),
    RCS_TLM("RcsTLM"),
    RCS_TMM("RcsTMM");

    public final String value;

    MessageType(String messageType) {
      this.value = messageType;
    }
  }

  public SendonRcs(String userId, String apiKey) {
    super(userId, apiKey);
  }

  public SendonRcs(String userId, String apiKey, boolean useOkHttp) {
    super(userId, apiKey, useOkHttp);
  }

  /**
   * Creates a presigned URL for CSV file upload.
   *
   * @return The response containing the presigned URL for CSV upload.
   */
  public CreateCsvUploadUrl createCsvUploadUrl() {
    String endpoint = "/v2/messages/rcs/csv/presigned";
    JsonObject params = new JsonObject();

    SendonJsonResponse response = parseJsonResponse(post(endpoint, params.toString()));
    return new CreateCsvUploadUrl(response);
  }

  /**
   * Sends an RCS message using a builder.
   *
   * @param builder The {@link RcsBuilder} containing the RCS message details.
   * @return The response of the RCS send operation.
   * @see RcsBuilder
   */
  public SendRcs send(RcsBuilder builder) {
    return send(builder.type, builder.from, builder.to, builder.rbcConfig, builder.useCredit, builder.useFallback, builder.fallback);
  }

  /**
   * Sends an RCS message.
   *
   * @param type        The RCS message type (RcsTSM, RcsTLM, RcsTMM).
   * @param from        The sender's phone number.
   * @param to          The list of recipient phone numbers.
   * @param rbcConfig   The RCS business configuration.
   * @param useCredit   Whether to use credit for sending.
   * @param useFallback Whether to use fallback SMS when RCS fails.
   * @param fallback    The fallback message configuration.
   * @return The response of the RCS send operation.
   */
  public SendRcs send(MessageType type, String from, List<String> to, RbcConfig rbcConfig,
      Boolean useCredit, Boolean useFallback, Fallback fallback) {
    String endpoint = "/v2/messages/rcs";
    JsonObject params = new JsonObject();

    params.addProperty("type", type.value);
    params.addProperty("from", from);

    JsonArray toArray = new JsonArray();
    for (String recipient : to) {
      toArray.add(recipient);
    }
    params.add("to", toArray);

    if (rbcConfig != null) {
      params.add("rbc", rbcConfig.toJson());
    }

    if (useCredit != null) {
      params.addProperty("useCredit", useCredit);
    }

    if (useFallback != null) {
      params.addProperty("useFallback", useFallback);
    }

    if (fallback != null && useFallback != null && useFallback) {
      params.add("fallback", fallback.toJson());
    }

    SendonJsonResponse response = parseJsonResponse(post(endpoint, params.toString()));
    return new SendRcs(response);
  }

  /**
   * Finds and retrieves details of an RCS message group.
   *
   * @param groupId The group ID to search for.
   * @return The response containing the RCS message details.
   */
  public GetRcs find(String groupId) {
    String endpoint = "/v2/messages/rcs/groups/" + groupId;

    SendonJsonResponse response = parseJsonResponse(get(endpoint));
    return new GetRcs(response);
  }
}