package io.sendon.payment;

import io.sendon.Log;
import io.sendon.base.SendonClient;
import io.sendon.base.SendonJsonResponse;
import io.sendon.payment.response.GetPaymentHistories;
import io.sendon.payment.response.GetPaymentHistory;

/**
 * SendonPayment is a client wrapper for interacting with the Sendon API.
 * It provides methods to retrieve payment-related information such as payment histories and details.
 */
public class SendonPayment extends SendonClient {

  /**
   * Constructor to initialize SendonPayment with user credentials.
   *
   * @param userId The user ID for authentication.
   * @param apiKey The API key for authentication.
   */
  public SendonPayment(String userId, String apiKey) {
    super(userId, apiKey);
  }

  /**
   * Constructor to initialize SendonPayment with user credentials and an option to use OkHttp.
   *
   * @param userId The user ID for authentication.
   * @param apiKey The API key for authentication.
   * @param useOkHttp Flag to indicate whether to use OkHttp for HTTP requests.
   */
  public SendonPayment(String userId, String apiKey, boolean useOkHttp) {
    super(userId, apiKey, useOkHttp);
  }

  /**
   * Fetches a paginated list of payment histories from the Sendon API.
   *
   * @param page The page number for pagination.
   * @param limit The maximum number of results to fetch per page.
   * @param startDate The start date for filtering payment histories (optional).
   * @param endDate The end date for filtering payment histories (optional).
   * @return A GetPaymentHistories object containing the response data, or null if an error occurs.
   */
  public GetPaymentHistories getPaymentHistories(int page, int limit, String startDate, String endDate) {
    Log.d("getPaymentHistories: page=" + page + ", limit=" + limit + ", startDate=" + startDate + ", endDate=" + endDate);

    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(get( "/v2/payment/payment-histories?page=" + page + "&limit=" + limit +
      (startDate != null ? "&startDate=" + startDate : "") +
      (endDate != null ? "&endDate=" + endDate : "")).toString());
      return new GetPaymentHistories(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Fetches the details of a specific payment history from the Sendon API.
   *
   * @param historyId The ID of the payment history to retrieve.
   * @return A GetPaymentHistory object containing the payment history details, or null if an error occurs.
   */
  public GetPaymentHistory getPaymentHistory(int historyId) {
    Log.d("getPaymentHistory: historyId=" + historyId);

    try {
      SendonJsonResponse sendonResponse = parseJsonResponse(get("/v2/payment/payment-histories/" + historyId));
      return new GetPaymentHistory(sendonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }
}
