package io.sendon.kakao.request;

import java.util.List;

/**
 * Builder class for constructing Friendtalk messages.
 * Use this class to set the required and optional parameters for sending a Friendtalk message.
 */
public class FriendtalkBuilder {
    public String profileId;
    public String templateId;
    public List<?> to;
    public String message;
    public List<Button> buttons;
    public boolean isAd;
    public Reservation reservation;
    public Fallback fallback;
    public Image image;
    public FriendtalkMessageType messageType = FriendtalkMessageType.FT;

    /**
     * Sets the message type.
     * @param messageType
     * @return The current instance of {@link FriendtalkBuilder}.
     */
    public FriendtalkBuilder setMessageType(FriendtalkMessageType messageType) {
        this.messageType = messageType;
        return this;
    }

    /**
     * Sets the profile ID.
     *
     * @param profileId The profile ID.
     * @return The current instance of {@link FriendtalkBuilder}.
     */
    public FriendtalkBuilder setProfileId(String profileId) {
        this.profileId = profileId;
        return this;
    }

    /**
     * Sets the template ID.
     *
     * @param templateId The template ID.
     * @return The current instance of {@link FriendtalkBuilder}.
     */
    public FriendtalkBuilder setTemplateId(String templateId) {
        this.templateId = templateId;
        return this;
    }

    /**
     * Sets the list of recipients.
     *
     * @param to The list of recipients. Can be {@code List<String>} (phone numbers) or {@code List<ToWithVariable>} (recipients with variables).
     * @return The current instance of {@link FriendtalkBuilder}.
     */
    public FriendtalkBuilder setTo(List<?> to) {
        this.to = to; // Assumes the caller provides a valid list of supported types.
        return this;
    }

    /**
     * Sets the message content.
     *
     * @param message The message content.
     * @return The current instance of {@link FriendtalkBuilder}.
     */
    public FriendtalkBuilder setMessage(String message) {
        this.message = message;
        return this;
    }

    /**
     * Sets the list of buttons.
     *
     * @param buttons The list of buttons. See {@link Button}.
     * @return The current instance of {@link FriendtalkBuilder}.
     */
    public FriendtalkBuilder setButtons(List<Button> buttons) {
        this.buttons = buttons;
        return this;
    }

    /**
     * Sets whether the message is an advertisement.
     *
     * @param isAd {@code true} if the message is an advertisement, {@code false} otherwise.
     * @return The current instance of {@link FriendtalkBuilder}.
     */
    public FriendtalkBuilder setIsAd(boolean isAd) {
        this.isAd = isAd;
        return this;
    }

    /**
     * Sets the reservation details for scheduling.
     *
     * @param reservation The reservation details. See {@link Reservation}.
     * @return The current instance of {@link FriendtalkBuilder}.
     */
    public FriendtalkBuilder setReservation(Reservation reservation) {
        this.reservation = reservation;
        return this;
    }

    /**
     * Sets the fallback options.
     *
     * @param fallback The fallback options. See {@link Fallback}.
     * @return The current instance of {@link FriendtalkBuilder}.
     */
    public FriendtalkBuilder setFallback(Fallback fallback) {
        this.fallback = fallback;
        return this;
    }

    /**
     * Sets the image details.
     *
     * @param image The image details. See {@link Image}.
     * @return The current instance of {@link FriendtalkBuilder}.
     */
    public FriendtalkBuilder setImage(Image image) {
        this.image = image;
        return this;
    }
}
