package io.sendon.kakao;

import java.io.File;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Locale;

import com.google.gson.JsonArray;
import com.google.gson.JsonObject;

import io.sendon.base.SendonClient;
import io.sendon.base.SendonJsonResponse;
import io.sendon.kakao.brandmessage.request.BrandMessageCarouselFeedTemplateRequest;
import io.sendon.kakao.brandmessage.request.BrandMessageImageTemplateRequest;
import io.sendon.kakao.brandmessage.request.BrandMessageSendBuilder;
import io.sendon.kakao.brandmessage.request.BrandMessageTemplateQuery;
import io.sendon.kakao.brandmessage.request.BrandMessageTextTemplateRequest;
import io.sendon.kakao.brandmessage.request.BrandMessageWideItemListTemplateRequest;
import io.sendon.kakao.brandmessage.request.BrandMessageWideTemplateRequest;
import io.sendon.kakao.brandmessage.response.BrandMessageDeleteResponse;
import io.sendon.kakao.brandmessage.response.BrandMessageImageUploadResponse;
import io.sendon.kakao.brandmessage.response.BrandMessageSendResponse;
import io.sendon.kakao.brandmessage.response.BrandMessageTemplateDetailResponse;
import io.sendon.kakao.brandmessage.response.BrandMessageTemplateIdResponse;
import io.sendon.kakao.brandmessage.response.BrandMessageTemplateListResponse;
import io.sendon.kakao.request.AlButton;
import io.sendon.kakao.request.AlimtalkBuilder;
import io.sendon.kakao.request.Button;
import io.sendon.kakao.request.Fallback;
import io.sendon.kakao.request.FriendtalkBuilder;
import io.sendon.kakao.request.FriendtalkMessageType;
import io.sendon.kakao.request.Image;
import io.sendon.kakao.request.Profile;
import io.sendon.kakao.request.Reservation;
import io.sendon.kakao.request.Template;
import io.sendon.kakao.request.TemplateQuerySort;
import io.sendon.kakao.request.TemplateStatus;
import io.sendon.kakao.request.WlButton;
import io.sendon.kakao.response.CancelGroup;
import io.sendon.kakao.response.CancelReviewTemplate;
import io.sendon.kakao.response.CreateTemplate;
import io.sendon.kakao.response.DeleteTemplate;
import io.sendon.kakao.response.GetGroup;
import io.sendon.kakao.response.GetProfileDetail;
import io.sendon.kakao.response.GetProfiles;
import io.sendon.kakao.response.GetTemplateDetail;
import io.sendon.kakao.response.GetTemplates;
import io.sendon.kakao.response.RegisterProfile;
import io.sendon.kakao.response.RequestAuthToken;
import io.sendon.kakao.response.RequestReviewTemplate;
import io.sendon.kakao.response.SendAlimtalk;
import io.sendon.kakao.response.SendFriendtalk;
import io.sendon.kakao.response.SendGroup;
import io.sendon.kakao.response.UpdateTemplate;
import io.sendon.kakao.response.UploadAlimtalkImage;
import io.sendon.kakao.response.UploadFallbackImage;

/**
 * The {@code SendonKakao} class provides methods to interact with the Kakao messaging API.
 * It supports operations such as sending messages, managing templates, and handling profiles.
 */
public class SendonKakao extends SendonClient {

  /**
   * Constructs a new {@code SendonKakao} instance with the specified user ID and API key.
   *
   * @param userId The user ID for authentication.
   * @param apiKey The API key for authentication.
   */
  public SendonKakao(String userId, String apiKey) {
    super(userId, apiKey);
  }

  /**
   * Constructs a new {@code SendonKakao} instance with the specified user ID, API key, and HTTP client option.
   *
   * @param userId    The user ID for authentication.
   * @param apiKey    The API key for authentication.
   * @param useOkHttp Whether to use OkHttp as the HTTP client.
   */
  public SendonKakao(String userId, String apiKey, boolean useOkHttp) {
    super(userId, apiKey, useOkHttp);
  }

  /**
   * Requests an authentication token for a specific channel and phone number.
   *
   * @param channelId   The ID of the channel.
   * @param phoneNumber The phone number to authenticate.
   * @return The response containing the authentication token. See {@link RequestAuthToken}.
   */
  public RequestAuthToken requestAuthToken(String channelId, String phoneNumber) {
    JsonObject bodyJson = new JsonObject();
    bodyJson.addProperty("channelId", channelId);
    bodyJson.addProperty("phoneNumber", phoneNumber);

    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/kakao/send-profiles/request-auth-token", bodyJson.toString()));
      return new RequestAuthToken(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Registers a new profile with the given details.
   *
   * @param profile The profile details to register. See {@link Profile}.
   * @return The response of the profile registration. See {@link RegisterProfile}.
   */
  public RegisterProfile registerProfile(Profile profile) {
    JsonObject bodyJson = new JsonObject();
    if(profile.id != null && !profile.id.isEmpty()) bodyJson.addProperty("id", profile.id);
    if(profile.token != null && !profile.token.isEmpty()) bodyJson.addProperty("token", profile.token);
    if(profile.phoneNumber != null && !profile.phoneNumber.isEmpty()) bodyJson.addProperty("phoneNumber", profile.phoneNumber);
    if(profile.channelId != null && !profile.channelId.isEmpty()) bodyJson.addProperty("channelId", profile.channelId);

    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/kakao/send-profiles/register", bodyJson.toString()));
      return new RegisterProfile(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Retrieves a list of profiles with pagination support.
   *
   * @param limit  The maximum number of profiles to retrieve.
   * @param cursor The cursor for pagination.
   * @return The response containing the list of profiles. See {@link GetProfiles}.
   */
  public GetProfiles getProfiles(int limit, String cursor) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(get("/v2/messages/kakao/send-profiles?limit=" + limit + "&cursor=" + cursor));
      return new GetProfiles(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Retrieves the details of a specific profile.
   *
   * @param profileId The ID of the profile to retrieve.
   * @return The response containing the profile details. See {@link GetProfileDetail}.
   */
  public GetProfileDetail getProfileDetail(String profileId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(get("/v2/messages/kakao/send-profiles/" + profileId));
      return new GetProfileDetail(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Creates a new template for a specific profile.
   *
   * @param profileId The ID of the profile.
   * @param template  The template details to create. See {@link Template}.
   * @return The response of the template creation. See {@link CreateTemplate}.
   */
  public CreateTemplate createTemplate(String profileId, Template template) {
    JsonObject bodyJson = new JsonObject();
    if (template.templateName != null && !template.templateName.isEmpty()) bodyJson.addProperty("templateName", template.templateName);
    if (template.templateContent != null && !template.templateContent.isEmpty()) bodyJson.addProperty("templateContent", template.templateContent);
    if (template.templateMessageType != null) bodyJson.addProperty("templateMessageType", template.templateMessageType.value);
    if (template.templateEmphasizeType != null) bodyJson.addProperty("templateEmphasizeType", template.templateEmphasizeType.value);
    if (template.templateExtra != null && !template.templateExtra.isEmpty()) bodyJson.addProperty("templateExtra", template.templateExtra);
    if (template.templateAd != null && !template.templateAd.isEmpty()) bodyJson.addProperty("templateAd", template.templateAd);
    if (template.templateTitle != null && !template.templateTitle.isEmpty()) bodyJson.addProperty("templateTitle", template.templateTitle);
    if (template.templateSubtitle != null && !template.templateSubtitle.isEmpty()) bodyJson.addProperty("templateSubtitle", template.templateSubtitle);
    if (template.templateImageName != null && !template.templateImageName.isEmpty()) bodyJson.addProperty("templateImageName", template.templateImageName);
    if (template.templateImageUrl != null && !template.templateImageUrl.isEmpty()) bodyJson.addProperty("templateImageUrl", template.templateImageUrl);
    bodyJson.addProperty("securityFlag", template.securityFlag);

    JsonArray buttonsArray = new JsonArray();
    for (Button button : template.buttons) {
      JsonObject buttonObject = new JsonObject();
      buttonObject.addProperty("name", button.name);
      buttonObject.addProperty("ordering", button.ordering);
      buttonObject.addProperty("type", button.type.value);

      if (button instanceof WlButton) {
        WlButton wlButton = (WlButton) button;
        buttonObject.addProperty("urlMobile", wlButton.urlMobile);
        buttonObject.addProperty("urlPc", wlButton.urlPc);
      } else if (button instanceof AlButton) {
        AlButton alButton = (AlButton) button;
        buttonObject.addProperty("schemeIos", alButton.schemeIos);
        buttonObject.addProperty("schemeAndroid", alButton.schemeAndroid);
      }

      buttonsArray.add(buttonObject);
    }
    bodyJson.add("buttons", buttonsArray);

    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/kakao/send-profiles/" + profileId + "/templates", bodyJson.toString()));
      return new CreateTemplate(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Retrieves a list of templates for a specific profile with filtering and sorting options.
   *
   * @param profileId The ID of the profile.
   * @param keyword   The keyword to filter templates.
   * @param limit     The maximum number of templates to retrieve.
   * @param cursor    The cursor for pagination.
   * @param status    The status to filter templates. See {@link TemplateStatus}.
   * @param sort      The sorting option for templates. See {@link TemplateQuerySort}.
   * @return The response containing the list of templates. See {@link GetTemplates}.
   */
  public GetTemplates getTemplates(String profileId, String keyword, int limit, String cursor, TemplateStatus status, TemplateQuerySort sort) {
    try {
      String query = "";
      if (keyword != null && !keyword.isEmpty()) query += "&keyword=" + keyword;
      if (limit > 0) query += "&limit=" + limit;
      if (cursor != null && !cursor.isEmpty()) query += "&nextCursor=" + cursor;
      if (status != null) query += "&status=" + status.value;
      if (sort != null) query += "&sort=" + sort.querySort;

      SendonJsonResponse sendonJsonResponse = parseJsonResponse(get("/v2/messages/kakao/send-profiles/" + profileId + "/templates?" + query));
      return new GetTemplates(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Deletes a specific template for a profile.
   *
   * @param profileId  The ID of the profile.
   * @param templateId The ID of the template to delete.
   * @return The response of the template deletion. See {@link DeleteTemplate}.
   */
  public DeleteTemplate deleteTemplate(String profileId, String templateId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/kakao/send-profiles/" + profileId + "/templates/" + templateId + "/delete", null));
      return new DeleteTemplate(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Updates an existing template for a specific profile.
   *
   * @param profileId  The ID of the profile.
   * @param templateId The ID of the template to update.
   * @param template   The updated template details. See {@link Template}.
   * @return The response of the template update. See {@link UpdateTemplate}.
   */
  public UpdateTemplate updateTemplate(String profileId, String templateId, Template template) {
    JsonObject bodyJson = new JsonObject();
    if (template.templateName != null && !template.templateName.isEmpty()) bodyJson.addProperty("templateName", template.templateName);
    if (template.templateContent != null && !template.templateContent.isEmpty()) bodyJson.addProperty("templateContent", template.templateContent);
    if (template.templateMessageType != null) bodyJson.addProperty("templateMessageType", template.templateMessageType.value);
    if (template.templateEmphasizeType != null) bodyJson.addProperty("templateEmphasizeType", template.templateEmphasizeType.value);
    if (template.templateExtra != null && !template.templateExtra.isEmpty()) bodyJson.addProperty("templateExtra", template.templateExtra);
    if (template.templateAd != null && !template.templateAd.isEmpty()) bodyJson.addProperty("templateAd", template.templateAd);
    if (template.templateTitle != null && !template.templateTitle.isEmpty()) bodyJson.addProperty("templateTitle", template.templateTitle);
    if (template.templateSubtitle != null && !template.templateSubtitle.isEmpty()) bodyJson.addProperty("templateSubtitle", template.templateSubtitle);
    if (template.templateImageName != null && !template.templateImageName.isEmpty()) bodyJson.addProperty("templateImageName", template.templateImageName);
    if (template.templateImageUrl != null && !template.templateImageUrl.isEmpty()) bodyJson.addProperty("templateImageUrl", template.templateImageUrl);

    bodyJson.addProperty("securityFlag", template.securityFlag);

    JsonArray buttonsArray = new JsonArray();
    for (Button button : template.buttons) {
      JsonObject buttonObject = new JsonObject();
      buttonObject.addProperty("name", button.name);
      buttonObject.addProperty("ordering", button.ordering);
      buttonObject.addProperty("type", button.type.value);

      if (button instanceof WlButton) {
        WlButton wlButton = (WlButton) button;
        buttonObject.addProperty("urlMobile", wlButton.urlMobile);
        buttonObject.addProperty("urlPc", wlButton.urlPc);
      } else if (button instanceof AlButton) {
        AlButton alButton = (AlButton) button;
        buttonObject.addProperty("schemeIos", alButton.schemeIos);
        buttonObject.addProperty("schemeAndroid", alButton.schemeAndroid);
      }

      buttonsArray.add(buttonObject);
    }
    bodyJson.add("buttons", buttonsArray);

    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/kakao/send-profiles/" + profileId + "/templates/" + templateId + "/update", bodyJson.toString()));
      return new UpdateTemplate(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Requests a review for a specific template.
   *
   * @param profileId  The ID of the profile.
   * @param templateId The ID of the template to review.
   * @return The response of the review request. See {@link RequestReviewTemplate}.
   */
  public RequestReviewTemplate requestReviewTemplate(String profileId, String templateId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/kakao/send-profiles/" + profileId + "/templates/" + templateId + "/review", null));
      return new RequestReviewTemplate(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public CancelReviewTemplate cancelReviewTemplate(String profileId, String templateId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/kakao/send-profiles/" + profileId + "/templates/" + templateId + "/review-cancel", null));
      return new CancelReviewTemplate(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public GetTemplateDetail getTemplateDetail(String profileId, String templateId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(get("/v2/messages/kakao/send-profiles/" + profileId + "/templates/" + templateId));
      return new GetTemplateDetail(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  // ===================================
  // Brand Message Templates
  // ===================================

  public BrandMessageTemplateIdResponse createBrandMessageTextTemplate(
      String sendProfileId,
      BrandMessageTextTemplateRequest request
  ) {
    try {
      JsonObject bodyJson = request.toJson();
      SendonJsonResponse response = parseJsonResponse(post(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates/text",
          bodyJson.toString()
      ));
      return new BrandMessageTemplateIdResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageTemplateIdResponse createBrandMessageImageTemplate(
      String sendProfileId,
      BrandMessageImageTemplateRequest request
  ) {
    try {
      JsonObject bodyJson = request.toJson();
      SendonJsonResponse response = parseJsonResponse(post(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates/image",
          bodyJson.toString()
      ));
      return new BrandMessageTemplateIdResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageTemplateIdResponse createBrandMessageWideTemplate(
      String sendProfileId,
      BrandMessageWideTemplateRequest request
  ) {
    try {
      JsonObject bodyJson = request.toJson();
      SendonJsonResponse response = parseJsonResponse(post(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates/wide",
          bodyJson.toString()
      ));
      return new BrandMessageTemplateIdResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageTemplateIdResponse createBrandMessageWideItemListTemplate(
      String sendProfileId,
      BrandMessageWideItemListTemplateRequest request
  ) {
    try {
      JsonObject bodyJson = request.toJson();
      SendonJsonResponse response = parseJsonResponse(post(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates/wide-item-list",
          bodyJson.toString()
      ));
      return new BrandMessageTemplateIdResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageTemplateIdResponse createBrandMessageCarouselFeedTemplate(
      String sendProfileId,
      BrandMessageCarouselFeedTemplateRequest request
  ) {
    try {
      JsonObject bodyJson = request.toJson();
      SendonJsonResponse response = parseJsonResponse(post(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates/carousel-feed",
          bodyJson.toString()
      ));
      return new BrandMessageTemplateIdResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageTemplateIdResponse updateBrandMessageTextTemplate(
      String sendProfileId,
      String templateCode,
      BrandMessageTextTemplateRequest request
  ) {
    try {
      JsonObject bodyJson = request.toJson();
      SendonJsonResponse response = parseJsonResponse(post(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates/" + templateCode + "/update/text",
          bodyJson.toString()
      ));
      return new BrandMessageTemplateIdResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageTemplateIdResponse updateBrandMessageImageTemplate(
      String sendProfileId,
      String templateCode,
      BrandMessageImageTemplateRequest request
  ) {
    try {
      JsonObject bodyJson = request.toJson();
      SendonJsonResponse response = parseJsonResponse(post(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates/" + templateCode + "/update/image",
          bodyJson.toString()
      ));
      return new BrandMessageTemplateIdResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageTemplateIdResponse updateBrandMessageWideTemplate(
      String sendProfileId,
      String templateCode,
      BrandMessageWideTemplateRequest request
  ) {
    try {
      JsonObject bodyJson = request.toJson();
      SendonJsonResponse response = parseJsonResponse(post(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates/" + templateCode + "/update/wide",
          bodyJson.toString()
      ));
      return new BrandMessageTemplateIdResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageTemplateIdResponse updateBrandMessageWideItemListTemplate(
      String sendProfileId,
      String templateCode,
      BrandMessageWideItemListTemplateRequest request
  ) {
    try {
      JsonObject bodyJson = request.toJson();
      SendonJsonResponse response = parseJsonResponse(post(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates/" + templateCode + "/update/wide-item-list",
          bodyJson.toString()
      ));
      return new BrandMessageTemplateIdResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageTemplateIdResponse updateBrandMessageCarouselFeedTemplate(
      String sendProfileId,
      String templateCode,
      BrandMessageCarouselFeedTemplateRequest request
  ) {
    try {
      JsonObject bodyJson = request.toJson();
      SendonJsonResponse response = parseJsonResponse(post(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates/" + templateCode + "/update/carousel-feed",
          bodyJson.toString()
      ));
      return new BrandMessageTemplateIdResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageTemplateDetailResponse getBrandMessageTemplate(String sendProfileId, String templateCode) {
    try {
      SendonJsonResponse response = parseJsonResponse(get(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates/" + templateCode
      ));
      return new BrandMessageTemplateDetailResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageTemplateListResponse getBrandMessageTemplates(String sendProfileId, BrandMessageTemplateQuery query) {
    try {
      StringBuilder urlBuilder = new StringBuilder(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates"
      );

      String queryString = buildBrandMessageTemplateQuery(query);
      if (!queryString.isEmpty()) {
        urlBuilder.append('?').append(queryString);
      }

      SendonJsonResponse response = parseJsonResponse(get(urlBuilder.toString()));
      return new BrandMessageTemplateListResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageDeleteResponse deleteBrandMessageTemplate(String sendProfileId, String templateCode) {
    try {
      SendonJsonResponse response = parseJsonResponse(post(
          "/v2/messages/kakao/send-profiles/" + sendProfileId + "/brand-message-templates/" + templateCode + "/delete",
          null
      ));
      return new BrandMessageDeleteResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  // ===================================
  // Brand Message Sending
  // ===================================

  public BrandMessageSendResponse sendBrandMessage(BrandMessageSendBuilder builder) {
    try {
      JsonObject bodyJson = builder.toJson();
      SendonJsonResponse response = parseJsonResponse(post(
          "/v2/messages/kakao/brand-message",
          bodyJson.toString()
      ));
      return new BrandMessageSendResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  // ===================================
  // Brand Message Assets
  // ===================================

  public BrandMessageImageUploadResponse uploadBrandMessageImage(File file) {
    try {
      String contentType = resolveBrandMessageImageContentType(file);
      SendonJsonResponse response = parseJsonResponse(postImageWithMultipartFormData(
          "/v2/messages/kakao/brand-message/image/upload",
          file,
          "file",
          contentType
      ));
      return new BrandMessageImageUploadResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public BrandMessageImageUploadResponse uploadBrandMessageWideImage(File file) {
    try {
      String contentType = resolveBrandMessageImageContentType(file);
      SendonJsonResponse response = parseJsonResponse(postImageWithMultipartFormData(
          "/v2/messages/kakao/brand-message/image/upload/wide",
          file,
          "file",
          contentType
      ));
      return new BrandMessageImageUploadResponse(response);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  private String resolveBrandMessageImageContentType(File file) {
    String fileName = file.getName().toLowerCase(Locale.ROOT);
    if (fileName.endsWith(".png")) {
      return "image/png";
    }
    if (fileName.endsWith(".jpg") || fileName.endsWith(".jpeg")) {
      return "image/jpeg";
    }
    throw new IllegalArgumentException("브랜드메시지 이미지는 JPG, JPEG, PNG 형식만 지원합니다.");
  }

  private String buildBrandMessageTemplateQuery(BrandMessageTemplateQuery query) {
    if (query == null) return "";

    List<String> params = new ArrayList<>();
    if (query.getMessageType() != null && !query.getMessageType().isEmpty()) {
      params.add("messageType=" + encodeURIComponent(query.getMessageType()));
    }
    if (query.getStatus() != null && !query.getStatus().isEmpty()) {
      params.add("status=" + encodeURIComponent(query.getStatus()));
    }
    if (query.getLimit() != null && query.getLimit() > 0) {
      params.add("limit=" + query.getLimit());
    }
    if (query.getCursor() != null && !query.getCursor().isEmpty()) {
      params.add("cursor=" + encodeURIComponent(query.getCursor()));
    }

    return String.join("&", params);
  }

  private String encodeURIComponent(String value) {
    try {
      return URLEncoder.encode(value, StandardCharsets.UTF_8.toString());
    } catch (Exception e) {
      throw new RuntimeException("Error encoding URL component", e);
    }
  }

  public UploadAlimtalkImage uploadAlimtalkImage(File image) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(postImageWithMultipartFormData("/v2/messages/kakao/alimtalk/image", image));
      return new UploadAlimtalkImage(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public UploadFallbackImage uploadFallbackImage(List<File> images) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(postImagesWithMultipartFormData("/v2/messages/kakao/fallback/image", images));
      return new UploadFallbackImage(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public GetGroup getGroup(String groupId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(get("/v2/messages/kakao/groups/" + groupId));
      return new GetGroup(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public CancelGroup cancelGroup(String groupId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/kakao/groups/" + groupId + "/cancel", null));
      return new CancelGroup(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  public SendGroup sendGroup(String groupId) {
    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/kakao/groups/" + groupId + "/send", null));
      return new SendGroup(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Sends an Alimtalk message with reservation and fallback options.
   *
   * @param profileId   The profile ID.
   * @param templateId  The template ID.
   * @param to          The list of recipients. Can be {@code List<String>} or {@code List<ToWithVariable>}.
   * @param reservation The reservation details for scheduling. See {@link Reservation}.
   * @param fallback    The fallback options. See {@link Fallback}.
   * @return The response of the Alimtalk send operation. See {@link SendAlimtalk}.
   */
  public SendAlimtalk sendAlimtalk(String profileId, String templateId, List<?> to, Reservation reservation, Fallback fallback) {
    return sendAlimtalkInternal(profileId, templateId, to, reservation, fallback);
  }

  /**
   * Sends an Alimtalk message using a builder.
   *
   * @param builder The {@link AlimtalkBuilder} containing the Alimtalk message details.
   * @return The response of the Alimtalk send operation. See {@link SendAlimtalk}.
   */
  public SendAlimtalk sendAlimtalk(AlimtalkBuilder builder) {
    return sendAlimtalkInternal(builder.profileId, builder.templateId, builder.to, builder.reservation, builder.fallback);
  }

  private SendAlimtalk sendAlimtalkInternal(String profileId, String templateId, List<?> to, Reservation reservation, Fallback fallback) {
      JsonObject bodyJson = new JsonObject();
      bodyJson.addProperty("sendProfileId", profileId);
      bodyJson.addProperty("templateId", templateId);
      bodyJson.add("to", gson.toJsonTree(to));

      if (reservation != null) {
        JsonObject reservationJson = new JsonObject();
        reservationJson.addProperty("datetime", reservation.datetime);
      bodyJson.add("reservation", reservationJson);
    }

    if (fallback != null) {
      JsonObject fallbackJson = new JsonObject();
      fallbackJson.addProperty("fallbackType", fallback.fallbackType.value);
      JsonObject fallbackCustom = new JsonObject();
      fallbackCustom.addProperty("type", fallback.custom.type.value);
      fallbackCustom.addProperty("senderNumber", fallback.custom.senderNumber);
      fallbackCustom.addProperty("isAd", fallback.custom.isAd);
      fallbackCustom.addProperty("message", fallback.custom.message);
      fallbackCustom.addProperty("title", fallback.custom.title);
      fallbackCustom.add("images", gson.toJsonTree(fallback.custom.images));
      fallbackJson.add("custom", fallbackCustom);
      bodyJson.add("fallback", fallbackJson);
    }

    try {
      SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/kakao/alim-talk", bodyJson.toString()));
      return new SendAlimtalk(sendonJsonResponse);
    } catch (Exception e) {
      e.printStackTrace();
    }
    return null;
  }

  private <T> SendFriendtalk sendFriendtalkInternal(
      String profileId, String templateId, List<T> to, FriendtalkMessageType messageType, String message,
      List<Button> buttons, boolean isAd, Reservation reservation,
      Fallback fallback, Image image
  ) {
      JsonObject bodyJson = new JsonObject();
      bodyJson.addProperty("sendProfileId", profileId);
      bodyJson.addProperty("templateId", templateId);
      bodyJson.addProperty("messageType", messageType.value);
      bodyJson.add("to", gson.toJsonTree(to));
      bodyJson.addProperty("message", message);
      bodyJson.addProperty("isAd", isAd);

      if (buttons != null) {
          JsonArray buttonsArray = new JsonArray();
          for (Button button : buttons) {
              JsonObject buttonObject = new JsonObject();
              buttonObject.addProperty("name", button.name);
              buttonObject.addProperty("ordering", button.ordering);
              buttonObject.addProperty("type", button.type.value);

              if (button instanceof WlButton) {
                  WlButton wlButton = (WlButton) button;
                  buttonObject.addProperty("urlMobile", wlButton.urlMobile);
                  buttonObject.addProperty("urlPc", wlButton.urlPc);
              } else if (button instanceof AlButton) {
                  AlButton alButton = (AlButton) button;
                  buttonObject.addProperty("schemeIos", alButton.schemeIos);
                  buttonObject.addProperty("schemeAndroid", alButton.schemeAndroid);
              }
              buttonsArray.add(buttonObject);
          }
          bodyJson.add("buttons", buttonsArray);
      }

      if (reservation != null) {
          JsonObject reservationJson = new JsonObject();
          reservationJson.addProperty("datetime", reservation.datetime);
          bodyJson.add("reservation", reservationJson);
      }

      if (fallback != null) {
          JsonObject fallbackJson = new JsonObject();
          fallbackJson.addProperty("fallbackType", fallback.fallbackType.value);
          JsonObject fallbackCustom = new JsonObject();
          fallbackCustom.addProperty("type", fallback.custom.type.value);
          fallbackCustom.addProperty("senderNumber", fallback.custom.senderNumber);
          fallbackCustom.addProperty("isAd", fallback.custom.isAd);
          fallbackCustom.addProperty("message", fallback.custom.message);
          fallbackCustom.addProperty("title", fallback.custom.title);
          fallbackCustom.add("images", gson.toJsonTree(fallback.custom.images));
          fallbackJson.add("custom", fallbackCustom);
          bodyJson.add("fallback", fallbackJson);
      }

      if (image != null) {
          JsonObject imageJson = new JsonObject();
          imageJson.addProperty("imageUrl", image.url);
          imageJson.addProperty("imageLink", image.link);
          bodyJson.add("image", imageJson);
      }

      try {
          SendonJsonResponse sendonJsonResponse = parseJsonResponse(post("/v2/messages/kakao/friend-talk", bodyJson.toString()));
          return new SendFriendtalk(sendonJsonResponse);
      } catch (Exception e) {
          e.printStackTrace();
      }
      return null;
  }

  /**
   * Sends a Friendtalk message using a builder.
   *
   * @param builder The {@link FriendtalkBuilder} containing the Friendtalk message details.
   * @return The response of the Friendtalk send operation. See {@link SendFriendtalk}.
   */
  public SendFriendtalk sendFriendtalk(FriendtalkBuilder builder) {
      return sendFriendtalkInternal(
          builder.profileId, builder.templateId, builder.to, builder.messageType, builder.message,
          builder.buttons, builder.isAd, builder.reservation, builder.fallback, builder.image
      );
  }

  /**
   * Sends a Friendtalk message.
   *
   * @param profileId   The profile ID.
   * @param templateId  The template ID.
   * @param to          The list of recipients. Can be {@code List<String>} (phone numbers) or {@code List<ToWithVariable>} (recipients with variables).
   * @param message     The message content.
   * @param buttons     The list of buttons. See {@link Button}.
   * @param isAd        Whether the message is an advertisement.
   * @param reservation The reservation details for scheduling. See {@link Reservation}.
   * @param fallback    The fallback options. See {@link Fallback}.
   * @param image       The image details. See {@link Image}.
   * @return The response of the Friendtalk send operation. See {@link SendFriendtalk}.
   */
  public SendFriendtalk sendFriendtalk(
      String profileId, String templateId, List<?> to, FriendtalkMessageType messageType, String message,
      List<Button> buttons, boolean isAd, Reservation reservation,
      Fallback fallback, Image image
  ) {
      return sendFriendtalkInternal(profileId, templateId, to, messageType, message, buttons, isAd, reservation, fallback, image);
  }
}
