package io.sendon.contacts.request;

/**
 * Builder class for constructing contact blocklist operations.
 * Provides a fluent API for creating and configuring blocklist requests.
 */
public class ContactBuilder {

  /**
   * Builder for creating blocklist entries.
   */
  public static class CreateBlocklistBuilder {
    private String phoneNumber;
    private String messageType;
    private String senderNumber;
    private String kakaoChannelId;
    private String rcsChatbotId;
    private String rcsBrandId;

    /**
     * Sets the phone number to be added to the blocklist.
     *
     * @param phoneNumber The phone number without hyphens, numbers only.
     * @return The current instance of {@link CreateBlocklistBuilder}.
     */
    public CreateBlocklistBuilder setPhoneNumber(String phoneNumber) {
      this.phoneNumber = phoneNumber;
      return this;
    }

    /**
     * Sets the message type for blocking.
     *
     * @param messageType The message type (SMS, KAKAO, RCS). Default is SMS.
     * @return The current instance of {@link CreateBlocklistBuilder}.
     */
    public CreateBlocklistBuilder setMessageType(String messageType) {
      this.messageType = messageType;
      return this;
    }

    /**
     * Sets the sender number to block from.
     *
     * @param senderNumber The sender number to block from.
     * @return The current instance of {@link CreateBlocklistBuilder}.
     */
    public CreateBlocklistBuilder setSenderNumber(String senderNumber) {
      this.senderNumber = senderNumber;
      return this;
    }

    /**
     * Sets the Kakao channel ID for Kakao message blocking.
     *
     * @param kakaoChannelId The Kakao channel ID.
     * @return The current instance of {@link CreateBlocklistBuilder}.
     */
    public CreateBlocklistBuilder setKakaoChannelId(String kakaoChannelId) {
      this.kakaoChannelId = kakaoChannelId;
      return this;
    }

    /**
     * Sets the RCS chatbot ID for RCS message blocking.
     *
     * @param rcsChatbotId The RCS chatbot ID.
     * @return The current instance of {@link CreateBlocklistBuilder}.
     */
    public CreateBlocklistBuilder setRcsChatbotId(String rcsChatbotId) {
      this.rcsChatbotId = rcsChatbotId;
      return this;
    }

    /**
     * Sets the RCS brand ID for RCS message blocking.
     *
     * @param rcsBrandId The RCS brand ID.
     * @return The current instance of {@link CreateBlocklistBuilder}.
     */
    public CreateBlocklistBuilder setRcsBrandId(String rcsBrandId) {
      this.rcsBrandId = rcsBrandId;
      return this;
    }

    /**
     * Builds the CreateBlocklistRequest object.
     *
     * @return A configured {@link CreateBlocklistRequest} instance.
     */
    public CreateBlocklistRequest build() {
      return new CreateBlocklistRequest(phoneNumber, messageType, senderNumber, 
                                       kakaoChannelId, rcsChatbotId, rcsBrandId);
    }
  }

  /**
   * Builder for retrieving blocklist entries with filtering and pagination.
   */
  public static class GetBlocklistBuilder {
    private String senderNumber;
    private String[] kakaoChannelIds;
    private String[] blockTypes;
    private String startDate;
    private String endDate;
    private Integer cursor;
    private Integer limit;

    /**
     * Sets the sender number filter.
     *
     * @param senderNumber Filter by specific sender number.
     * @return The current instance of {@link GetBlocklistBuilder}.
     */
    public GetBlocklistBuilder setSenderNumber(String senderNumber) {
      this.senderNumber = senderNumber;
      return this;
    }

    /**
     * Sets the Kakao channel IDs filter.
     *
     * @param kakaoChannelIds Array of Kakao channel IDs to filter by.
     * @return The current instance of {@link GetBlocklistBuilder}.
     */
    public GetBlocklistBuilder setKakaoChannelIds(String[] kakaoChannelIds) {
      this.kakaoChannelIds = kakaoChannelIds;
      return this;
    }

    /**
     * Sets the block types filter.
     *
     * @param blockTypes Array of block types (WEB, API, ARS) to filter by.
     * @return The current instance of {@link GetBlocklistBuilder}.
     */
    public GetBlocklistBuilder setBlockTypes(String[] blockTypes) {
      this.blockTypes = blockTypes;
      return this;
    }

    /**
     * Sets the start date filter.
     *
     * @param startDate Start date for filtering (YYYY-MM-DD HH:mm:ss format).
     * @return The current instance of {@link GetBlocklistBuilder}.
     */
    public GetBlocklistBuilder setStartDate(String startDate) {
      this.startDate = startDate;
      return this;
    }

    /**
     * Sets the end date filter.
     *
     * @param endDate End date for filtering (YYYY-MM-DD HH:mm:ss format).
     * @return The current instance of {@link GetBlocklistBuilder}.
     */
    public GetBlocklistBuilder setEndDate(String endDate) {
      this.endDate = endDate;
      return this;
    }

    /**
     * Sets the pagination cursor.
     *
     * @param cursor The cursor for pagination (default: 0).
     * @return The current instance of {@link GetBlocklistBuilder}.
     */
    public GetBlocklistBuilder setCursor(Integer cursor) {
      this.cursor = cursor;
      return this;
    }

    /**
     * Sets the page limit.
     *
     * @param limit The maximum number of entries to retrieve (default: 20).
     * @return The current instance of {@link GetBlocklistBuilder}.
     */
    public GetBlocklistBuilder setLimit(Integer limit) {
      this.limit = limit;
      return this;
    }

    /**
     * Builds the GetBlocklistRequest object.
     *
     * @return A configured {@link GetBlocklistRequest} instance.
     */
    public GetBlocklistRequest build() {
      return new GetBlocklistRequest(senderNumber, kakaoChannelIds, blockTypes,
                                   startDate, endDate, cursor, limit);
    }
  }

  /**
   * Creates a new builder for creating blocklist entries.
   *
   * @return A new {@link CreateBlocklistBuilder} instance.
   */
  public static CreateBlocklistBuilder createBlocklist() {
    return new CreateBlocklistBuilder();
  }

  /**
   * Creates a new builder for retrieving blocklist entries.
   *
   * @return A new {@link GetBlocklistBuilder} instance.
   */
  public static GetBlocklistBuilder getBlocklist() {
    return new GetBlocklistBuilder();
  }
}